<?php

namespace Tests\Tenant;

use App\Facades\Settings;
use App\Http\Resources\LiveChatResource;
use App\Http\Resources\SiteResource;
use App\IntegrationRepository;
use Mtc\MercuryDataModels\Franchise;
use Tests\TenantTestCase;

class LiveChatTest extends TenantTestCase
{
    public function testFuzeySettingsInSiteResource()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-fuzey-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-fuzey-client_id',
            'type' => 'string',
            'value' => 'foo',
        ]);

        $response = $this
            ->getJson(route('get-site'));
        $response->assertStatus(200);

        $this->assertArrayHasKey('liveChat', $response->json());
        $this->assertArrayHasKey('data', $response->json('liveChat'));
        $this->assertArrayHasKey('key', $response->json('liveChat.data'));
        $this->assertEquals('foo', $response->json('liveChat.data.key'));
    }

    public function testMoneypennySettingsInSiteResourceNoFranchises()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-moneypenny-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-moneypenny-key',
            'type' => 'string',
            'value' => 'foo',
        ]);

        $response = $this
            ->getJson(route('get-site'));
        $response->assertStatus(200);

        $this->assertArrayHasKey('liveChat', $response->json());
        $this->assertArrayHasKey('data', $response->json('liveChat'));
        $this->assertArrayHasKey('key', $response->json('liveChat.data'));
        $this->assertArrayHasKey('franchise_keys', $response->json('liveChat.data'));
        $this->assertEquals('foo', $response->json('liveChat.data.key'));
        $this->assertEmpty($response->json('liveChat.data.franchise_keys'));
    }

    public function testMoneypennySettingsInSiteResourceWithFranchises()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-moneypenny-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-moneypenny-key',
            'type' => 'string',
            'value' => 'foo',
        ]);

        // create one franchise without a key
        Franchise::factory()->create([
            'slug' => 'audi',
        ]);

        // create one franchise with a key
        Franchise::factory()->create([
            'slug' => 'bmw',
            'data' => [
                'chat-moneypenny-franchise-key' => 'bmw_key',
            ],
        ]);

        $response = $this
            ->getJson(route('get-site'));
        $response->assertStatus(200);

        $this->assertArrayHasKey('liveChat', $response->json());
        $this->assertArrayHasKey('data', $response->json('liveChat'));
        $this->assertArrayHasKey('key', $response->json('liveChat.data'));
        $this->assertEquals('foo', $response->json('liveChat.data.key'));
        $this->assertArrayHasKey('franchise_keys', $response->json('liveChat.data'));
        $this->assertCount(2, $response->json('liveChat.data.franchise_keys'));
        $this->assertArrayHasKey('audi', $response->json('liveChat.data.franchise_keys'));
        $this->assertArrayHasKey('bmw', $response->json('liveChat.data.franchise_keys'));
        $this->assertEquals('', $response->json('liveChat.data.franchise_keys.audi'));
        $this->assertEquals('bmw_key', $response->json('liveChat.data.franchise_keys.bmw'));
    }

    public function testMoneypennyEnabled()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-moneypenny-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        $widget = new IntegrationRepository()->getLiveChat();

        $this->assertEquals('moneypenny', $widget['name']);
        $this->assertArrayHasKey('key', $widget);
        $this->assertEquals('', $widget['key']);
    }

    public function testMoneypennyFranchiseSettings()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-moneypenny-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-moneypenny-key',
            'type' => 'string',
            'value' => 'default-key',
        ]);

        // create one franchise without a key
        Franchise::factory()->create([
            'slug' => 'audi',
        ]);

        // create one franchise with a key
        Franchise::factory()->create([
            'slug' => 'bmw',
            'data' => [
                'chat-moneypenny-franchise-key' => 'foo',
            ],
        ]);

        $widget = new IntegrationRepository()->getLiveChat();

        $this->assertEquals('moneypenny', $widget['name']);

        // assert that we have the expected site settings key
        $this->assertArrayHasKey('key', $widget);
        $this->assertEquals('default-key', $widget['key']);

        // assert that we get the expected, optional franchise keys
        $this->assertArrayHasKey('data', $widget);
        $this->assertArrayHasKey('franchise_keys', $widget['data']);
        $this->assertArrayHasKey('audi', $widget['data']['franchise_keys']);
        $this->assertArrayHasKey('bmw', $widget['data']['franchise_keys']);
        $this->assertEquals('', $widget['data']['franchise_keys']['audi']);
        $this->assertEquals('foo', $widget['data']['franchise_keys']['bmw']);
    }

    public function testLiveChatSiteResource()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-moneypenny-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-moneypenny-key',
            'type' => 'string',
            'value' => 'default-key',
        ]);

        // create one franchise without a key
        Franchise::factory()->create([
            'slug' => 'audi',
        ]);

        // create one franchise with a key
        Franchise::factory()->create([
            'slug' => 'bmw',
            'data' => [
                'chat-moneypenny-franchise-key' => 'foo',
            ],
        ]);

        $resource = new SiteResource([])->toArray(request());

        $this->assertInstanceOf(LiveChatResource::class, $resource['liveChat']);
        $data = $resource['liveChat']->toArray(request());
        $this->assertArrayHasKey('widget', $data);
        $this->assertArrayHasKey('data', $data);
        $this->assertArrayHasKey('key', $data['data']);
        $this->assertEquals('default-key', $data['data']['key']);
        $this->assertArrayHasKey('franchise_keys', $data['data']);
        $this->assertArrayHasKey('audi', $data['data']['franchise_keys']);
        $this->assertArrayHasKey('bmw', $data['data']['franchise_keys']);
        $this->assertEquals('', $data['data']['franchise_keys']['audi']);
        $this->assertEquals('foo', $data['data']['franchise_keys']['bmw']);
    }

    public function testAutoConverseSettingsInSiteResource()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-autoconverse-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-autoconverse-key',
            'type' => 'string',
            'value' => 'foo',
        ]);

        $response = $this
            ->getJson(route('get-site'));
        $response->assertStatus(200);

        $this->assertArrayHasKey('liveChat', $response->json());

        $this->assertArrayHasKey('data', $response->json('liveChat'));
        $this->assertArrayHasKey('key', $response->json('liveChat.data'));
        $this->assertEquals('foo', $response->json('liveChat.data.key'));
    }

    public function testCallDripSettingsInSiteResource()
    {
        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-calldrip-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-calldrip-key',
            'type' => 'string',
            'value' => 'foo_key',
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'chat-calldrip-id',
            'type' => 'string',
            'value' => 'foo_id',
        ]);

        $response = $this
            ->getJson(route('get-site'));
        $response->assertStatus(200);

        $this->assertArrayHasKey('liveChat', $response->json());

        $this->assertArrayHasKey('data', $response->json('liveChat'));
        $this->assertArrayHasKey('key', $response->json('liveChat.data'));
        $this->assertArrayHasKey('id', $response->json('liveChat.data'));
        $this->assertEquals('foo_key', $response->json('liveChat.data.key'));
        $this->assertEquals('foo_id', $response->json('liveChat.data.id'));

    }
}
