<?php

namespace App\Http\Controllers;

use App\Http\Requests\AddAddOnToDeal;
use App\Http\Requests\AddPartExchangeVehicle;
use App\Http\Requests\DealBuilderRequest;
use App\Http\Requests\InitiateDealPayment;
use App\Http\Requests\RemoveAddOnFromDeal;
use App\Http\Requests\RemovePartExchangeVehicle;
use App\Http\Requests\ResumeDealRequest;
use App\Http\Requests\SelectDealFinanceQuote;
use App\Http\Requests\SetDealFinanceTerms;
use App\Http\Requests\SetDealPaymentType;
use App\Http\Requests\StartDealRequest;
use App\Http\Requests\SubmitDealRequest;
use App\Http\Requests\UpdatePartExchangeVehicle;
use App\Http\Requests\ValuationRequest;
use App\Http\Resources\DealBuilderResource;
use App\Http\Resources\ValuationResource;
use Illuminate\Http\JsonResponse;
use Mtc\MercuryDataModels\DealBuilder\Deal;
use Mtc\VehicleValuation\Facades\Valuation;

class DealBuilderController extends Controller
{
    public function start(StartDealRequest $request): DealBuilderResource
    {
        return new DealBuilderResource($request->deal());
    }

    public function show(DealBuilderRequest $request): DealBuilderResource
    {
        return new DealBuilderResource($request->deal());
    }

    public function resume(ResumeDealRequest $request): DealBuilderResource
    {
        return new DealBuilderResource($request->deal());
    }

    public function addAddOn(AddAddOnToDeal $request): DealBuilderResource
    {
        $request->dealBuilder()->addAddOn(
            $request->deal(),
            $request->input('add_on_id'),
            $request->input('payment_term') ?? 'total'
        );
        return new DealBuilderResource($request->deal());
    }

    public function removeAddOn(RemoveAddOnFromDeal $request): DealBuilderResource
    {
        $request->dealBuilder()->removeAddOn($request->deal(), $request->input('add_on_id'));
        return new DealBuilderResource($request->deal());
    }

    public function lookupPartExchangeVehicle(ValuationRequest $request): ValuationResource
    {
        // Persist email on deal immediately for early customer capture
        if ($request->filled('email')) {
            $request->deal()->update(['email' => $request->input('email')]);
        }

        if (Valuation::hasMultiStep() && !$request->filled('variant_id')) {
            return new ValuationResource(Valuation::getVehicleVariants(
                $request->input('registration_number'),
                $request->input('mileage')
            ));
        }
        $valuation = Valuation::get(
            $request->input('registration_number'),
            $request->input('mileage'),
            $request->input('variant_id')
        );

        return new ValuationResource($valuation);
    }

    public function addPartExchange(AddPartExchangeVehicle $request): DealBuilderResource
    {
        $request->dealBuilder()->addPartExchange($request->deal(), $request->validated());
        return new DealBuilderResource($request->deal());
    }

    public function removePartExchange(RemovePartExchangeVehicle $request): DealBuilderResource
    {
        $request->dealBuilder()->removePartExchange($request->deal(), $request->input('part_exchange_id'));
        return new DealBuilderResource($request->deal());
    }

    public function updatePartExchange(UpdatePartExchangeVehicle $request): DealBuilderResource
    {
        $request->dealBuilder()->updatePartExchange($request->deal(), $request->validated());
        return new DealBuilderResource($request->deal());
    }

    public function setFinanceTerms(SetDealFinanceTerms $request): DealBuilderResource
    {
        $request->dealBuilder()->setFinanceTerms($request->deal(), $request->validated());
        return new DealBuilderResource($request->deal());
    }
    public function selectFinanceQuote(SelectDealFinanceQuote $request): DealBuilderResource
    {
        $request->dealBuilder()->selectFinanceQuote($request->deal(), $request->input('quote_id'));
        return new DealBuilderResource($request->deal());
    }

    public function setPaymentType(SetDealPaymentType $request): DealBuilderResource
    {
        $request->dealBuilder()->setPaymentType($request->deal(), $request->input('payment_type'));
        return new DealBuilderResource($request->deal());
    }

    public function initiatePayment(InitiateDealPayment $request): JsonResponse
    {
        $payment = $request->dealBuilder()->initiatePayment($request->deal(), $request->validated());

        $data = (new DealBuilderResource($request->deal()->fresh()))->toArray($request);
        $data['payment'] = $payment;
        return response()->json($data);
    }

    public function submitDeal(SubmitDealRequest $request): DealBuilderResource
    {
        $deal = $request->deal();
        $request->dealBuilder()->submitDeal($deal, $request->validated());
        return new DealBuilderResource($deal);
    }


    public function verify(string $reference): JsonResponse
    {
        $deal = Deal::query()->where('reference', $reference)->first();

        if (!$deal) {
            return response()->json(['error' => 'Deal not found'], 404);
        }

        return response()->json([
            'deal' => [
                'reference' => $deal->reference,
                'vehicle' => $deal->vehicle ? [
                    'title' => $deal->vehicle->title,
                ] : null,
            ],
        ]);
    }
}
