<?php

namespace App\Http\Resources;

use App\Facades\Settings;
use App\Repositories\DealBuilderRepository;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\DealBuilder\AddOn;

class DealBuilderResource extends JsonResource
{
    public static $wrap = '';

    private DealBuilderRepository $dealBuilderRepository;

    public function __construct($resource)
    {
        parent::__construct($resource);
        $this->dealBuilderRepository = new DealBuilderRepository();
    }

    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        $selected_add_ons = $this->resource->addOns->pluck('add_on_id')->toArray();
        $defaults = $this->dealBuilderRepository->getDefaultFinanceSettings($this->resource->vehicle);

        $allowMultiplePartExchange = Settings::get('deal-builder-allow-multiple-part-exchange', false);
        $hasPartExchange = $this->resource->partExchange->isNotEmpty();

        return [
            'deal' => [
                'annual_mileage' => $this->resource->annual_mileage ?? $defaults['annual_mileage'],
                'term' => $this->resource->term ?? $defaults['term'],
                'customer_deposit' => $this->resource->customer_deposit ?? $defaults['customer_deposit'],
                'deposit_amount' => $this->resource->deposit_amount,
                'part_ex_contribution' => $this->resource->part_ex_contribution,
                'total_amount' => $this->resource->total_amount,
                'payable_amount' => $this->resource->payable_amount,
                'credit_rating' => $this->resource->credit_rating ?? $defaults['credit_rating'],
                'reference' => $this->resource->reference,
                'first_name' => $this->resource->first_name,
                'last_name' => $this->resource->last_name,
                'email' => $this->resource->email,
                'contact_number' => $this->resource->contact_number,
                'add_ons' => $this->resource->addOns,
                'finance' => $this->resource->finance,
                'part_exchange' => $this->resource->partExchange,
                'payment_type' => $this->resource->payment_type,
                'vehicle' => new VehicleQuickViewResource($this->resource->vehicle),
                'can_add_part_exchange' => $allowMultiplePartExchange || !$hasPartExchange,
            ],
            'add_ons' => $this->dealBuilderRepository->addOnsForVehicle($this->resource->vehicle)
                ->each(fn(AddOn $addOn) => $addOn->selected = in_array($addOn->id, $selected_add_ons)),
            'defaults' => $defaults,
            'allowed_payment_types' => $this->getAllowedPaymentTypes(),
        ];
    }

    private function getAllowedPaymentTypes(): array
    {
        // If part exchange covers the full amount (total is 0), only allow enquiry and reservation
        $paymentNotRequired = ($this->resource->total_amount ?? 0) <= 0;

        return [
            'enquiry' => true,
            'reservation' => !$paymentNotRequired,
            'deposit' => !$paymentNotRequired && (bool) Settings::get('deal-builder-allow-deposit-payment', true),
            'full_payment' => !$paymentNotRequired && (bool) Settings::get('deal-builder-allow-full-payment', true),
        ];
    }
}
