<?php

namespace App;

use App\Facades\Settings;
use App\Filter\Sorting\VehicleAttribute as VehicleAttributeSorting;
use Illuminate\Database\Eloquent\Model;
use Mtc\MercuryDataModels\CatalogOffer;
use Mtc\MercuryDataModels\CatalogOfferRule;
use Mtc\MercuryDataModels\Vehicle;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Facades\Config;
use Mtc\Filter\Contracts\FilterObject;
use App\Http\Resources\VehicleList;
use Mtc\MercuryDataModels\VehicleAttribute;

class VehicleFilter implements FilterObject
{
    private ?Builder $currentQuery = null;
    private ?CatalogOffer $selectedCatalogOffer = null;
    private bool $catalogOfferResolved = false;

    /**
     * Instantiate a query for object
     *
     * @return Builder
     */
    public function createQuery(): Builder
    {
        return Vehicle::query()
            ->active()
            ->withRelationshipsForCardView()
            ->when(Settings::get('automotive-vehicle-brand-on-filter-card'), fn($query) => $query->with('make'));
    }

    /**
     * Fetch results
     *
     * @param Builder $query
     * @return LengthAwarePaginator
     */
    public function getResults(Builder $query): LengthAwarePaginator
    {
        $this->currentQuery = clone $query;
        $pageLimit = Settings::get('filter-results-per-page', Config::get('filter.result_page_limit'));

        if ($this->hasCatalogOffers() && $this->getResultCount($query, false) >= $pageLimit) {
            $pageLimit--;
        }

        return $query->paginate(request('perPage', $pageLimit));
    }

    /**
     * Fetch results
     *
     * @param Builder $query
     * @return int
     */
    public function getResultCount(Builder $query, bool $include_extras = true): int
    {
        $this->currentQuery = clone $query;
        $count = $query->count();

        if ($include_extras && $this->hasCatalogOffers()) {
            $count++;
        }

        return $count;
    }

    /**
     * Format results to response data
     *
     * @param LengthAwarePaginator $results
     * @return JsonResource
     */
    public function format(LengthAwarePaginator $results): JsonResource
    {
        return new VehicleList($results, $this->getCatalogOffer(), true);
    }

    /**
     * Apply filters to query (e.g. when getting values for filters not main object)
     *
     * @param Builder $query
     */
    public function applyFilter(Builder $query): void
    {
        $query->active();
    }

    /**
     * @return bool
     */
    private function catalogOffersEnabled(): bool
    {
        return Settings::get('filter-inject-offers-into-used-car-filter') === true;
    }

    /**
     * @return bool
     */
    private function hasCatalogOffers(): bool
    {
        if (!$this->catalogOffersEnabled()) {
            return false;
        }

        return $this->getCatalogOffer() !== null;
    }

    /**
     * @return CatalogOffer|null
     */
    private function getCatalogOffer(): ?CatalogOffer
    {
        if (!$this->catalogOffersEnabled()) {
            return null;
        }

        if ($this->catalogOfferResolved) {
            return $this->selectedCatalogOffer;
        }

        $this->catalogOfferResolved = true;

        $offers = CatalogOffer::query()
            ->active()
            ->forVehicles()
            ->with('rules')
            ->inRandomOrder()
            ->get();


        foreach ($offers->shuffle() as $offer) {
            if ($this->catalogOfferMatchesQuery($offer)) {
                $this->selectedCatalogOffer = $offer;
                return $offer;
            }
        }

        return null;
    }

    /**
     * Check if catalog offer rules match vehicles in the current query
     */
    private function catalogOfferMatchesQuery(CatalogOffer $offer): bool
    {
        if ($offer->rules->isEmpty()) {
            return true;
        }

        if ($this->currentQuery === null) {
            return true;
        }

        $query = clone $this->currentQuery;

        foreach ($offer->rules as $rule) {
            $this->applyRuleToQuery($query, $rule);
        }

        return $query->exists();
    }

    /**
     * Apply a single rule to the query
     */
    private function applyRuleToQuery(Builder $query, CatalogOfferRule $rule): void
    {
        $value = $this->parseRuleValue($rule->value);

        match ($rule->condition) {
            '=' => $query->where($rule->field, '=', $value),
            '!=' => $query->where($rule->field, '!=', $value),
            '>' => $query->where($rule->field, '>', $value),
            '>=' => $query->where($rule->field, '>=', $value),
            '<' => $query->where($rule->field, '<', $value),
            '<=' => $query->where($rule->field, '<=', $value),
            'is_null' => $query->whereNull($rule->field),
            'exists' => $query->whereNotNull($rule->field),
            'in' => $query->whereIn($rule->field, (array) $value),
            'not_in' => $query->whereNotIn($rule->field, (array) $value),
            default => null,
        };
    }

    /**
     * Parse rule value - decode JSON arrays if present
     */
    private function parseRuleValue(mixed $value): mixed
    {
        if (is_string($value) && str_starts_with($value, '[')) {
            $decoded = json_decode($value, true);
            if (json_last_error() === JSON_ERROR_NONE) {
                return $decoded;
            }
        }

        return $value;
    }

    public function sortOptionName(): string
    {
        return 'vehicles';
    }

    public function additionalSortOptions($activeSortOptionName = '')
    {
        $sortOptions = [];
        $sortableAttributes = VehicleAttribute::query()
            ->where('sortable', 1)
            ->get();

        foreach ($sortableAttributes as $sortableAttribute) {
            $sortOptions[$sortableAttribute->slug] = VehicleAttributeSorting::class;
        }

        return $sortOptions;
    }
}
