<?php

namespace Tests\Tenant;

use App\Facades\Settings;
use Carbon\Carbon;
use Illuminate\Http\Client\Request;
use Illuminate\Support\Facades\Http;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceRequestData;
use Mtc\MercuryDataModels\Finance\FinanceServiceHelper;
use Mtc\MercuryDataModels\Finance\Jobs\RunFinanceFetchForVehicle;
use Mtc\MercuryDataModels\Finance\Services\IVendi;
use Mtc\MercuryDataModels\Finance\Services\IVendiPeterVardy;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleAttribute;
use Mtc\MercuryDataModels\VehicleAttributeValue;
use Tests\Data\FinanceSettingSeeder;
use Tests\Data\IvendiUsernameSeeder;
use Tests\Data\PeterVardyFinanceSeeder;
use Tests\TenantTestCase;

class IvendiServiceTest extends TenantTestCase
{

    public function testGoodRequestCalculation()
    {
        Http::fake([
            "https://quoteware3.ivendi.com/quotes" => Http::response($this->responseFromDocumentation()),
            "*" => Http::response([], 400),
        ]);

        $this->seed(FinanceSettingSeeder::class);
        $this->seed(IvendiUsernameSeeder::class);
        Settings::update('finance-ivendi-enabled', true);
        Settings::update('finance-ivendi-username', 'foo-baz-bar');
        $provider = FinanceServiceHelper::initializeForSite();

        $result = $provider->calculate(new FinanceRequestData(
            uuid: 'abc123',
            registration_number: 'SD12 ABC',
            cap_id: 1231122,
            dealer_id: 'abc123',
            engine_size: 1999,
            condition: 'new',
            registration_date: Carbon::create(2020, 1, 1),
            term: 60,
            mileage: 100000,
            annual_mileage: 13000,
            price: 23456,
            deposit: 3000,
            credit_rating: 'excellent',
        ));

        Http::assertSent(function (Request $request) {
            return $request->url() === 'https://quoteware3.ivendi.com/quotes'
                && $request['Credentials']['Username'] === 'foo-baz-bar'
                && $request['QuoteRequests']['GlobalRequestParameters']['Term'] === 60
                && $request['QuoteRequests']['Requests'][0]['Figures']['CashPrice'] == 23456;
        });

        $this->assertEquals(2, $result->count());
        $finance = $result->get('HP');
        $this->assertEquals(136.92, $finance->monthly_price);
        $this->assertEquals(52, $finance->number_of_payments);
        $this->assertEquals(9.9, $finance->apr);
        $this->assertEquals(1000, $finance->total_deposit);
    }

    public function testRequestSubmissionViaJob()
    {
        Http::fake([
            "https://quoteware3.ivendi.com/quotes" => Http::response($this->responseFromDocumentation()),
            "*" => Http::response([], 400),
        ]);

        $this->seed(FinanceSettingSeeder::class);
        $this->seed(IvendiUsernameSeeder::class);
        Settings::update('finance-ivendi-enabled', true);
        Settings::update('finance-ivendi-username', 'foo-baz-bar');

        $dealership = Dealership::factory()->create([
            'location_finance' => 'abc123',
            'data' => [
                'ivendi' => 'test123'
            ]
        ]);
        $vehicle = Vehicle::factory()->create([
            'uuid' =>  'abc123',
            'registration_number' => 'SD12 ABC',
            'cap_id' => 1231122,
            'dealership_id' => $dealership->id,
            'engine_size_cc' => 1999,
            'is_new' => true,
            'first_registration_date' => Carbon::create(2020, 1, 1),
            'odometer_mi' => 100000,
            'price' => 23456,
            'type' => 'car',
        ]);

        $vehicle->financeExamples()->delete();
        (new RunFinanceFetchForVehicle($vehicle, 60, 13000, 10, 'excellent'))->handle();

        Http::assertSent(function (Request $request) {
            return $request->url() === 'https://quoteware3.ivendi.com/quotes'
                && $request['Credentials']['Username'] === 'foo-baz-bar'
                && $request['QuoteRequests']['GlobalRequestParameters']['Term'] === 60
                && $request['QuoteRequests']['Requests'][0]['Figures']['CashPrice'] == 23456;
        });

        $this->assertEquals(2, $vehicle->financeExamples()->count());

        $finance = $vehicle
            ->financeExamples
            ->filter(fn($finance_example) => $finance_example['finance_type'] == 'HP')
            ->first();

        $this->assertEquals(136.92, $finance->monthly_price);
        $this->assertEquals(52, $finance->number_of_payments);
        $this->assertEquals(9.9, $finance->apr);
        $this->assertEquals(1000, $finance->total_deposit);
    }

    public function testRiskBasedFinance()
    {
        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'type' => 'boolean',
            'value' => true,
            'config_key' => 'finance-risk-based-pricing-enabled'
        ]);

        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'type' => 'string',
            'value' => '1',
            'config_key' => 'finance-risk-based-pricing-finance-options'
        ]);

        $this->seed(FinanceSettingSeeder::class);
        $this->seed(IvendiUsernameSeeder::class);
        Settings::update('finance-risk-based-pricing-enabled', true);
        Settings::update('finance-risk-based-pricing-finance-options', '1');
        Settings::update('finance-ivendi-enabled', true);
        Settings::update('finance-ivendi-username', 'foo-baz-bar');
        $provider = FinanceServiceHelper::initializeForSite();

        $dealership = Dealership::factory()->create([
            'location_finance' => 'abc123',
            'data' => [
                'ivendi' => 'test123'
            ]
        ]);

        $vehicle = Vehicle::factory()->create([
            'uuid' =>  'abc123',
            'registration_number' => 'SD12 ABC',
            'cap_id' => 1231122,
            'dealership_id' => $dealership->id,
            'engine_size_cc' => 1999,
            'is_new' => true,
            'first_registration_date' => Carbon::create(2020, 1, 1),
            'odometer_mi' => 100000,
            'price' => 23456,
            'type' => 'car',
        ]);

        // add vehicle attribute for finance_option
        $attribute = VehicleAttribute::factory()->create([
            'name' => 'finance option',
            'model' => 'vehicle',
            'slug' => 'finance_option',
            'type' => 'string',
            'count' => 1
        ]);

        // set attribute value
        VehicleAttributeValue::factory()->create([
            'attribute_id' => $attribute->id,
            'owner_id' => $vehicle->id,
            'owner_type' => 'vehicle',
            'slug' => 'finance_option',
            'type' => 'string',
            'value' => '1'
        ]);

        $this->assertCount(1, VehicleAttribute::all());
        $this->assertCount(1, VehicleAttributeValue::all());
        $this->assertFalse($provider->isEligibleForCreditRatingFinance($vehicle));

        $dealership_with_risk_based = Dealership::factory()->create([
            'location_finance' => 'abc123',
            'data' => [
                'ivendi' => 'test123',
                'finance-risk-based-pricing' => true
            ]
        ]);

        $vehicle_for_risk_based = Vehicle::factory()->create([
            'uuid' =>  'abc123',
            'registration_number' => 'SD12 ABC',
            'cap_id' => 1231122,
            'dealership_id' => $dealership_with_risk_based->id,
            'engine_size_cc' => 1999,
            'is_new' => true,
            'first_registration_date' => Carbon::create(2020, 1, 1),
            'odometer_mi' => 100000,
            'price' => 23456,
            'type' => 'car',
        ]);

        VehicleAttributeValue::factory()->create([
            'attribute_id' => $attribute->id,
            'owner_id' => $vehicle_for_risk_based->id,
            'slug' => 'finance_option',
            'type' => 'string',
            'value' => '1'
        ]);

        $this->assertTrue($provider->isEligibleForCreditRatingFinance($vehicle_for_risk_based));
    }

    public function testPeterVardyRiskBasedFinanceIneligibleOption()
    {
        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'type' => 'boolean',
            'value' => true,
            'config_key' => 'finance-risk-based-pricing-enabled'
        ]);

        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'type' => 'string',
            'value' => '1',
            'config_key' => 'finance-risk-based-pricing-finance-options'
        ]);

        $this->seed(FinanceSettingSeeder::class);
        $this->seed(IvendiUsernameSeeder::class);
        $this->seed(PeterVardyFinanceSeeder::class);
        Settings::update('finance-ivendi-enabled', true);
        Settings::update('finance-ivendi-username', 'foo-baz-bar');
        Settings::update('vehicle-finance-type', 'peter-vardy');
        $provider = FinanceServiceHelper::initializeForSite();

        $dealership = Dealership::factory()->create([
            'location_finance' => 'abc123',
            'data' => [
                'ivendi' => 'test123',
                'finance-risk-based-pricing' => true
            ]
        ]);

        $vehicle = Vehicle::factory()->create([
            'uuid' =>  'abc123',
            'registration_number' => 'SD12 ABC',
            'cap_id' => 1231122,
            'dealership_id' => $dealership->id,
            'engine_size_cc' => 1999,
            'is_new' => true,
            'first_registration_date' => Carbon::create(2020, 1, 1),
            'odometer_mi' => 100000,
            'price' => 23456,
            'type' => 'car',
        ]);

        // add vehicle attribute for finance_option
        $attribute = VehicleAttribute::factory()->create([
            'name' => 'finance option',
            'model' => 'vehicle',
            'slug' => 'finance_option',
            'type' => 'string',
            'count' => 1
        ]);

        // set attribute value
        VehicleAttributeValue::factory()->create([
            'attribute_id' => $attribute->id,
            'owner_id' => $vehicle->id,
            'owner_type' => 'vehicle',
            'slug' => 'finance_option',
            'type' => 'string',
            'value' => '2'
        ]);

        $this->assertFalse($provider->isEligibleForCreditRatingFinance($vehicle));
    }

    public function testPeterVardyRiskBasedFinanceIneligibleDealer()
    {
        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'type' => 'boolean',
            'value' => true,
            'config_key' => 'finance-risk-based-pricing-enabled'
        ]);

        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'type' => 'string',
            'value' => '1',
            'config_key' => 'finance-risk-based-pricing-finance-options'
        ]);

        $this->seed(FinanceSettingSeeder::class);
        $this->seed(PeterVardyFinanceSeeder::class);
        Settings::update('finance-ivendi-enabled', true);
        Settings::update('finance-ivendi-api_key', 'foo-baz-bar');
        Settings::update('vehicle-finance-type', 'peter-vardy');
        $provider = FinanceServiceHelper::initializeForSite();

        $dealership = Dealership::factory()->create([
            'location_finance' => 'abc123',
            'data' => [
                'ivendi' => 'test123',
            ]
        ]);

        $vehicle = Vehicle::factory()->create([
            'uuid' =>  'abc123',
            'registration_number' => 'SD12 ABC',
            'cap_id' => 1231122,
            'dealership_id' => $dealership->id,
            'engine_size_cc' => 1999,
            'is_new' => true,
            'first_registration_date' => Carbon::create(2020, 1, 1),
            'odometer_mi' => 100000,
            'price' => 23456,
            'type' => 'car',
        ]);

        // add vehicle attribute for finance_option
        $attribute = VehicleAttribute::factory()->create([
            'name' => 'finance option',
            'model' => 'vehicle',
            'slug' => 'finance_option',
            'type' => 'string',
            'count' => 1
        ]);

        // set attribute value
        VehicleAttributeValue::factory()->create([
            'attribute_id' => $attribute->id,
            'owner_id' => $vehicle->id,
            'owner_type' => 'vehicle',
            'slug' => 'finance_option',
            'type' => 'string',
            'value' => '2'
        ]);

        $this->assertFalse($provider->isEligibleForCreditRatingFinance($vehicle));
    }

    public function testPeterVardyRiskBasedFinanceIneligibleFuelType()
    {
        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'type' => 'boolean',
            'value' => true,
            'config_key' => 'finance-risk-based-pricing-enabled'
        ]);

        Settings::make([
            'tab' => 'foo-baz',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'type' => 'string',
            'value' => '1',
            'config_key' => 'finance-risk-based-pricing-finance-options'
        ]);

        $this->seed(FinanceSettingSeeder::class);
        $this->seed(PeterVardyFinanceSeeder::class);
        Settings::update('finance-ivendi-enabled', true);
        Settings::update('finance-ivendi-api_key', 'foo-baz-bar');
        Settings::update('vehicle-finance-type', 'peter-vardy');
        $provider = FinanceServiceHelper::initializeForSite();

        $dealership = Dealership::factory()->create([
            'location_finance' => 'abc123',
            'data' => [
                'ivendi' => 'test123',
                'finance-risk-based-pricing' => true
            ]
        ]);

        $fuel_type = FuelType::factory()->create([
            'name' => 'electric'
        ]);

        $vehicle = Vehicle::factory()->create([
            'uuid' =>  'abc123',
            'registration_number' => 'SD12 ABC',
            'cap_id' => 1231122,
            'dealership_id' => $dealership->id,
            'engine_size_cc' => 1999,
            'is_new' => true,
            'first_registration_date' => Carbon::create(2020, 1, 1),
            'odometer_mi' => 100000,
            'price' => 23456,
            'type' => 'car',
            'fuel_type_id' => $fuel_type->id
        ]);

        // add vehicle attribute for finance_option
        $attribute = VehicleAttribute::factory()->create([
            'name' => 'finance option',
            'model' => 'vehicle',
            'slug' => 'finance_option',
            'type' => 'string',
            'count' => 1
        ]);

        // set attribute value
        VehicleAttributeValue::factory()->create([
            'attribute_id' => $attribute->id,
            'owner_id' => $vehicle->id,
            'owner_type' => 'vehicle',
            'slug' => 'finance_option',
            'type' => 'string',
            'value' => '1'
        ]);

        $this->assertFalse($provider->isEligibleForCreditRatingFinance($vehicle));
    }

    public function testDefaultFinanceService()
    {
        $this->seed(FinanceSettingSeeder::class);
        Settings::update('finance-ivendi-enabled', true);
        Settings::update('finance-ivendi-api_key', 'foo-baz-bar');

        $this->assertInstanceOf(IVendi::class, FinanceServiceHelper::initializeForSite());
        $this->assertNotInstanceOf(IVendiPeterVardy::class, FinanceServiceHelper::initializeForSite());
    }

    public function testPeterVardyFinanceService()
    {
        $this->seed(FinanceSettingSeeder::class);
        $this->seed(PeterVardyFinanceSeeder::class);
        Settings::update('finance-ivendi-enabled', true);
        Settings::update('finance-ivendi-api_key', 'foo-baz-bar');
        Settings::update('vehicle-finance-type', 'peter-vardy');

        $this->assertInstanceOf(IVendiPeterVardy::class, FinanceServiceHelper::initializeForSite());
    }

    public function testPeterVardyFinanceRetrieval()
    {
        Http::fake([
            "https://quoteware3.ivendi.com/quotes" => Http::response($this->responseFromDocumentation()),
            "*" => Http::response([], 400),
        ]);

        $this->seed(FinanceSettingSeeder::class);
        $this->seed(IvendiUsernameSeeder::class);
        $this->seed(PeterVardyFinanceSeeder::class);
        Settings::update('finance-ivendi-enabled', true);
        Settings::update('finance-ivendi-username', 'foo-baz-bar');
        Settings::update('vehicle-finance-type', 'peter-vardy');

        $provider = FinanceServiceHelper::initializeForSite();

        $this->assertInstanceOf(IVendiPeterVardy::class, $provider);

        $result = $provider->calculate(new FinanceRequestData(
            uuid: 'abc123',
            registration_number: 'SD12 ABC',
            cap_id: 1231122,
            dealer_id: 'abc123',
            engine_size: 1999,
            condition: 'new',
            registration_date: Carbon::create(2020, 1, 1),
            term: 60,
            mileage: 100000,
            annual_mileage: 13000,
            price: 23456,
            deposit: 3000,
            credit_rating: 'excellent',
            finance_option: 2,
        ));

        Http::assertSent(function (Request $request) {
            return $request->url() === 'https://quoteware3.ivendi.com/quotes'
                && $request['Credentials']['Username'] === 'foo-baz-bar'
                && $request['QuoteRequests']['GlobalRequestParameters']['Term'] === 60
                && $request['QuoteRequests']['Requests'][0]['Figures']['CashPrice'] == 23456;
        });

        $this->assertEquals(1, $result->count());

        $required_string = 'c best buy';
        $this->assertStringContainsString(strtoupper($required_string), strtoupper($result->first()->product_name));

        $finance = $result->first();
        $this->assertEquals(136.92, $finance->monthly_price);
        $this->assertEquals(52, $finance->number_of_payments);
        $this->assertEquals(9.9, $finance->apr);
        $this->assertEquals(1000, $finance->total_deposit);
        $this->assertEquals(7000, $finance->cash_price);
        $this->assertEquals(0, $finance->dealer_deposit);
        $this->assertEquals(3000, $finance->customer_deposit);
        $this->assertEquals(8393.68, $finance->payable_amount);
        $this->assertEquals(10, $finance->option_to_purchase_fee);
        $this->assertEquals(0, $finance->documentation_fee);
    }

    public function testPeterVardyFinanceRetrievalRiskBased()
    {
        Http::fake([
            "https://quoteware3.ivendi.com/quotes" => Http::response($this->responseFromDocumentation()),
            "*" => Http::response([], 400),
        ]);

        $this->seed(FinanceSettingSeeder::class);
        $this->seed(IvendiUsernameSeeder::class);
        $this->seed(PeterVardyFinanceSeeder::class);
        Settings::update('finance-ivendi-enabled', true);
        Settings::update('finance-ivendi-username', 'foo-baz-bar');
        Settings::update('vehicle-finance-type', 'peter-vardy');

        $provider = FinanceServiceHelper::initializeForSite();

        $this->assertInstanceOf(IVendiPeterVardy::class, $provider);

        $dealership_with_risk_based = Dealership::factory()->create([
            'location_finance' => 'abc123',
            'data' => [
                'ivendi' => 'test123',
                'finance-risk-based-pricing' => true
            ]
        ]);

        $vehicle_for_risk_based = Vehicle::factory()->create([
            'uuid' =>  'abc123',
            'registration_number' => 'SD12 ABC',
            'cap_id' => 1231122,
            'dealership_id' => $dealership_with_risk_based->id,
            'engine_size_cc' => 1999,
            'is_new' => true,
            'first_registration_date' => Carbon::create(2020, 1, 1),
            'odometer_mi' => 100000,
            'price' => 23456,
            'type' => 'car',
        ]);

        $this->assertFalse($provider->isEligibleForCreditRatingFinance($vehicle_for_risk_based));

        $result = $provider->calculateRiskBased(new FinanceRequestData(
            uuid: $vehicle_for_risk_based->uuid,
            registration_number: $vehicle_for_risk_based->registration_number,
            cap_id: $vehicle_for_risk_based->cap_id,
            dealer_id: $dealership_with_risk_based->id,
            engine_size: $vehicle_for_risk_based->engine_size_cc,
            condition: 'new',
            registration_date: Carbon::create(2020, 1, 1),
            term: 60,
            mileage: 100000,
            annual_mileage: 13000,
            price: 23456,
            deposit: 3000,
            credit_rating: 'excellent',
            finance_option: 2,
        ));

        Http::assertSent(function (Request $request) {
            return $request->url() === 'https://quoteware3.ivendi.com/quotes'
                && $request['Credentials']['Username'] === 'foo-baz-bar'
                && $request['QuoteRequests']['GlobalRequestParameters']['Term'] === 60
                && $request['QuoteRequests']['Requests'][0]['Figures']['CashPrice'] == 23456;
        });

        $this->assertEquals(2, $result->count());

        $result_array = $result->toArray();

        $this->assertArrayHasKey('HP', $result_array);
        $this->assertArrayHasKey('PCP', $result_array);

        // we expect the cheapest quote
        $this->assertEquals($result_array['PCP']['monthly_price'], 126.29);

        $finance = $result->first();
        $this->assertEquals(136.92, $finance->monthly_price);
        $this->assertEquals(52, $finance->number_of_payments);
        $this->assertEquals(9.9, $finance->apr);
        $this->assertEquals(1000, $finance->total_deposit);
        $this->assertEquals(7000, $finance->cash_price);
        $this->assertEquals(0, $finance->dealer_deposit);
        $this->assertEquals(3000, $finance->customer_deposit);
        $this->assertEquals(8393.68, $finance->payable_amount);
        $this->assertEquals(10, $finance->option_to_purchase_fee);
        $this->assertEquals(0, $finance->documentation_fee);
    }

    public function testServiceUnavailable()
    {
        Http::fake([
            "https://quoteware3.ivendi.com/quotes" => Http::response([], 400),
            "*" => Http::response([], 400),
        ]);

        $this->seed(FinanceSettingSeeder::class);
        $this->seed(IvendiUsernameSeeder::class);
        Settings::update('finance-ivendi-enabled', true);
        Settings::update('finance-ivendi-username', 'foo-baz-bar');
        $provider = FinanceServiceHelper::initializeForSite();

        $result = $provider->calculate(new FinanceRequestData(
            uuid: 'abc123',
            registration_number: 'SD12 ABC',
            cap_id: 1231122,
            dealer_id: 'abc123',
            engine_size: 1999,
            condition: 'new',
            registration_date: Carbon::create(2020, 1, 1),
            term: 60,
            mileage: 100000,
            annual_mileage: 13000,
            price: 23456,
            deposit: 3000,
            credit_rating: 'excellent',
        ));

        Http::assertSent(function (Request $request) {
            return $request->url() === 'https://quoteware3.ivendi.com/quotes'
                && $request['Credentials']['Username'] === 'foo-baz-bar'
                && $request['QuoteRequests']['GlobalRequestParameters']['Term'] === 60
                && $request['QuoteRequests']['Requests'][0]['Figures']['CashPrice'] == 23456;
        });

        $this->assertEmpty($result);
    }

    public function testNoQuotesAvailable()
    {
        Http::fake([
            "https://quoteware3.ivendi.com/quotes" => Http::response($this->responseFromDocumentationNoQuotes()),
                "*" => Http::response([], 400),
        ]);

        $this->seed(FinanceSettingSeeder::class);
        $this->seed(IvendiUsernameSeeder::class);
        Settings::update('finance-ivendi-enabled', true);
        Settings::update('finance-ivendi-username', 'foo-baz-bar');
        $provider = FinanceServiceHelper::initializeForSite();

        $result = $provider->calculate(new FinanceRequestData(
            uuid: 'abc123',
            registration_number: 'SD12 ABC',
            cap_id: 1231122,
            dealer_id: 'abc123',
            engine_size: 1999,
            condition: 'new',
            registration_date: Carbon::create(2020, 1, 1),
            term: 60,
            mileage: 100000,
            annual_mileage: 13000,
            price: 23456,
            deposit: 3000,
            credit_rating: 'excellent',
        ));

        Http::assertSent(function (Request $request) {
            return $request->url() === 'https://quoteware3.ivendi.com/quotes'
                && $request['Credentials']['Username'] === 'foo-baz-bar'
                && $request['QuoteRequests']['GlobalRequestParameters']['Term'] === 60
                && $request['QuoteRequests']['Requests'][0]['Figures']['CashPrice'] == 23456;
        });

        $this->assertEmpty($result);
    }

    public function testPVNoQuotesAvailable()
    {
        Http::fake([
            "https://quoteware3.ivendi.com/quotes" => Http::response($this->responseFromDocumentationNoQuotes()),
            "*" => Http::response([], 400),
        ]);

        $this->seed(FinanceSettingSeeder::class);
        $this->seed(IvendiUsernameSeeder::class);
        $this->seed(PeterVardyFinanceSeeder::class);
        Settings::update('finance-ivendi-enabled', true);
        Settings::update('finance-ivendi-username', 'foo-baz-bar');
        Settings::update('vehicle-finance-type', 'peter-vardy');
        $provider = FinanceServiceHelper::initializeForSite();

        $result = $provider->calculate(new FinanceRequestData(
            uuid: 'abc123',
            registration_number: 'SD12 ABC',
            cap_id: 1231122,
            dealer_id: 'abc123',
            engine_size: 1999,
            condition: 'new',
            registration_date: Carbon::create(2020, 1, 1),
            term: 60,
            mileage: 100000,
            annual_mileage: 13000,
            price: 23456,
            deposit: 3000,
            credit_rating: 'excellent',
        ));

        Http::assertSent(function (Request $request) {
            return $request->url() === 'https://quoteware3.ivendi.com/quotes'
                && $request['Credentials']['Username'] === 'foo-baz-bar'
                && $request['QuoteRequests']['GlobalRequestParameters']['Term'] === 60
                && $request['QuoteRequests']['Requests'][0]['Figures']['CashPrice'] == 23456;
        });

        $this->assertEmpty($result);
    }

    private function responseFromDocumentation(): array
    {
        return json_decode('{
            "hasQuoteResults": true,
            "QuotedResultsUID": "d3d0350d-8ad5-477b-9408-2523d1bb42bd",
            "QuoteResults": [
                {
                    "hasResults": true,
                    "ProductRequestUID": "f9e788bc-dbe2-4718-b6cc-ef63cc83a193",
                    "QuotedResultUID": "a43f8b64-db2d-4388-b6bf-247e0a89c2c1",
                    "QuoteeUID": "268e8202-338e-4b26-a6fe-74bcdab0a357",
                    "QuoteRequest": null,
                    "Results": [
                        {
                            "Asset": {
                                "__type": "QuotewareV3.Core.ResultAssetMotorVehicle, QuotewareV3.Core",
                                "Age": 48,
                                "AgeIdentifier": "17",
                                "AgeUnit": "Months",
                                "Class": "Car",
                                "Condition": "Used",
                                "CurrentOdometerReading": 1300,
                                "EntityType": "None",
                                "Identity": "CX17HHA",
                                "IdentityType": "VRM",
                                "CapCodeOverridden": null,
                                "RegistrationDate": "/Date(1490313600000)/",
                                "RegistrationMark": "CX17HHA",
                                "RequestedAnnualDistance": 10000,
                                "RequestedDeposit": 1000,
                                "RequestedTerm": 54,
                                "RequestedTermUnit": "Default",
                                "Source": "RegionCurrent",
                                "StockIdentity": "39736298",
                                "StockingDays": 0,
                                "StockLocation": null,
                                "TermDistance": 46300,
                                "VrmEnabled": true
                            },
                            "hasProductGroup": true,
                            "hasProductGroupWithinPaymentRange": "False",
                            "ProductGroups": [
                                {
                                    "FacilityType": "HP",
                                    "hasProductQuote": true,
                                    "hasProductQuoteWithinPaymentRange": "False",
                                    "ProductGroupUID": "cc1c74e1-e24d-4b76-0a11-011eb60d5c1a",
                                    "ProductQuotes": [
                                        {
                                            "BaseCfcUrl": "https://carfinancechecker.com",
                                            "Commissions": [
                                                {
                                                    "Amount": 0,
                                                    "Method": "RRPDiscounting",
                                                    "Type": "None",
                                                    "Value": 0,
                                                    "VolumeBonus": 0
                                                }
                                            ],
                                            "ExpiryDate": "/Date(1615939200000+0000)/",
                                            "FacilitatorCode": "BLA",
                                            "FacilityType": "HP",
                                            "Figures": {
                                                "AcceptanceFee": 0,
                                                "AcceptanceFeeSpread": false,
                                                "AdditionalDeposit": 0,
                                                "Advance": 6000,
                                                "AER": 9.85,
                                                "APR": 9.9,
                                                "Asset": {
                                                    "__type": "QuotewareV3.Core.ProductQuoteAssetMotorVehicle, QuotewareV3.Core",
                                                    "AdjustedDistance": false,
                                                    "AdjustedTerm": false,
                                                    "AnnualDistanceQuoted": 10000,
                                                    "ChargePerOverDistanceUnit": 0,
                                                    "Equity": 0,
                                                    "FinanceDepositAllowance": 0,
                                                    "FinanceDepositAllowances": null,
                                                    "NegativeEquity": 0,
                                                    "OutstandingSettlement": 0,
                                                    "OutstandingSource": null,
                                                    "PartExchange": 0,
                                                    "RVDataSet": null,
                                                    "RVDataSetMonth": 0,
                                                    "RVDataSetYear": 0,
                                                    "RVPercentageUsed": 0,
                                                    "ValueAddedProducts": null,
                                                    "VATQualifying": "None",
                                                    "ExcludeVatAsDeposit": "None"
                                                },
                                                "Balloon": 0,
                                                "BalloonAtTerm": false,
                                                "BaseRate": 9.9,
                                                "CustomerRate": 9.9,
                                                "EarlySettlements": null,
                                                "FinalPayment": 136.92,
                                                "FirstPayment": 136.92,
                                                "FlatRate": 5.13,
                                                "FullFees": null,
                                                "InitialPause": 0,
                                                "InterestCharges": 1383.68,
                                                "InterestRate": 9.43,
                                                "InterestRateRateType": "Yield",
                                                "NumberOfRegularPayments": 52,
                                                "OptionFeeSpread": false,
                                                "OptionToPurchaseFee": 10,
                                                "PaymentSchedules": [
                                                    {
                                                        "Amount": 136.92,
                                                        "Number": 54,
                                                        "Period": 1
                                                    }
                                                ],
                                                "ProductRateType": "APR",
                                                "RegularPayment": 136.92,
                                                "Repossession": 2797.89,
                                                "Subsidy": 0,
                                                "Term": 54,
                                                "Termination": 4196.84,
                                                "TermQuoted": 54,
                                                "TermUnit": "Default",
                                                "TotalCashPrice": 7000,
                                                "TotalCharges": 1393.68,
                                                "TotalDeposit": 1000,
                                                "TotalPayable": 8393.68,
                                                "ValueAddedTax": 0,
                                                "ValueAddedTaxAsDeposit": false
                                            },
                                            "FunderCode": "BLA",
                                            "FunderProductCode": "HP",
                                            "ParentQuoteUID": null,
                                            "ProductCode": "HP",
                                            "ProductName": "Blackhorse HP C best buy",
                                            "ProductUID": "b644028d-c545-4a54-ba3a-004007009d3b",
                                            "CreditTier": "None",
                                            "QuotedDateTime": "/Date(1614772549066+0000)/",
                                            "QuoteeProductCode": "HP",
                                            "QuoteeUid": "268e8202-338e-4b26-a6fe-74bcdab0a357",
                                            "QuoteUID": "03031155-49f1-41f3-b332-5235d0263664",
                                            "WithinPaymentRange": "None",
                                            "Errors": null,
                                            "hasErrors": false,
                                            "hasWarnings": false,
                                            "Warnings": null
                                        }
                                    ],
                                    "QuotedProductGroupUID": "0e08878c-f20a-4e5c-b09b-85ca2db39a4f",
                                    "Errors": null,
                                    "hasErrors": false,
                                    "hasWarnings": false,
                                    "Warnings": null
                                },
                                {
                                    "FacilityType": "PCP",
                                    "hasProductQuote": true,
                                    "hasProductQuoteWithinPaymentRange": "False",
                                    "ProductGroupUID": "d17cc875-b043-4d99-94e0-d56bbf556054",
                                    "ProductQuotes": [
                                    {
                                            "BaseCfcUrl": "https://carfinancechecker.com",
                                            "Commissions": [
                                                {
                                                    "Amount": 0,
                                                    "Method": "RRPDiscounting",
                                                    "Type": "None",
                                                    "Value": 0,
                                                    "VolumeBonus": 0
                                                }
                                            ],
                                            "ExpiryDate": "/Date(1615939200000+0000)/",
                                            "FacilitatorCode": "BLA",
                                            "FacilityType": "PCP",
                                            "Figures": {
                                                "AcceptanceFee": 0,
                                                "AcceptanceFeeSpread": false,
                                                "AdditionalDeposit": 0,
                                                "Advance": 6000,
                                                "AER": 9.83,
                                                "APR": 9.9,
                                                "Asset": {
                                                    "__type": "QuotewareV3.Core.ProductQuoteAssetMotorVehicle, QuotewareV3.Core",
                                                    "AdjustedDistance": false,
                                                    "AdjustedTerm": true,
                                                    "AnnualDistanceQuoted": 10000,
                                                    "ChargePerOverDistanceUnit": 0,
                                                    "Equity": 0,
                                                    "FinanceDepositAllowance": 0,
                                                    "FinanceDepositAllowances": null,
                                                    "NegativeEquity": 0,
                                                    "OutstandingSettlement": 0,
                                                    "OutstandingSource": null,
                                                    "PartExchange": 0,
                                                    "RVDataSet": "Blackhorse Base",
                                                    "RVDataSetMonth": 3,
                                                    "RVDataSetYear": 2017,
                                                    "RVPercentageUsed": 100,
                                                    "ValueAddedProducts": null,
                                                    "VATQualifying": "None",
                                                    "ExcludeVatAsDeposit": "None"
                                                },
                                                "Balloon": 2741,
                                                "BalloonAtTerm": true,
                                                "BaseRate": 9.9,
                                                "CustomerRate": 9.9,
                                                "EarlySettlements": null,
                                                "FinalPayment": 2751,
                                                "FirstPayment": 999.29,
                                                "FlatRate": 5.06,
                                                "FullFees": null,
                                                "InitialPause": 0,
                                                "InterestCharges": 1287.44,
                                                "InterestRate": 9.42,
                                                "InterestRateRateType": "Yield",
                                                "NumberOfRegularPayments": 35,
                                                "OptionFeeSpread": false,
                                                "OptionToPurchaseFee": 10,
                                                "PaymentSchedules": [
                                                    {
                                                        "Amount": 999.29,
                                                        "Number": 36,
                                                        "Period": 1
                                                    },
                                                    {
                                                        "Amount": 2751,
                                                        "Number": 1,
                                                        "Period": 2
                                                    }
                                                ],
                                                "ProductRateType": "APR",
                                                "RegularPayment": 999.29,
                                                "Repossession": 2765.81,
                                                "Subsidy": 0,
                                                "Term": 37,
                                                "Termination": 4148.72,
                                                "TermQuoted": 37,
                                                "TermUnit": "Default",
                                                "TotalCashPrice": 7000,
                                                "TotalCharges": 1297.44,
                                                "TotalDeposit": 1000,
                                                "TotalPayable": 8297.44,
                                                "ValueAddedTax": 0,
                                                "ValueAddedTaxAsDeposit": false
                                            },
                                            "FunderCode": "BLA",
                                            "FunderProductCode": "54",
                                            "ParentQuoteUID": null,
                                            "ProductCode": "54",
                                            "ProductName": "Blackhorse PCP APR - Standard Rate",
                                            "ProductUID": "113c7b7a-0491-44ed-b1b9-85524e9be316",
                                            "CreditTier": "None",
                                            "QuotedDateTime": "/Date(1614772549066+0000)/",
                                            "QuoteeProductCode": "54",
                                            "QuoteeUid": "268e8202-338e-4b26-a6fe-74bcdab0a357",
                                            "QuoteUID": "03031155-4954-46a4-9db5-2bc42f3cc2e8",
                                            "WithinPaymentRange": "None",
                                            "Errors": null,
                                            "hasErrors": false,
                                            "hasWarnings": false,
                                            "Warnings": null
                                        },
                                        {
                                            "BaseCfcUrl": "https://carfinancechecker.com",
                                            "Commissions": [
                                                {
                                                    "Amount": 0,
                                                    "Method": "RRPDiscounting",
                                                    "Type": "None",
                                                    "Value": 0,
                                                    "VolumeBonus": 0
                                                }
                                            ],
                                            "ExpiryDate": "/Date(1615939200000+0000)/",
                                            "FacilitatorCode": "BLA",
                                            "FacilityType": "PCP",
                                            "Figures": {
                                                "AcceptanceFee": 0,
                                                "AcceptanceFeeSpread": false,
                                                "AdditionalDeposit": 0,
                                                "Advance": 6000,
                                                "AER": 9.83,
                                                "APR": 9.9,
                                                "Asset": {
                                                    "__type": "QuotewareV3.Core.ProductQuoteAssetMotorVehicle, QuotewareV3.Core",
                                                    "AdjustedDistance": false,
                                                    "AdjustedTerm": true,
                                                    "AnnualDistanceQuoted": 10000,
                                                    "ChargePerOverDistanceUnit": 0,
                                                    "Equity": 0,
                                                    "FinanceDepositAllowance": 0,
                                                    "FinanceDepositAllowances": null,
                                                    "NegativeEquity": 0,
                                                    "OutstandingSettlement": 0,
                                                    "OutstandingSource": null,
                                                    "PartExchange": 0,
                                                    "RVDataSet": "Blackhorse Base",
                                                    "RVDataSetMonth": 3,
                                                    "RVDataSetYear": 2017,
                                                    "RVPercentageUsed": 100,
                                                    "ValueAddedProducts": null,
                                                    "VATQualifying": "None",
                                                    "ExcludeVatAsDeposit": "None"
                                                },
                                                "Balloon": 2741,
                                                "BalloonAtTerm": true,
                                                "BaseRate": 9.9,
                                                "CustomerRate": 9.9,
                                                "EarlySettlements": null,
                                                "FinalPayment": 2751,
                                                "FirstPayment": 126.29,
                                                "FlatRate": 5.06,
                                                "FullFees": null,
                                                "InitialPause": 0,
                                                "InterestCharges": 1287.44,
                                                "InterestRate": 9.42,
                                                "InterestRateRateType": "Yield",
                                                "NumberOfRegularPayments": 35,
                                                "OptionFeeSpread": false,
                                                "OptionToPurchaseFee": 10,
                                                "PaymentSchedules": [
                                                    {
                                                        "Amount": 126.29,
                                                        "Number": 36,
                                                        "Period": 1
                                                    },
                                                    {
                                                        "Amount": 2751,
                                                        "Number": 1,
                                                        "Period": 2
                                                    }
                                                ],
                                                "ProductRateType": "APR",
                                                "RegularPayment": 126.29,
                                                "Repossession": 2765.81,
                                                "Subsidy": 0,
                                                "Term": 37,
                                                "Termination": 4148.72,
                                                "TermQuoted": 37,
                                                "TermUnit": "Default",
                                                "TotalCashPrice": 7000,
                                                "TotalCharges": 1297.44,
                                                "TotalDeposit": 1000,
                                                "TotalPayable": 8297.44,
                                                "ValueAddedTax": 0,
                                                "ValueAddedTaxAsDeposit": false
                                            },
                                            "FunderCode": "BLA",
                                            "FunderProductCode": "54",
                                            "ParentQuoteUID": null,
                                            "ProductCode": "54",
                                            "ProductName": "Blackhorse PCP APR - Standard Rate",
                                            "ProductUID": "113c7b7a-0491-44ed-b1b9-85524e9be316",
                                            "CreditTier": "None",
                                            "QuotedDateTime": "/Date(1614772549066+0000)/",
                                            "QuoteeProductCode": "54",
                                            "QuoteeUid": "268e8202-338e-4b26-a6fe-74bcdab0a357",
                                            "QuoteUID": "03031155-4954-46a4-9db5-2bc42f3cc2e8",
                                            "WithinPaymentRange": "None",
                                            "Errors": null,
                                            "hasErrors": false,
                                            "hasWarnings": false,
                                            "Warnings": null
                                        },
                                        {
                                            "BaseCfcUrl": "https://carfinancechecker.com",
                                            "Commissions": null,
                                            "ExpiryDate": "/Date(1614772549066+0000)/",
                                            "FacilitatorCode": "BLA",
                                            "FacilityType": "PCP",
                                            "Figures": null,
                                            "FunderCode": "BLA",
                                            "FunderProductCode": "54",
                                            "ParentQuoteUID": null,
                                            "ProductCode": "54",
                                            "ProductName": "Ducati PCP c BEST BuY",
                                            "ProductUID": "c2a9e51d-4a7a-4c04-a8ec-e1eb24ee767a",
                                            "CreditTier": "None",
                                            "QuotedDateTime": "/Date(1614772549066+0000)/",
                                            "QuoteeProductCode": "54",
                                            "QuoteeUid": "268e8202-338e-4b26-a6fe-74bcdab0a357",
                                            "QuoteUID": "03031155-49cb-40df-bfa7-043ac9708dff",
                                            "WithinPaymentRange": "False",
                                            "Errors": {
                                                "hasPrivateErrors": false,
                                                "hasPublicErrors": true,
                                                "PublicErrors": [
                                                    {
                                                        "Message": "Motor Vehicle not allowed for product",
                                                        "Number": 522
                                                    },
                                                    {
                                                        "Message": "Motor Vehicle Class of Car not allowed",
                                                        "Number": 516
                                                    },
                                                    {
                                                        "Message": "Motor Vehicle Condition of Used not allowed",
                                                        "Number": 517
                                                    }
                                                ]
                                            },
                                            "hasErrors": true,
                                            "hasWarnings": false,
                                            "Warnings": null
                                        },
                                        {
                                            "BaseCfcUrl": "https://carfinancechecker.com",
                                            "Commissions": null,
                                            "ExpiryDate": "/Date(1614772549066+0000)/",
                                            "FacilitatorCode": "BLA",
                                            "FacilityType": "PCP",
                                            "Figures": null,
                                            "FunderCode": "BLA",
                                            "FunderProductCode": "54",
                                            "ParentQuoteUID": null,
                                            "ProductCode": "54",
                                            "ProductName": "Ducati PCP 1",
                                            "ProductUID": "967726b1-8e03-4524-91f4-5246699c4739",
                                            "CreditTier": "None",
                                            "QuotedDateTime": "/Date(1614772549066+0000)/",
                                            "QuoteeProductCode": "54",
                                            "QuoteeUid": "268e8202-338e-4b26-a6fe-74bcdab0a357",
                                            "QuoteUID": "03031155-4932-47a3-98d7-dd830585054c",
                                            "WithinPaymentRange": "False",
                                            "Errors": {
                                                "hasPrivateErrors": false,
                                                "hasPublicErrors": true,
                                                "PublicErrors": [
                                                    {
                                                        "Message": "Motor Vehicle not allowed for product",
                                                        "Number": 522
                                                    },
                                                    {
                                                        "Message": "Motor Vehicle Class of Car not allowed",
                                                        "Number": 516
                                                    },
                                                    {
                                                        "Message": "Motor Vehicle Condition of Used not allowed",
                                                        "Number": 517
                                                    }
                                                ]
                                            },
                                            "hasErrors": true,
                                            "hasWarnings": false,
                                            "Warnings": null
                                        },
                                        {
                                            "BaseCfcUrl": "https://carfinancechecker.com",
                                            "Commissions": null,
                                            "ExpiryDate": "/Date(1614772549066+0000)/",
                                            "FacilitatorCode": "NOR",
                                            "FacilityType": "PCP",
                                            "Figures": null,
                                            "FunderCode": "NOR",
                                            "FunderProductCode": "NORPCP01",
                                            "ParentQuoteUID": null,
                                            "ProductCode": "NORPCP01",
                                            "ProductName": "Northridge PCP",
                                            "ProductUID": "bf925f2c-75ca-47a0-8ee5-0fdce5a1b251",
                                            "CreditTier": "None",
                                            "QuotedDateTime": "/Date(1614772549066+0000)/",
                                            "QuoteeProductCode": "PCP: F Y CAR APR",
                                            "QuoteeUid": "268e8202-338e-4b26-a6fe-74bcdab0a357",
                                            "QuoteUID": "03031155-4913-40e7-97cb-935d724712e8",
                                            "WithinPaymentRange": "False",
                                            "Errors": {
                                                "hasPrivateErrors": true,
                                                "hasPublicErrors": false,
                                                "PrivateErrors": [
                                                    {
                                                        "Message": "Rate not found for RRP Discounting rate method",
                                                        "Number": 523
                                                    }
                                                ]
                                            },
                                            "hasErrors": true,
                                            "hasWarnings": false,
                                            "Warnings": null
                                        }
                                    ],
                                    "QuotedProductGroupUID": "9adf6e0f-3b24-4e6a-8de6-834a7a5182e0",
                                    "Errors": null,
                                    "hasErrors": false,
                                    "hasWarnings": false,
                                    "Warnings": null
                                }
                            ],
                            "ProductPackageUID": "f9e788bc-dbe2-4718-b6cc-ef63cc83a193",
                            "QuotedProductPackageUID": "aabd0e4c-b172-42a1-bccf-59769ef1e1e3",
                            "Request": null,
                            "ResultUID": "d99d5ad3-ebc7-45e4-b7e9-48a2a05765c2",
                            "Errors": null,
                            "hasErrors": false,
                            "hasWarnings": false,
                            "Warnings": null
                        }
                    ],
                    "Errors": null,
                    "hasErrors": false,
                    "hasWarnings": false,
                    "Warnings": null
                }
            ],
            "ResponseUID": "958ac3cc-5152-47ec-9db7-463eef976c75",
            "Errors": null,
            "hasErrors": false,
            "hasWarnings": false,
            "Warnings": null
        }', true);
    }

    private function responseFromDocumentationNoQuotes(): array
    {
        return json_decode('{
            "hasQuoteResults": true,
            "QuotedResultsUID": "d3d0350d-8ad5-477b-9408-2523d1bb42bd",
            "QuoteResults": [
                {
                    "hasResults": true,
                    "ProductRequestUID": "f9e788bc-dbe2-4718-b6cc-ef63cc83a193",
                    "QuotedResultUID": "a43f8b64-db2d-4388-b6bf-247e0a89c2c1",
                    "QuoteeUID": "268e8202-338e-4b26-a6fe-74bcdab0a357",
                    "QuoteRequest": null,
                    "Results": [
                        {
                            "Asset": {
                                "__type": "QuotewareV3.Core.ResultAssetMotorVehicle, QuotewareV3.Core",
                                "Age": 48,
                                "AgeIdentifier": "17",
                                "AgeUnit": "Months",
                                "Class": "Car",
                                "Condition": "Used",
                                "CurrentOdometerReading": 1300,
                                "EntityType": "None",
                                "Identity": "CX17HHA",
                                "IdentityType": "VRM",
                                "CapCodeOverridden": null,
                                "RegistrationDate": "/Date(1490313600000)/",
                                "RegistrationMark": "CX17HHA",
                                "RequestedAnnualDistance": 10000,
                                "RequestedDeposit": 1000,
                                "RequestedTerm": 54,
                                "RequestedTermUnit": "Default",
                                "Source": "RegionCurrent",
                                "StockIdentity": "39736298",
                                "StockingDays": 0,
                                "StockLocation": null,
                                "TermDistance": 46300,
                                "VrmEnabled": true
                            },
                            "hasProductGroup": true,
                            "hasProductGroupWithinPaymentRange": "False",
                            "ProductGroups": [
                                {
                                    "FacilityType": "HP",
                                    "hasProductQuote": true,
                                    "hasProductQuoteWithinPaymentRange": "False",
                                    "ProductGroupUID": "cc1c74e1-e24d-4b76-0a11-011eb60d5c1a",
                                    "ProductQuotes": [
                                        {
                                            "BaseCfcUrl": null,
                                            "Commissions": null,
                                            "ExpiryDate": "/Date(1695800529111)/",
                                            "FacilitatorCode": "POR",
                                            "FacilityType": "HP",
                                            "Figures": null,
                                            "FunderCode": "POR",
                                            "FunderProductCode": "PORHP",
                                            "FunderProductUID": "b23e2e3a-1c48-4705-a535-0e4ee7aebad1",
                                            "ParentQuoteUID": null,
                                            "ProductCode": "PORHP",
                                            "ProductName": "Porsche HP B Wee Gem",
                                            "ProductUID": "e47d0464-4b96-4d72-088d-f717b68e2b2c",
                                            "CreditTier": "None",
                                            "QuotedDateTime": "/Date(1695800529111)/",
                                            "QuoteeProductCode": "PORHP",
                                            "QuoteeUid": "046ca67d-43e9-4110-bd6b-0143a08ea9c0",
                                            "QuoteUID": "09270742-099a-4197-807c-3e24857ae30f",
                                            "Version": "3.0.1",
                                            "WithinPaymentRange": "False",
                                            "Errors": {
                                                "hasPrivateErrors": true,
                                                "hasPublicErrors": true,
                                                "PrivateErrors": [
                                                    {
                                                        "Message": "Rate not found for RRP Discounting rate method",
                                                        "Number": 523
                                                    }
                                                ],
                                                "PublicErrors": [
                                                    {
                                                        "Message": "Advance below minimum amount of 2000",
                                                        "Number": 506
                                                    },
                                                    {
                                                        "Message": "Total Deposit above maximum amount of 187900.",
                                                        "Number": 503
                                                    }
                                                ]
                                            },
                                            "hasErrors": true,
                                            "hasWarnings": false,
                                            "Warnings": null
                                        }
                                    ],
                                    "QuotedProductGroupUID": "0e08878c-f20a-4e5c-b09b-85ca2db39a4f",
                                    "Errors": null,
                                    "hasErrors": false,
                                    "hasWarnings": false,
                                    "Warnings": null
                                }
                            ],
                            "ProductPackageUID": "f9e788bc-dbe2-4718-b6cc-ef63cc83a193",
                            "QuotedProductPackageUID": "aabd0e4c-b172-42a1-bccf-59769ef1e1e3",
                            "Request": null,
                            "ResultUID": "d99d5ad3-ebc7-45e4-b7e9-48a2a05765c2",
                            "Errors": null,
                            "hasErrors": false,
                            "hasWarnings": false,
                            "Warnings": null
                        }
                    ],
                    "Errors": null,
                    "hasErrors": false,
                    "hasWarnings": false,
                    "Warnings": null
                }
            ],
            "ResponseUID": "958ac3cc-5152-47ec-9db7-463eef976c75",
            "Errors": null,
            "hasErrors": false,
            "hasWarnings": false,
            "Warnings": null
        }', true);
    }
}
