<?php

namespace Tests\Feature\Controllers;

use App\GraphService;
use App\Services\Matomo;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Mockery;
use Mtc\MercuryDataModels\Repositories\FeatureRepository;
use Tests\TenantTestCase;

class StatsControllerTest extends TenantTestCase
{
    protected function setUp(): void
    {
        parent::setUp();
        Carbon::setTestNow(Carbon::parse('2024-06-15'));
    }

    protected function tearDown(): void
    {
        Carbon::setTestNow();
        parent::tearDown();
    }

    public function testGraphEndpointReturnsOkStatus(): void
    {
        $this->mockGraphService();

        $response = $this->asTenant()->postJson('/graph', [
            'metrics' => 'leads',
            'date_range' => 7,
        ]);

        $response->assertStatus(200);
        $response->assertJsonPath('status', 'ok');
    }

    public function testGraphEndpointReturnsLabels(): void
    {
        $this->mockGraphService();

        $response = $this->asTenant()->postJson('/graph', [
            'metrics' => 'leads',
            'date_range' => 7,
        ]);

        $response->assertStatus(200);
        $response->assertJsonStructure([
            'status',
            'labels',
            'date_range',
            'type',
            'datasets',
        ]);
    }

    public function testGraphEndpointReturnsCorrectDateRange(): void
    {
        $this->mockGraphService();

        $response = $this->asTenant()->postJson('/graph', [
            'metrics' => 'vehicles',
            'date_range' => 14,
        ]);

        $response->assertStatus(200);
        $response->assertJsonPath('date_range', 14);
    }

    public function testGraphEndpointReturnsCorrectMetricType(): void
    {
        $this->mockGraphService();

        $response = $this->asTenant()->postJson('/graph', [
            'metrics' => 'visitors',
            'date_range' => 7,
        ]);

        $response->assertStatus(200);
        $response->assertJsonPath('type', 'visitors');
    }

    public function testGraphEndpointUsesDefaultValues(): void
    {
        $this->mockGraphService();

        $response = $this->asTenant()->postJson('/graph');

        $response->assertStatus(200);
        $response->assertJsonPath('date_range', 30);
        $response->assertJsonPath('type', 'leads');
    }

    public function testGraphEndpointReturnsDatasets(): void
    {
        $this->mockGraphServiceWithData();

        $response = $this->asTenant()->postJson('/graph', [
            'metrics' => 'leads',
            'date_range' => 3,
        ]);

        $response->assertStatus(200);
        $this->assertArrayHasKey('datasets', $response->json());
    }

    public function testGraphEndpointDatasetsHaveCorrectStructure(): void
    {
        $this->mockGraphServiceWithData();

        $response = $this->asTenant()->postJson('/graph', [
            'metrics' => 'leads',
            'date_range' => 3,
        ]);

        $response->assertStatus(200);
        $datasets = $response->json('datasets');

        if (!empty($datasets)) {
            $firstDataset = $datasets[0];
            $this->assertArrayHasKey('label', $firstDataset);
            $this->assertArrayHasKey('backgroundColor', $firstDataset);
            $this->assertArrayHasKey('data', $firstDataset);
            $this->assertArrayHasKey('tension', $firstDataset);
            $this->assertArrayHasKey('hidden', $firstDataset);
        }
    }

    public function testGraphEndpointReturnsFormattedDateLabels(): void
    {
        $this->mockGraphServiceWithData();

        $response = $this->asTenant()->postJson('/graph', [
            'metrics' => 'leads',
            'date_range' => 3,
        ]);

        $response->assertStatus(200);
        $labels = $response->json('labels');

        // Labels should be formatted dates like "15th Jun"
        $this->assertNotEmpty($labels);
        foreach ($labels as $label) {
            $this->assertMatchesRegularExpression('/\d{1,2}(st|nd|rd|th)\s+[A-Z][a-z]{2}/', $label);
        }
    }

    public function testGraphEndpointHandlesLeadsMetric(): void
    {
        $this->mockGraphService();

        $response = $this->asTenant()->postJson('/graph', [
            'metrics' => 'leads',
            'date_range' => 7,
        ]);

        $response->assertStatus(200);
        $response->assertJsonPath('status', 'ok');
    }

    public function testGraphEndpointHandlesVehiclesMetric(): void
    {
        $this->mockGraphService();

        $response = $this->asTenant()->postJson('/graph', [
            'metrics' => 'vehicles',
            'date_range' => 7,
        ]);

        $response->assertStatus(200);
        $response->assertJsonPath('status', 'ok');
    }

    public function testGraphEndpointHandlesVisitorsMetric(): void
    {
        $this->mockGraphService();

        $response = $this->asTenant()->postJson('/graph', [
            'metrics' => 'visitors',
            'date_range' => 7,
        ]);

        $response->assertStatus(200);
        $response->assertJsonPath('status', 'ok');
    }

    public function testGraphEndpointReturnsErrorOnException(): void
    {
        $mockService = Mockery::mock(GraphService::class);
        $mockService->shouldReceive('handle')
            ->andThrow(new \Exception('Test error'));

        $this->app->instance(GraphService::class, $mockService);

        $response = $this->asTenant()->postJson('/graph', [
            'metrics' => 'leads',
        ]);

        $response->assertStatus(200);
        $response->assertJsonPath('status', 'fail');
        $response->assertJsonStructure(['status', 'message']);
    }

    public function testGraphEndpointDatasetTensionIsCorrect(): void
    {
        $this->mockGraphServiceWithData();

        $response = $this->asTenant()->postJson('/graph', [
            'metrics' => 'leads',
            'date_range' => 3,
        ]);

        $response->assertStatus(200);
        $datasets = $response->json('datasets');

        if (!empty($datasets)) {
            foreach ($datasets as $dataset) {
                $this->assertEquals(0.3, $dataset['tension']);
            }
        }
    }

    public function testGraphEndpointWithDifferentDateRanges(): void
    {
        $this->mockGraphService();

        foreach ([7, 14, 30, 90] as $range) {
            $response = $this->asTenant()->postJson('/graph', [
                'metrics' => 'leads',
                'date_range' => $range,
            ]);

            $response->assertStatus(200);
            $response->assertJsonPath('date_range', $range);
        }
    }

    private function mockGraphService(): void
    {
        $mockFeatures = Mockery::mock(FeatureRepository::class);
        $mockFeatures->shouldReceive('isEnabled')->andReturn(false);

        $mockMatomo = Mockery::mock(Matomo::class);
        $mockMatomo->shouldReceive('setSiteId')->andReturnSelf();
        $mockMatomo->shouldReceive('runReport')->andReturn([]);

        $this->app->instance(FeatureRepository::class, $mockFeatures);
        $this->app->instance(Matomo::class, $mockMatomo);
    }

    private function mockGraphServiceWithData(): void
    {
        $mockService = Mockery::mock(GraphService::class);
        $mockService->shouldReceive('handle')
            ->andReturn(collect([
                '20240613' => ['date' => '20240613', 'enquiries' => 5],
                '20240614' => ['date' => '20240614', 'enquiries' => 8],
                '20240615' => ['date' => '20240615', 'enquiries' => 3],
            ]));
        $mockService->shouldReceive('getLabels')
            ->andReturn(collect(['13th Jun', '14th Jun', '15th Jun']));
        $mockService->shouldReceive('isHidden')
            ->andReturn(false);

        $this->app->instance(GraphService::class, $mockService);
    }
}
