<?php

namespace Tests\Unit;

use App\GraphService;
use App\Services\Matomo;
use Carbon\Carbon;
use Illuminate\Support\Facades\Http;
use Mockery;
use Mtc\MercuryDataModels\Repositories\FeatureRepository;
use Tests\TenantTestCase;

class GraphServiceTest extends TenantTestCase
{
    protected function setUp(): void
    {
        parent::setUp();
        Carbon::setTestNow(Carbon::parse('2024-06-15'));
    }

    protected function tearDown(): void
    {
        Carbon::setTestNow();
        parent::tearDown();
    }

    public function testHandleReturnsCollectionForLeadsMetric(): void
    {
        $features = Mockery::mock(FeatureRepository::class);
        $features->shouldReceive('isEnabled')->with('reservations')->andReturn(true);
        $features->shouldReceive('isEnabled')->with('service-booking')->andReturn(true);

        $matomo = Mockery::mock(Matomo::class);

        $service = new GraphService($features, $matomo);
        $result = $service->handle('leads', 7);

        $this->assertInstanceOf(\Illuminate\Support\Collection::class, $result);
    }

    public function testHandleReturnsCollectionForVehiclesMetric(): void
    {
        $features = Mockery::mock(FeatureRepository::class);
        $matomo = Mockery::mock(Matomo::class);

        $service = new GraphService($features, $matomo);
        $result = $service->handle('vehicles', 7);

        $this->assertInstanceOf(\Illuminate\Support\Collection::class, $result);
    }

    public function testHandleReturnsCorrectDateRange(): void
    {
        $features = Mockery::mock(FeatureRepository::class);
        $features->shouldReceive('isEnabled')->andReturn(false);

        $matomo = Mockery::mock(Matomo::class);

        $service = new GraphService($features, $matomo);
        $result = $service->handle('leads', 7);

        // Should have 8 entries (today + 7 days back)
        $this->assertCount(8, $result);
    }

    public function testHandleWithVisitorsMetricCallsMatomo(): void
    {
        $this->tenant->matomo_site_id = 123;

        $features = Mockery::mock(FeatureRepository::class);

        $matomo = Mockery::mock(Matomo::class);
        $matomo->shouldReceive('setSiteId')
            ->with(123)
            ->andReturnSelf();
        $matomo->shouldReceive('runReport')
            ->once()
            ->andReturn([
                [
                    '2024-06-15' => ['nb_visits' => 100, 'nb_actions' => 500],
                    '2024-06-14' => ['nb_visits' => 80, 'nb_actions' => 400],
                ]
            ]);

        $service = new GraphService($features, $matomo);
        $result = $service->handle('visitors', 7);

        $this->assertInstanceOf(\Illuminate\Support\Collection::class, $result);
    }

    public function testHandleWithoutMatomoSiteIdSkipsMatomo(): void
    {
        $this->tenant->matomo_site_id = null;

        $features = Mockery::mock(FeatureRepository::class);
        $matomo = Mockery::mock(Matomo::class);
        $matomo->shouldNotReceive('runReport');

        $service = new GraphService($features, $matomo);
        $result = $service->handle('visitors', 7);

        $this->assertInstanceOf(\Illuminate\Support\Collection::class, $result);
    }

    public function testHandleCatchesMatomoExceptions(): void
    {
        $this->tenant->matomo_site_id = 123;

        $features = Mockery::mock(FeatureRepository::class);

        $matomo = Mockery::mock(Matomo::class);
        $matomo->shouldReceive('setSiteId')->andReturnSelf();
        $matomo->shouldReceive('runReport')->andThrow(new \Exception('Matomo error'));

        $service = new GraphService($features, $matomo);

        // Should not throw, should return empty collection gracefully
        $result = $service->handle('visitors', 7);
        $this->assertInstanceOf(\Illuminate\Support\Collection::class, $result);
    }

    public function testGetLabelsFormatsDatesProperly(): void
    {
        $features = Mockery::mock(FeatureRepository::class);
        $features->shouldReceive('isEnabled')->andReturn(false);

        $matomo = Mockery::mock(Matomo::class);

        $service = new GraphService($features, $matomo);
        $service->handle('leads', 2);

        $labels = $service->getLabels();

        // Labels should be formatted as "jS M" (e.g., "15th Jun")
        $this->assertStringContainsString('Jun', $labels->first());
    }

    public function testIsHiddenReturnsTrueForHiddenMetrics(): void
    {
        $features = Mockery::mock(FeatureRepository::class);
        $matomo = Mockery::mock(Matomo::class);

        $service = new GraphService($features, $matomo);

        $this->assertTrue($service->isHidden('sum_visit_length'));
        $this->assertTrue($service->isHidden('nb_actions'));
        $this->assertTrue($service->isHidden('max_actions'));
    }

    public function testIsHiddenReturnsFalseForVisibleMetrics(): void
    {
        $features = Mockery::mock(FeatureRepository::class);
        $matomo = Mockery::mock(Matomo::class);

        $service = new GraphService($features, $matomo);

        $this->assertFalse($service->isHidden('nb_visits'));
        $this->assertFalse($service->isHidden('enquiries'));
        $this->assertFalse($service->isHidden('vehicles'));
    }

    public function testHandleLeadsIncludesReservationsWhenEnabled(): void
    {
        $features = Mockery::mock(FeatureRepository::class);
        $features->shouldReceive('isEnabled')
            ->with('reservations')
            ->andReturn(true);
        $features->shouldReceive('isEnabled')
            ->with('service-booking')
            ->andReturn(false);

        $matomo = Mockery::mock(Matomo::class);

        $service = new GraphService($features, $matomo);
        $result = $service->handle('leads', 7);

        $this->assertInstanceOf(\Illuminate\Support\Collection::class, $result);
    }

    public function testHandleLeadsExcludesReservationsWhenDisabled(): void
    {
        $features = Mockery::mock(FeatureRepository::class);
        $features->shouldReceive('isEnabled')
            ->with('reservations')
            ->andReturn(false);
        $features->shouldReceive('isEnabled')
            ->with('service-booking')
            ->andReturn(false);

        $matomo = Mockery::mock(Matomo::class);

        $service = new GraphService($features, $matomo);
        $result = $service->handle('leads', 7);

        $this->assertInstanceOf(\Illuminate\Support\Collection::class, $result);
    }

    public function testHandleWithUnknownMetricReturnsEmptyData(): void
    {
        $features = Mockery::mock(FeatureRepository::class);
        $matomo = Mockery::mock(Matomo::class);

        $service = new GraphService($features, $matomo);
        $result = $service->handle('unknown-metric', 7);

        // Should still have date entries but no metric data
        $this->assertInstanceOf(\Illuminate\Support\Collection::class, $result);
        $this->assertNotEmpty($result);
    }

    public function testHandleWithDefaultDateRange(): void
    {
        $features = Mockery::mock(FeatureRepository::class);
        $features->shouldReceive('isEnabled')->andReturn(false);

        $matomo = Mockery::mock(Matomo::class);

        $service = new GraphService($features, $matomo);
        $result = $service->handle('leads');

        // Default is 30 days, so 31 entries
        $this->assertCount(31, $result);
    }

    public function testDataEntriesContainDateKey(): void
    {
        $features = Mockery::mock(FeatureRepository::class);
        $features->shouldReceive('isEnabled')->andReturn(false);

        $matomo = Mockery::mock(Matomo::class);

        $service = new GraphService($features, $matomo);
        $result = $service->handle('leads', 3);

        foreach ($result as $entry) {
            $this->assertArrayHasKey('date', $entry);
        }
    }
}
