<?php

namespace Tests\Unit\Services;

use App\Services\Matomo;
use Illuminate\Support\Facades\Http;
use Tests\TestCase;

class MatomoTest extends TestCase
{
    public function testSetSiteIdReturnsInstance(): void
    {
        $matomo = new Matomo();

        $result = $matomo->setSiteId(123);

        $this->assertInstanceOf(Matomo::class, $result);
    }

    public function testSetSiteIdAcceptsNumericString(): void
    {
        $matomo = new Matomo();

        $result = $matomo->setSiteId('456');

        $this->assertInstanceOf(Matomo::class, $result);
    }

    public function testAllowedMetricsContainsVisitsSummary(): void
    {
        $this->assertContains('VisitsSummary.get', Matomo::$allowedMetrics);
    }

    public function testRunReportMakesHttpRequest(): void
    {
        config(['services.matomo.domain' => 'https://matomo.example.com']);
        config(['services.matomo.auth_token' => 'test-token']);

        Http::fake([
            'matomo.example.com*' => Http::response([
                [
                    '2024-01-01' => ['nb_visits' => 100],
                    '2024-01-02' => ['nb_visits' => 150],
                ]
            ], 200),
        ]);

        $matomo = new Matomo();
        $result = $matomo->setSiteId(123)->runReport(30, ['VisitsSummary.get']);

        $this->assertIsArray($result);
        Http::assertSent(function ($request) {
            return str_contains($request->url(), 'matomo.example.com') &&
                   $request['token_auth'] === 'test-token';
        });
    }

    public function testRunReportIncludesSiteIdInRequest(): void
    {
        config(['services.matomo.domain' => 'https://matomo.example.com']);
        config(['services.matomo.auth_token' => 'test-token']);

        Http::fake([
            'matomo.example.com*' => Http::response([[]], 200),
        ]);

        $matomo = new Matomo();
        $matomo->setSiteId(999)->runReport(7, ['VisitsSummary.get']);

        Http::assertSent(function ($request) {
            // URL-encoded idSite=999 in the urls parameter
            return str_contains(urldecode($request->url()), 'idSite=999');
        });
    }

    public function testRunReportIncludesDateRangeInRequest(): void
    {
        config(['services.matomo.domain' => 'https://matomo.example.com']);
        config(['services.matomo.auth_token' => 'test-token']);

        Http::fake([
            'matomo.example.com*' => Http::response([[]], 200),
        ]);

        $matomo = new Matomo();
        $matomo->setSiteId(123)->runReport(14, ['VisitsSummary.get']);

        Http::assertSent(function ($request) {
            // URL-encoded date=last14 in the urls parameter
            return str_contains(urldecode($request->url()), 'date=last14');
        });
    }

    public function testRunReportThrowsExceptionOnFailure(): void
    {
        config(['services.matomo.domain' => 'https://matomo.example.com']);
        config(['services.matomo.auth_token' => 'test-token']);

        Http::fake([
            'matomo.example.com*' => Http::response([], 401),
        ]);

        $this->expectException(\Exception::class);
        $this->expectExceptionMessage('Failed to authenticate Matomo');

        $matomo = new Matomo();
        $matomo->setSiteId(123)->runReport(30, ['VisitsSummary.get']);
    }

    public function testRunReportHandlesMultipleMetrics(): void
    {
        config(['services.matomo.domain' => 'https://matomo.example.com']);
        config(['services.matomo.auth_token' => 'test-token']);

        Http::fake([
            'matomo.example.com*' => Http::response([
                ['2024-01-01' => ['nb_visits' => 100]],
                ['2024-01-01' => ['nb_actions' => 500]],
            ], 200),
        ]);

        $matomo = new Matomo();
        $result = $matomo->setSiteId(123)->runReport(30, ['VisitsSummary.get', 'Actions.get']);

        $this->assertCount(2, $result);
    }

    public function testRunReportUsesCorrectApiMethod(): void
    {
        config(['services.matomo.domain' => 'https://matomo.example.com']);
        config(['services.matomo.auth_token' => 'test-token']);

        Http::fake([
            'matomo.example.com*' => Http::response([[]], 200),
        ]);

        $matomo = new Matomo();
        $matomo->setSiteId(123)->runReport(30, ['VisitsSummary.get']);

        Http::assertSent(function ($request) {
            return str_contains($request->url(), 'method=API.getBulkRequest') &&
                   str_contains($request->url(), 'format=json');
        });
    }
}
