<?php

namespace App;

use App\Metrics\EnquiryCount;
use App\Metrics\ServiceBookingCount;
use App\Metrics\VehicleCount;
use App\Metrics\VehiclesAdded;
use App\Metrics\VehiclesRemoved;
use App\Metrics\VehiclesReserved;
use App\Metrics\VehiclesSold;
use App\Services\Matomo;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Repositories\FeatureRepository;
use Mtc\VehicleReservations\Reservation;
use Spatie\Analytics\Period;

class GraphService
{
    private Collection $data;

    public function __construct(
        private readonly FeatureRepository $features,
        private readonly Matomo $matomo,
    ) {
    }

    public function handle(string $metric_name, int $dateRange = 30)
    {
        $metrics = array_filter($this->getMetrics($metric_name));
        $this->prepareDateRange(Period::days($dateRange));
        if ($this->requiresMatomo($metrics) && tenant('matomo_site_id')) {
            try {
                $this->addMatomoCollectionData($this->runAnalytics($metrics, $dateRange));
            } catch (\Exception $exception) {
                // Ignore issue with Analytics
            }
        }
        $remainingMetrics = array_diff($metrics, Matomo::$allowedMetrics);
        if (!empty($remainingMetrics)) {
            collect($remainingMetrics)
                ->each(function ($metric) use ($dateRange) {
                    $metric_data = $this->runMetric($metric, $dateRange);
                    if ($metric_data->isNotEmpty()) {
                        $this->addToData($metric_data, $metric);
                    }
                });
        }

        return $this->data;
    }

    public function getLabels()
    {
        return $this->data
            ->sortBy('date')
            ->pluck('date')
            ->map(fn($date) => Carbon::parse($date)->format('jS M'));
    }

    public function isHidden(string $dataEntry): bool
    {
        return in_array($dataEntry, [
            'sum_visit_length',
            'nb_actions',
            'max_actions',
            'enquiries-previous-year',
        ]);
    }
    private function getMatomoMetricsToUse(array $allMetrics): array
    {
        return array_values(array_intersect(Matomo::$allowedMetrics, $allMetrics));
    }

    private function requiresMatomo(array $allMetrics): bool
    {
        return !empty($this->getMatomoMetricsToUse($allMetrics));
    }

    private function runAnalytics(array $allMetrics, int $dateRange)
    {
        return $this->matomo
            ->setSiteId(tenant('matomo_site_id'))
            ->runReport($dateRange, $this->getMatomoMetricsToUse($allMetrics));
    }

    private function runMetric(string $metric, int $dateRange)
    {
        try {
            return match ($metric) {
                'enquiries' => $this->ensureFilledArray(
                    (new EnquiryCount())->get(Period::days($dateRange)),
                    $dateRange
                ),

                'enquiries-previous-year' => $this->getEnquiriesPreviousYear($dateRange),

                'service-booking' => $this->ensureFilledArray(
                    (new ServiceBookingCount())->get(Period::days($dateRange)),
                    $dateRange
                ),

                'vehicles' => $this->ensureFilledArray(
                    (new VehicleCount())->get(Period::days($dateRange)),
                    $dateRange
                ),
                'vehicles-added' => $this->ensureFilledArray(
                    (new VehiclesAdded())->get(Period::days($dateRange)),
                    $dateRange
                ),
                'vehicles-removed' => $this->ensureFilledArray(
                    (new VehiclesRemoved())->get(Period::days($dateRange)),
                    $dateRange
                ),
                'vehicles-reserved' => $this->ensureFilledArray(
                    (new VehiclesReserved())->get(Period::days($dateRange)),
                    $dateRange
                ),
                'vehicles-sold' => $this->ensureFilledArray(
                    (new VehiclesSold())->get(Period::days($dateRange)),
                    $dateRange
                ),
                'reservations' => $this->ensureFilledArray($this->getReservationsForDateRange($dateRange), $dateRange),
                default => collect([]),
            };
        } catch (\Exception $exception) {
            return collect([]);
        }
    }

    private function prepareDateRange(Period $period): void
    {
        $this->data = collect();
        $from = Carbon::parse($period->startDate);
        $to = Carbon::parse($period->endDate);
        $this->data->put(
            $from->format('Ymd'),
            [
                'date' => $from->format('Ymd')
            ],
        );
        do {
            $from->addDay();
            $this->data->put(
                $from->format('Ymd'),
                [
                    'date' => $from->format('Ymd')
                ],
            );
        } while ($from->lt($to));
    }

    private function addToData(Collection $data, string $metric = null)
    {
        $data->each(function ($value, $date) use ($metric) {
            $dateValue = Carbon::parse($date)->format('Ymd');
            $this->data->put($dateValue, array_merge(
                [$metric => $value],
                $this->data[$dateValue] ?? []
            ));
        });
    }

    private function addMatomoCollectionData(array $data)
    {
        collect($data[0] ?? [])->each(function ($value, $date) {
            $dateValue = Carbon::parse($date)->format('Ymd');
            $this->data->put($dateValue, array_merge(
                $value,
                $this->data[$dateValue] ?? []
            ));
        });
    }

    private function ensureFilledArray(Collection $data, int $dateRange): Collection
    {
        return collect(range(0, $dateRange))
            ->map(fn($daysInPast) => Carbon::today()->subDays($daysInPast))
            ->keyBy(fn(Carbon $date) => $date->format('Y-m-d'))
            ->map(fn(Carbon $date) => $data[$date->format('Y-m-d')] ?? 0);
    }

    private function getReservationsForDateRange(int $dateRange)
    {
        return Reservation::query()
            ->whereNotNull('confirmed_at')
            ->where('confirmed_at', '>', Carbon::now()->startOfDay()->subDays($dateRange))
            ->where('confirmed_at', '<=', Carbon::now()->endOfDay())
            ->get()
            ->groupBy(fn($reservation) => $reservation->confirmed_at->format('Y-m-d'))
            ->map(fn($reservations_in_date) => $reservations_in_date->count());
    }

    private function getMetrics(string $metric_name): array
    {
        return match ($metric_name) {
            'visitors' => [
                'VisitsSummary.get',
            ],
            'leads' => array_filter([
                'enquiries',
                'sessions',
                $this->checkEnabled('reservations'),
                $this->checkEnabled('service-booking'),
                $this->checkLiveForYear() ? 'enquiries-previous-year' : null,
            ]),
            'vehicles' => [
                'vehicles',
                'vehicles-added',
                'vehicles-removed',
                'vehicles-reserved',
                'vehicles-sold',
            ],
            default => [],
        };
    }

    private function checkEnabled(string $feature): ?string
    {
        return $this->features->isEnabled($feature) ? $feature : null;
    }

    private function checkLiveForYear(): bool
    {
        $liveAt = tenant('live_at');

        if (!$liveAt) {
            return false;
        }

        return Carbon::parse($liveAt)->addYear()->isPast();
    }

    private function getEnquiriesPreviousYear(int $dateRange): Collection
    {
        $startDate = Carbon::today()->subDays($dateRange)->subYear();
        $endDate = Carbon::today()->subYear();

        $period = Period::create($startDate, $endDate);

        $data = (new EnquiryCount())->get($period);

        // Map dates to current year equivalents for chart alignment
        $mapped = $data->mapWithKeys(function ($count, $date) {
            $currentYearDate = Carbon::parse($date)->addYear()->format('Y-m-d');
            return [$currentYearDate => $count];
        });

        return $this->ensureFilledArray($mapped, $dateRange);
    }
}
