<?php

namespace App\Http\Controllers;

use App\Facades\Settings;
use App\GraphService;
use App\Services\Matomo;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class StatsController extends Controller
{
    public function __construct(
        private readonly Matomo $matomo,
    ) {
    }

    public function graph(Request $request, GraphService $service): array
    {
        try {
            $result = $service->handle(
                $request->input('metrics', 'leads'),
                $request->input('date_range', 30)
            );

            $columns = $result->map(fn($entry) => array_keys($entry))
                ->flatten(1)
                ->unique()
                ->reject(fn ($value) => $value === 'date');

            return [
                'status' => 'ok',
                'labels' => $result
                    ->sortBy('date')
                    ->pluck('date')
                    ->map(fn ($date) => Carbon::parse($date)->format('jS M')),
                'date_range' => $request->input('date_range', 30),
                'type' => $request->input('metrics', 'leads'),
                'datasets' => $columns
                    ->map(fn ($column, $index) => [
                        'label' => __("labels.$column"),
                        'backgroundColor' => Settings::get($this->getColourName($index)),
                        'data' => $result->sortBy('date')->pluck($column)->map(fn($value) => $value ?? 0),
                        'tension' => 0.3,
                        'hidden' => $service->isHidden($column),
                    ])->values()->toArray(),
            ];
        } catch (\Exception $exception) {
            Log::error($exception->getMessage(), $exception->getTrace());
            return $this->graphError();
        }
    }


    private function graphError(): array
    {
        return [
            'status' => 'fail',
            'message' => 'Could not load stats for site visitors. Please ensure '
                . '<a href="/settings/site/seo">GA account is set up and access is granted</a>',
        ];
    }

    private function getColourName($index)
    {
        $colours = [
            'app-primary-colour',
            'app-secondary-colour',
            'app-tertiary-colour',
            'app-success-colour',
            'app-error-colour',
            'app-information-colour',
            'app-text-colour',
            'app-subtext-colour',
            'app-layer-dark-colour',
            'app-layer-accent-colour',
            'app-layer-light-colour',
        ];

        return $colours[$index] ?? 'app-primary-colour';
    }

    /**
     * Get visitor count for a specific date.
     */
    public function visitors(Request $request): array
    {
        $date = $request->input('date', Carbon::yesterday()->format('Y-m-d'));
        $matomoSiteId = tenant('matomo_site_id');

        if (!$matomoSiteId) {
            return [
                'status' => 'ok',
                'visitors' => 0,
                'date' => $date,
            ];
        }

        try {
            $result = $this->matomo
                ->setSiteId($matomoSiteId)
                ->getVisitorsForDate($date);

            return [
                'status' => 'ok',
                'visitors' => $result,
                'date' => $date,
            ];
        } catch (\Exception $e) {
            Log::error('Failed to fetch visitors from Matomo: ' . $e->getMessage());
            return [
                'status' => 'error',
                'visitors' => 0,
                'date' => $date,
                'message' => 'Failed to fetch visitor data',
            ];
        }
    }
}
