<?php

namespace Mtc\BigCommerceRates\Livewire\BigCommerce;

use Livewire\Component;
use Mtc\BigCommerceRates\Models\BigCommerce\Rules\ShippingRule;
use Mtc\BigCommerceRates\Models\BigCommerce\Rules\RuleCondition;
use Mtc\BigCommerceRates\Models\BigCommerce\Rules\RuleAction;
use Mtc\BigCommerceRates\Models\BigCommerce\Rates\StoreCarrier;
use Mtc\BigCommerceRates\Models\StoreInstallation;

class RuleBuilder extends Component
{
    public $storeHash;
    public $ruleId = null;
    public $isEditing = false;

    // Basic rule info
    public $name = '';
    public $description = '';
    public $enabled = true;
    public $priority = 0;
    public $match_type = 'all';
    public $condition_logic = 'and';

    // Conditions (grouped by group_index)
    public $conditions = [];
    public $nextGroupIndex = 0;

    // Actions
    public $actions = [];

    // Available field options
    public $fieldSources = [];
    public $operators = [];
    public $actionTypes = [];
    public $patternOperators = [];
    public $cartFields = [];
    public $itemFields = [];
    public $productFields = [];

    // Available rates for the store
    public $availableRates = [];

    protected $rules = [
        'name' => 'required|string|max:255',
        'priority' => 'required|integer|min:0',
        'match_type' => 'required|in:all,any',
    ];

    public function mount($storeHash = null, $rule = null)
    {
        $this->storeHash = $storeHash
            ?? (auth()->user()?->storeInstallation?->store_hash ?? null);

        // Load available options from models
        $this->fieldSources = RuleCondition::FIELD_SOURCES;
        $this->operators = RuleCondition::OPERATORS;
        $this->actionTypes = RuleAction::ACTION_TYPES;
        $this->patternOperators = RuleAction::PATTERN_OPERATORS;
        $this->cartFields = RuleCondition::CART_FIELDS;
        $this->itemFields = RuleCondition::ITEM_FIELDS;
        $this->productFields = RuleCondition::PRODUCT_FIELDS;

        // Load available rates for the store
        $this->loadAvailableRates();

        // If editing, load existing rule
        if ($rule) {
            $this->loadRule($rule);
        } else {
            // Set default priority for new rules
            $store = StoreInstallation::findByStoreHash($this->storeHash);
            if ($store) {
                $maxPriority = ShippingRule::forStore($store->id)->max('priority') ?? -1;
                $this->priority = $maxPriority + 1;
            }
            // Add first empty condition group
            $this->addConditionGroup();
        }
    }

    protected function loadRule($ruleId)
    {
        $rule = ShippingRule::with(['conditions', 'actions'])->find($ruleId);
        if (!$rule) return;

        $this->isEditing = true;
        $this->ruleId = $rule->id;
        $this->name = $rule->name;
        $this->description = $rule->description ?? '';
        $this->enabled = $rule->enabled;
        $this->priority = $rule->priority;
        $this->match_type = $rule->match_type;
        $this->condition_logic = $rule->condition_logic;

        // Load conditions grouped by group_index
        $groupedConditions = $rule->conditions->groupBy('group_index');
        $this->conditions = [];
        $this->nextGroupIndex = 0;

        foreach ($groupedConditions as $groupIndex => $groupConditions) {
            $group = [
                'id' => $this->nextGroupIndex,
                'conditions' => [],
            ];

            foreach ($groupConditions as $condition) {
                $group['conditions'][] = [
                    'id' => $condition->id,
                    'field_source' => $condition->field_source,
                    'field_name' => $condition->field_name,
                    'field_operator' => $condition->field_operator,
                    'field_value' => $condition->field_value,
                    'field_value_max' => $condition->field_value_max,
                    'case_sensitive' => $condition->case_sensitive,
                ];
            }

            $this->conditions[] = $group;
            $this->nextGroupIndex++;
        }

        // If no conditions, add empty group
        if (empty($this->conditions)) {
            $this->addConditionGroup();
        }

        // Load actions
        $this->actions = $rule->actions->map(function ($action) {
            return [
                'id' => $action->id,
                'action_type' => $action->action_type,
                'rate_pattern' => $action->rate_pattern,
                'pattern_operator' => $action->pattern_operator,
            ];
        })->toArray();
    }

    public function addConditionGroup()
    {
        $this->conditions[] = [
            'id' => $this->nextGroupIndex++,
            'conditions' => [
                $this->getEmptyCondition(),
            ],
        ];
    }

    public function addConditionToGroup($groupIndex)
    {
        if (isset($this->conditions[$groupIndex])) {
            $this->conditions[$groupIndex]['conditions'][] = $this->getEmptyCondition();
        }
    }

    public function removeConditionFromGroup($groupIndex, $conditionIndex)
    {
        if (isset($this->conditions[$groupIndex]['conditions'][$conditionIndex])) {
            unset($this->conditions[$groupIndex]['conditions'][$conditionIndex]);
            $this->conditions[$groupIndex]['conditions'] = array_values($this->conditions[$groupIndex]['conditions']);

            // If group is empty, remove it (unless it's the only one)
            if (empty($this->conditions[$groupIndex]['conditions'])) {
                if (count($this->conditions) > 1) {
                    unset($this->conditions[$groupIndex]);
                    $this->conditions = array_values($this->conditions);
                } else {
                    // Add empty condition to keep at least one
                    $this->conditions[$groupIndex]['conditions'][] = $this->getEmptyCondition();
                }
            }
        }
    }

    public function removeConditionGroup($groupIndex)
    {
        if (count($this->conditions) > 1 && isset($this->conditions[$groupIndex])) {
            unset($this->conditions[$groupIndex]);
            $this->conditions = array_values($this->conditions);
        }
    }

    protected function getEmptyCondition(): array
    {
        return [
            'id' => null,
            'field_source' => 'option',
            'field_name' => '',
            'field_operator' => 'equals',
            'field_value' => '',
            'field_value_max' => '',
            'case_sensitive' => false,
        ];
    }

    public function addAction()
    {
        $this->actions[] = [
            'id' => null,
            'action_type' => 'include',
            'rate_pattern' => '',
            'pattern_operator' => 'contains',
        ];
    }

    public function removeAction($index)
    {
        if (isset($this->actions[$index])) {
            unset($this->actions[$index]);
            $this->actions = array_values($this->actions);
        }
    }

    public function getFieldsForSource($source): array
    {
        return match ($source) {
            'cart' => $this->cartFields,
            'item' => $this->itemFields,
            'product' => $this->productFields,
            default => [],
        };
    }

    /**
     * Load available rates for the store
     */
    protected function loadAvailableRates(): void
    {
        $store = StoreInstallation::findByStoreHash($this->storeHash);
        if (!$store) {
            $this->availableRates = [];
            return;
        }

        $carriers = StoreCarrier::where('store_id', $store->id)
            ->with('rates')
            ->get();

        $rates = [];
        foreach ($carriers as $carrier) {
            foreach ($carrier->rates as $rate) {
                $rates[] = [
                    'code' => $rate->code,
                    'display_name' => $rate->display_name,
                    'carrier' => $carrier->carrier_info_display_name,
                ];
            }
        }

        $this->availableRates = $rates;
    }

    public function save()
    {
        $this->validate();

        $store = StoreInstallation::findByStoreHash($this->storeHash);
        if (!$store) {
            $this->addError('store', 'Store not found.');
            return;
        }

        // Create or update the rule
        if ($this->isEditing && $this->ruleId) {
            $rule = ShippingRule::find($this->ruleId);
            if (!$rule) {
                $this->addError('rule', 'Rule not found.');
                return;
            }
        } else {
            $rule = new ShippingRule();
            $rule->store_id = $store->id;
        }

        $rule->name = $this->name;
        $rule->description = $this->description ?: null;
        $rule->enabled = $this->enabled;
        $rule->priority = $this->priority;
        $rule->match_type = $this->match_type;
        $rule->condition_logic = $this->condition_logic;
        $rule->save();

        // Delete existing conditions and actions if editing
        if ($this->isEditing) {
            $rule->conditions()->delete();
            $rule->actions()->delete();
        }

        // Save conditions
        foreach ($this->conditions as $groupIndex => $group) {
            foreach ($group['conditions'] as $condition) {
                // Skip empty conditions
                if (empty($condition['field_name']) && empty($condition['field_value'])) {
                    continue;
                }

                RuleCondition::create([
                    'rule_id' => $rule->id,
                    'group_index' => $groupIndex,
                    'field_source' => $condition['field_source'],
                    'field_name' => $condition['field_name'],
                    'field_operator' => $condition['field_operator'],
                    'field_value' => $condition['field_value'],
                    'field_value_max' => $condition['field_value_max'] ?: null,
                    'case_sensitive' => $condition['case_sensitive'] ?? false,
                ]);
            }
        }

        // Save actions
        foreach ($this->actions as $action) {
            // Skip empty actions
            if (empty($action['rate_pattern'])) {
                continue;
            }

            RuleAction::create([
                'rule_id' => $rule->id,
                'action_type' => $action['action_type'],
                'rate_pattern' => $action['rate_pattern'],
                'pattern_operator' => $action['pattern_operator'],
            ]);
        }

        // Redirect back to rules list
        return redirect()->route('bigcommerce.rules.index', ['storeHash' => $this->storeHash])
            ->with('message', $this->isEditing ? 'Rule updated successfully' : 'Rule created successfully');
    }

    public function cancel()
    {
        return redirect()->route('bigcommerce.rules.index', ['storeHash' => $this->storeHash]);
    }

    public function render()
    {
        return view('bigcommerce::livewire.bigcommerce.rules.builder');
    }
}
