<?php

namespace Mtc\BigCommerceRates\Services;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Mtc\BigCommerceRates\Models\BigCommerce\Rules\ShippingRule;
use Mtc\BigCommerceRates\Models\BigCommerce\Rules\RuleCondition;

class RuleEvaluator
{
    protected array $cartData = [];
    protected array $productCache = [];
    protected $productFetcher = null;

    /**
     * Set cart-level data for evaluation
     */
    public function setCartData(array $cartData): self
    {
        $this->cartData = $cartData;
        return $this;
    }

    /**
     * Set the product fetcher callback for lazy loading product data
     */
    public function setProductFetcher(callable $fetcher): self
    {
        $this->productFetcher = $fetcher;
        return $this;
    }

    /**
     * Evaluate rules against cart items and return the matching rule's actions
     *
     * @param Collection $items Cart line items
     * @param Collection $rules Shipping rules ordered by priority
     * @return array|null Actions from the first matching rule, or null if no match
     */
    public function evaluate(Collection $items, Collection $rules): ?array
    {
        foreach ($rules as $rule) {
            Log::debug("Evaluating rule: {$rule->name} (priority: {$rule->priority})");

            if ($this->ruleMatches($rule, $items)) {
                Log::debug("Rule matched: {$rule->name}");
                return $this->getActionsArray($rule);
            }
        }

        Log::debug("No rules matched, returning default behavior");
        return null;
    }

    /**
     * Check if a rule matches the cart items
     */
    protected function ruleMatches(ShippingRule $rule, Collection $items): bool
    {
        $conditions = $rule->conditions;

        // If no conditions, rule always matches (useful for default/fallback rules)
        if ($conditions->isEmpty()) {
            return true;
        }

        // Group conditions by group_index for OR logic between groups
        $conditionGroups = $conditions->groupBy('group_index');

        // Evaluate each group - groups are OR'd together
        foreach ($conditionGroups as $groupIndex => $groupConditions) {
            $groupMatches = $this->evaluateConditionGroup($rule, $groupConditions, $items);

            // If any group matches, the rule matches (OR between groups)
            if ($groupMatches) {
                return true;
            }
        }

        return false;
    }

    /**
     * Evaluate a group of conditions against items
     * Conditions within a group are AND'd together
     */
    protected function evaluateConditionGroup(ShippingRule $rule, Collection $conditions, Collection $items): bool
    {
        // For each condition in the group, check if it matches according to match_type
        foreach ($conditions as $condition) {
            $conditionResult = $this->evaluateCondition($condition, $items, $rule->matchesAllItems());

            // Within a group, all conditions must match (AND logic)
            if (!$conditionResult) {
                return false;
            }
        }

        return true;
    }

    /**
     * Evaluate a single condition against items
     *
     * @param RuleCondition $condition
     * @param Collection $items
     * @param bool $matchAll Whether ALL items must match (true) or ANY item (false)
     * @return bool
     */
    protected function evaluateCondition(RuleCondition $condition, Collection $items, bool $matchAll): bool
    {
        // Cart-level conditions are evaluated once, not per-item
        if ($condition->field_source === 'cart') {
            return $this->evaluateCartCondition($condition);
        }

        // Item-level conditions are evaluated per item
        if ($matchAll) {
            // ALL items must match
            return $items->every(fn($item) => $this->evaluateItemCondition($condition, $item));
        } else {
            // ANY item must match
            return $items->contains(fn($item) => $this->evaluateItemCondition($condition, $item));
        }
    }

    /**
     * Evaluate a cart-level condition
     */
    protected function evaluateCartCondition(RuleCondition $condition): bool
    {
        $value = data_get($this->cartData, $condition->field_name);

        Log::debug("Evaluating cart condition: {$condition->field_name} = {$value}");

        return $condition->evaluate($value);
    }

    /**
     * Evaluate an item-level condition
     */
    protected function evaluateItemCondition(RuleCondition $condition, array $item): bool
    {
        $value = match ($condition->field_source) {
            'item' => $this->getItemFieldValue($condition, $item),
            'product' => $this->getProductFieldValue($condition, $item),
            'option' => $this->getOptionValue($condition, $item),
            'custom_field' => $this->getCustomFieldValue($condition, $item),
            default => null,
        };

        Log::debug("Evaluating {$condition->field_source} condition: {$condition->field_name} = {$value}");

        return $condition->evaluate($value);
    }

    /**
     * Get a field value from the line item
     */
    protected function getItemFieldValue(RuleCondition $condition, array $item): mixed
    {
        return data_get($item, $condition->field_name);
    }

    /**
     * Get a field value from the product (requires API call)
     */
    protected function getProductFieldValue(RuleCondition $condition, array $item): mixed
    {
        $productId = $item['product_id'] ?? null;
        if (!$productId) {
            return null;
        }

        $product = $this->getProduct($productId);
        if (!$product) {
            return null;
        }

        return data_get($product, 'data.' . $condition->field_name);
    }

    /**
     * Get an option value from the line item
     */
    protected function getOptionValue(RuleCondition $condition, array $item): mixed
    {
        $options = $item['options'] ?? [];

        foreach ($options as $option) {
            $optionName = strtolower($option['name'] ?? '');
            $conditionName = strtolower($condition->field_name);

            if ($optionName === $conditionName) {
                return $option['value'] ?? null;
            }
        }

        return null;
    }

    /**
     * Get a custom field value from the product
     */
    protected function getCustomFieldValue(RuleCondition $condition, array $item): mixed
    {
        $productId = $item['product_id'] ?? null;
        if (!$productId) {
            return null;
        }

        $product = $this->getProduct($productId);
        if (!$product) {
            return null;
        }

        $customFields = data_get($product, 'data.custom_fields', []);

        foreach ($customFields as $field) {
            $fieldName = strtolower($field['name'] ?? '');
            $conditionName = strtolower($condition->field_name);

            // Support both exact match and contains for field name
            if ($fieldName === $conditionName || str_contains($fieldName, $conditionName)) {
                return $field['value'] ?? null;
            }
        }

        return null;
    }

    /**
     * Get product data, using cache to avoid repeated API calls
     */
    protected function getProduct($productId): ?array
    {
        if (isset($this->productCache[$productId])) {
            return $this->productCache[$productId];
        }

        if ($this->productFetcher) {
            $product = ($this->productFetcher)($productId);
            $this->productCache[$productId] = $product;
            return $product;
        }

        return null;
    }

    /**
     * Convert rule actions to an array format for rate filtering
     */
    protected function getActionsArray(ShippingRule $rule): array
    {
        $includes = [];
        $excludes = [];

        foreach ($rule->actions as $action) {
            $actionData = [
                'pattern' => $action->rate_pattern,
                'operator' => $action->pattern_operator,
            ];

            if ($action->isInclude()) {
                $includes[] = $actionData;
            } else {
                $excludes[] = $actionData;
            }
        }

        return [
            'rule_id' => $rule->id,
            'rule_name' => $rule->name,
            'includes' => $includes,
            'excludes' => $excludes,
        ];
    }

    /**
     * Apply actions to filter rates
     *
     * @param Collection $rates Collection of rate objects with 'code' property
     * @param array $actions Actions from evaluate()
     * @return Collection Filtered rates
     */
    public function applyActions(Collection $rates, ?array $actions): Collection
    {
        // If no actions (no rule matched), apply default behavior
        if ($actions === null) {
            return $this->applyDefaultFilter($rates);
        }

        $includes = $actions['includes'] ?? [];
        $excludes = $actions['excludes'] ?? [];

        return $rates->filter(function ($rate) use ($includes, $excludes) {
            $code = strtolower($rate->code ?? '');

            // If there are include patterns, rate must match at least one
            if (!empty($includes)) {
                $matchesInclude = false;
                foreach ($includes as $include) {
                    if ($this->patternMatches($code, $include['pattern'], $include['operator'])) {
                        $matchesInclude = true;
                        break;
                    }
                }
                if (!$matchesInclude) {
                    return false;
                }
            }

            // Rate must not match any exclude patterns
            foreach ($excludes as $exclude) {
                if ($this->patternMatches($code, $exclude['pattern'], $exclude['operator'])) {
                    return false;
                }
            }

            return true;
        });
    }

    /**
     * Apply default filter (exclude free, sample, accessory rates)
     */
    protected function applyDefaultFilter(Collection $rates): Collection
    {
        return $rates->filter(function ($rate) {
            $code = strtolower($rate->code ?? '');

            return !str_contains($code, 'free')
                && !str_contains($code, 'sample')
                && !str_contains($code, 'accessory');
        });
    }

    /**
     * Check if a rate code matches a pattern
     */
    protected function patternMatches(string $code, string $pattern, string $operator): bool
    {
        $pattern = strtolower($pattern);

        return match ($operator) {
            'contains' => str_contains($code, $pattern),
            'equals' => $code === $pattern,
            'regex' => (bool) preg_match('/' . $pattern . '/i', $code),
            default => false,
        };
    }
}
