<?php

namespace Mtc\BigCommerceRates\Models\BigCommerce\Rules;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class RuleAction extends Model
{
    protected $table = 'shipping_rule_actions';

    protected $fillable = [
        'rule_id',
        'action_type',
        'rate_pattern',
        'pattern_operator',
    ];

    /**
     * Available action types
     */
    public const ACTION_TYPES = [
        'include' => 'Include',
        'exclude' => 'Exclude',
    ];

    /**
     * Available pattern operators
     */
    public const PATTERN_OPERATORS = [
        'contains' => 'Contains',
        'equals' => 'Equals',
        'regex' => 'Regex',
    ];

    /**
     * Get the shipping rule this action belongs to
     */
    public function rule(): BelongsTo
    {
        return $this->belongsTo(ShippingRule::class, 'rule_id');
    }

    /**
     * Check if a rate code matches this action's pattern
     */
    public function matchesRate(string $rateCode): bool
    {
        $pattern = strtolower($this->rate_pattern);
        $code = strtolower($rateCode);

        return match ($this->pattern_operator) {
            'contains' => str_contains($code, $pattern),
            'equals' => $code === $pattern,
            'regex' => (bool) preg_match('/' . $this->rate_pattern . '/i', $rateCode),
            default => false,
        };
    }

    /**
     * Check if this is an include action
     */
    public function isInclude(): bool
    {
        return $this->action_type === 'include';
    }

    /**
     * Check if this is an exclude action
     */
    public function isExclude(): bool
    {
        return $this->action_type === 'exclude';
    }

    /**
     * Get human-readable description of this action
     */
    public function getDescription(): string
    {
        $type = self::ACTION_TYPES[$this->action_type] ?? $this->action_type;
        $operator = self::PATTERN_OPERATORS[$this->pattern_operator] ?? $this->pattern_operator;

        return "{$type} rates where code {$operator} \"{$this->rate_pattern}\"";
    }
}
