<?php

namespace Mtc\BigCommerceRates\Console\Commands;

use Illuminate\Console\Command;
use Mtc\BigCommerceRates\Models\StoreInstallation;
use Illuminate\Http\Client\HttpClientException;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Mtc\BigCommerceRates\Models\BigCommerce\Product;

class Webhooks extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'bc:webhook';

    protected $mainEndpointUrl = null;
    protected $storeHashes = [];

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'get / update / create / delete webhooks for a specific store';


    public function __construct()
    {
        parent::__construct();

        // Fetch store_hashes
        $this->storeHashes = StoreInstallation::pluck('store_hash')->toArray();
        if (empty($this->storeHashes)) {
            $this->error('No store installations found.');
            return Command::FAILURE;
        }
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {

        // Prompt selection
        $selectedStore = $this->choice(
            'Select the store_hash to manage webhooks:',
            $this->storeHashes
        );

        $store = StoreInstallation::where('store_hash', $selectedStore)->first();
        if (!$store) {
            $this->error("Store installation not found for: {$selectedStore}");
            return Command::FAILURE;
        }


        $this->info("You selected: {$selectedStore}");

        $this->manageWebhook($selectedStore);

        return Command::SUCCESS;
    }

    protected function manageWebhook($selectedStore)
    {
        $this->mainEndpointUrl = env('APP_URL') . "bc-api/{$selectedStore}/v3/hooks";

        $options = ['create webhooks', 'get webhooks'];

        $selectedOption = $this->choice(
            'Select an option to manage webhooks:',
            $options
        );

        match($selectedOption) {
            'create webhooks' => $this->createWebhook(),
            'get webhooks' => $this->getWebhooks(),
        };
    }

    protected function createWebhook()
    {
        $this->info("Creating Product webhook for: {$this->mainEndpointUrl}");

        $scopes=[
            "store/product/deleted"=>"deleted",
            "store/product/created"=>"created",
            "store/product/updated"=>"updated",
        ];

        try {
            $existing = Http::timeout(400)
                ->connectTimeout(1)
                ->get($this->mainEndpointUrl)
                ->json();

            $existingScopes = collect(data_get($existing, 'data', []))
                ->pluck('scope')
                ->unique()
                ->all();
        } catch (\Exception $e) {
            $this->error("Failed to fetch existing webhooks: " . $e->getMessage());
            return Command::FAILURE;
        }

        foreach($scopes as $scope=>$action) {


            if (in_array($scope, $existingScopes)) {
                $this->warn("Webhook for scope [{$scope}] already exists. Skipping...");
                continue;
            }


            $body = [
                "scope" => $scope,
                "destination" =>  env('APP_URL')."webhook/product/{$action}",
                "is_active" => true,
            ];
            try{
                $response = Http::timeout(400)
                    ->connectTimeout(1)
                    ->post( $this->mainEndpointUrl, $body);


                if ($response->successful()) {
                    $this->info("Webhook for [{$scope}] created.");
                } else {
                    $this->error("Failed to create webhook for [{$scope}]: " . $response->body());
                }


            }catch (HttpClientException $clientException){
                $this->error("HTTP Client Exception: " . $clientException->getMessage());
                return Command::FAILURE;
            }
        }
        return Command::SUCCESS;
    }

    protected function getWebhooks()
    {

        $this->info("Getting webhook for: {$this->mainEndpointUrl}");

        try{
            $response = Http::timeout(400)
                ->connectTimeout(1)
                ->get( $this->mainEndpointUrl)
                ->json();



        }catch (HttpClientException $clientException){
            $this->error("HTTP Client Exception: " . $clientException->getMessage());
            return Command::FAILURE;
        }

        $webhooks = data_get($response, 'data', []);


        if (empty($webhooks)) {
            $this->warn("No webhooks found.");
            return Command::SUCCESS;
        }

        $choices = collect($webhooks)->mapWithKeys(function ($webhook) {
            $label = "{$webhook['id']} - {$webhook['scope']}";
            return [$label => $webhook['id']];
        })->toArray();

        $selectedLabel = $this->choice(
            'Select the webhook to manage:',
            array_keys($choices)
        );

        $selectedId = $choices[$selectedLabel];
        $selected = collect($webhooks)->firstWhere('id', (int) $selectedId);
        if (!$selected) {
            $this->error("Webhook not found.");
            return Command::FAILURE;
        }

        // Step 2: Choose action
        $action = $this->choice('What do you want to do?', ['Update', 'Delete']);

        if ($action === 'Delete') {
            $confirmed = $this->confirm("Are you sure you want to delete webhook ID {$selectedId}?");
            if ($confirmed) {
                $deleteEndpoint = "{$this->mainEndpointUrl}/{$selectedId}";
                $response = Http::delete($deleteEndpoint);

                if ($response->successful()) {
                    $this->info("Webhook deleted.");
                } else {
                    $this->error("Failed to delete webhook.");
                }
            }
            return Command::SUCCESS;
        }
        // Step 3: Update options
        $this->line("Updating webhook ID {$selectedId}");
        $newIsActive = $this->confirm("Toggle 'is_active'? (currently " . ($selected['is_active'] ? 'true' : 'false') . ")");
        $newDestination = $this->ask("Enter new destination (leave blank to keep current):", $selected['destination']);
        $toggleEventsHistory = $this->confirm("Toggle 'events_history_enabled'? (currently " . ($selected['events_history_enabled'] ? 'true' : 'false') . ")");

        // Prepare updated data
        $updatedData = [
            'is_active' => $newIsActive ? !$selected['is_active'] : $selected['is_active'],
            'destination' => $newDestination,
            'events_history_enabled' => $toggleEventsHistory ? !$selected['events_history_enabled'] : $selected['events_history_enabled'],
        ];

        $updateEndpoint = "{$this->mainEndpointUrl}/{$selectedId}";
        $updateResponse = Http::put($updateEndpoint, $updatedData);

        if ($updateResponse->successful()) {
            $this->info("Webhook updated successfully.");
        } else {
            $this->error("Failed to update webhook.");
        }

        return Command::SUCCESS;

    }

    protected function updateWebhook(string $webhookId=null)
    {
        $this->info("Updating webhook for: {$this->mainEndpointUrl}");
    }

}
