<?php

namespace Mtc\BigCommerceRates\Http\Controllers\BigCommerce\Shipping;

use App\Http\Controllers\Controller;
use Illuminate\Http\Client\HttpClientException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\BigCommerceRates\Models\BigCommerce\Rates\StoreCarrier;
use Mtc\BigCommerceRates\Models\BigCommerce\Rules\ShippingRule;
use Mtc\BigCommerceRates\Models\StoreInstallation;
use Mtc\BigCommerceRates\Services\RuleEvaluator;
use Illuminate\Support\Collection;


class CarrierController extends Controller
{

    public function __construct(
        public Request $request
    )
    {
    }


    public function rates()
    {
        Log::debug("");
        Log::debug("Request:",$this->request->toArray());
        Log::debug("");



        /**
         * determine country and return appropriate rates
         */
        $storeHash = $this->request->input('base_options.store_id');
        $countryCode = $this->request->input('base_options.destination.country_iso2');

        $cartId = collect($this->request->input('base_options.request_context.reference_values'))->firstWhere('name', 'cart_id')['value'] ?? null;;

        $checkoutEndpoint =  env('APP_URL')."bc-api/{$storeHash}/v3/checkouts/$cartId";
        $data = [
            'include' => 'cart.line_items.physical_items.options,cart.line_items.digital_items.options,consignments.available_shipping_options'
        ];


        try {
            $response = Http::timeout(400)
                ->connectTimeout(1)
                ->get($checkoutEndpoint,$data)->json();

            $lineItems = data_get($response, 'data.cart.line_items', []);


            $allItems = collect($lineItems)->flatMap(function ($items) {
                return is_array($items) ? $items : [];
            })->values();


        } catch (\Illuminate\Http\Client\HttpClientException $clientException) {
            Log::error("Error: ", $clientException->getMessage());
        }

        $rates = $this->storeRates(
                    storeHash: $storeHash,
                    countryCode: $countryCode,
                    items: $allItems
                );

        Log::debug("");
        Log::debug("");
        Log::debug("Response:",$rates);
        Log::debug("");
        Log::debug("");

        /**
         * return rates
         */
        return response()->json($rates);
    }


    public function storeRates(string $storeHash, string $countryCode, Collection $items)
    {
        $store = StoreInstallation::where('store_hash', $storeHash)->first();
        $storeId = $store->id;

        // Prepare cart data for rule evaluation
        $cartData = [
            'destination' => $this->request->input('base_options.destination', []),
            'cart_total' => $items->sum('extended_sale_price'),
            'cart_item_count' => $items->count(),
            'cart_quantity_total' => $items->sum('quantity'),
        ];

        // Initialize rule evaluator with product fetcher for lazy loading
        $ruleEvaluator = (new RuleEvaluator())
            ->setCartData($cartData)
            ->setProductFetcher(fn($productId) => $this->getProduct($productId));

        // Get enabled rules for this store ordered by priority
        $rules = ShippingRule::forStore($storeId)
            ->enabled()
            ->byPriority()
            ->with(['conditions', 'actions'])
            ->get();

        Log::debug("Found {$rules->count()} enabled shipping rules for store");

        // Evaluate rules against cart items
        $actions = $ruleEvaluator->evaluate($items, $rules);

        if ($actions) {
            Log::debug("Matched rule: {$actions['rule_name']} (ID: {$actions['rule_id']})");
        } else {
            Log::debug("No rules matched, using default filtering");
        }

        // Get carriers with rates
        $carriers = StoreCarrier::with(['rates.rateCost'])
            ->where('store_id', $storeId)
            ->get();

        $quotes = [
            'quote_id' => uuid_create(UUID_TYPE_TIME),
            'messages' => [],
            'carrier_quotes' => $carriers
                ->map(function ($carrier) use ($ruleEvaluator, $actions) {
                    // Apply rule actions to filter rates
                    $filteredRates = $ruleEvaluator->applyActions($carrier->rates, $actions);

                    if ($filteredRates->isEmpty()) return null;

                    return [
                        'carrier_info' => [
                            'code' => $carrier->carrier_info_code,
                            'display_name' => $carrier->carrier_info_display_name,
                        ],
                        'quotes' => $filteredRates->map(function ($rate) use ($carrier) {
                            return [
                                'code' => $rate->code,
                                'rate_id' => $rate->rate_id,
                                'display_name' => $rate->display_name,
                                'description' => $carrier->carrier_info_display_name,
                                'cost' => [
                                    'currency' => optional($rate->rateCost)->currency,
                                    'amount' => round(optional($rate->rateCost)->amount, 3),
                                ],
                            ];
                        })->values()->toArray(),
                    ];
                })
                ->filter()
                ->values()
                ->toArray(),
        ];

        return $quotes;
    }






    protected function emptyRate()
    {

        return [
                "quote_id"=>uuid_create(UUID_TYPE_TIME),
                "messages"=>[],
                "ttl"=>3600,
            "carrier_quotes"=>[]
        ];

    }

    protected function getProduct($productId)
    {
        $storeHash= $this->request->input('base_options.store_id');
        $productEndpoint =  env('APP_URL')."bc-api/{$storeHash}/v3/catalog/products/{$productId}";
        $queryParams = [
            "include" => "custom_fields,options,modifiers,variants",
            "include_fields" => "id,name,sku,price,categories,date_created,is_visible,inventory_level,total_sold,brand_id",
        ];

        try{

            $response = Http::timeout(400)
                ->connectTimeout(1)
                ->get($productEndpoint, $queryParams)
                ->throw()->collect();
            return $response;

        }catch (HttpClientException $e){
            Log::error("Error fetching product data: " . $e->getMessage());
            return false;
        }
    }


    public function validateConnection(Request $request)
    {
        return response()->json([
            'valid' => true,
            'messages' => [],
        ]);
    }
}
