<?php

namespace Mtc\BigCommerceRates\Models\BigCommerce\Rules;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class RuleCondition extends Model
{
    protected $table = 'shipping_rule_conditions';

    protected $fillable = [
        'rule_id',
        'group_index',
        'field_source',
        'field_name',
        'field_operator',
        'field_value',
        'field_value_max',
        'case_sensitive',
    ];

    protected $casts = [
        'group_index' => 'integer',
        'case_sensitive' => 'boolean',
    ];

    /**
     * Available field sources
     */
    public const FIELD_SOURCES = [
        'cart' => 'Cart',
        'item' => 'Item',
        'product' => 'Product',
        'option' => 'Option',
        'custom_field' => 'Custom Field',
    ];

    /**
     * Available operators
     */
    public const OPERATORS = [
        'equals' => 'Equals',
        'not_equals' => 'Not Equals',
        'contains' => 'Contains',
        'not_contains' => 'Does Not Contain',
        'greater_than' => 'Greater Than',
        'less_than' => 'Less Than',
        'between' => 'Between',
        'in_list' => 'In List',
        'regex' => 'Regex',
    ];

    /**
     * Cart-level fields available for conditions
     */
    public const CART_FIELDS = [
        'destination.country_iso2' => 'Country',
        'destination.state_or_province' => 'State/Province',
        'destination.city' => 'City',
        'destination.postal_code' => 'Postal Code',
        'cart_total' => 'Cart Total',
        'cart_item_count' => 'Item Count',
        'cart_quantity_total' => 'Total Quantity',
    ];

    /**
     * Item-level fields available for conditions
     */
    public const ITEM_FIELDS = [
        'product_id' => 'Product ID',
        'variant_id' => 'Variant ID',
        'sku' => 'SKU',
        'name' => 'Product Name',
        'quantity' => 'Quantity',
        'list_price' => 'List Price',
        'sale_price' => 'Sale Price',
        'extended_sale_price' => 'Extended Price',
        'discount_amount' => 'Discount Amount',
        'is_taxable' => 'Is Taxable',
    ];

    /**
     * Product-level fields available for conditions
     */
    public const PRODUCT_FIELDS = [
        'categories' => 'Categories',
        'brand_id' => 'Brand ID',
        'inventory_level' => 'Inventory Level',
        'total_sold' => 'Total Sold',
        'is_visible' => 'Is Visible',
        'weight' => 'Weight',
    ];

    /**
     * Get the shipping rule this condition belongs to
     */
    public function rule(): BelongsTo
    {
        return $this->belongsTo(ShippingRule::class, 'rule_id');
    }

    /**
     * Evaluate this condition against a value
     */
    public function evaluate($value): bool
    {
        $fieldValue = $this->case_sensitive ? $this->field_value : strtolower($this->field_value);
        $compareValue = $this->case_sensitive ? $value : strtolower((string) $value);

        return match ($this->field_operator) {
            'equals' => $compareValue === $fieldValue,
            'not_equals' => $compareValue !== $fieldValue,
            'contains' => str_contains($compareValue, $fieldValue),
            'not_contains' => !str_contains($compareValue, $fieldValue),
            'greater_than' => is_numeric($value) && (float) $value > (float) $this->field_value,
            'less_than' => is_numeric($value) && (float) $value < (float) $this->field_value,
            'between' => is_numeric($value) &&
                (float) $value >= (float) $this->field_value &&
                (float) $value <= (float) $this->field_value_max,
            'in_list' => in_array($compareValue, array_map(
                fn($v) => $this->case_sensitive ? trim($v) : strtolower(trim($v)),
                explode(',', $this->field_value)
            )),
            'regex' => (bool) preg_match('/' . $this->field_value . '/', (string) $value),
            default => false,
        };
    }

    /**
     * Get human-readable description of this condition
     */
    public function getDescription(): string
    {
        $source = self::FIELD_SOURCES[$this->field_source] ?? $this->field_source;
        $operator = self::OPERATORS[$this->field_operator] ?? $this->field_operator;

        $desc = "{$source}: \"{$this->field_name}\" {$operator} \"{$this->field_value}\"";

        if ($this->field_operator === 'between' && $this->field_value_max) {
            $desc .= " and \"{$this->field_value_max}\"";
        }

        return $desc;
    }
}
