<?php

namespace Mtc\BigCommerceRates\Models\BigCommerce\Rules;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Mtc\BigCommerceRates\Models\StoreInstallation;

class ShippingRule extends Model
{
    protected $table = 'shipping_rules';

    protected $fillable = [
        'store_id',
        'name',
        'description',
        'enabled',
        'priority',
        'match_type',
        'condition_logic',
    ];

    protected $casts = [
        'enabled' => 'boolean',
        'priority' => 'integer',
    ];

    /**
     * Get the store installation this rule belongs to
     */
    public function storeInstallation(): BelongsTo
    {
        return $this->belongsTo(StoreInstallation::class, 'store_id');
    }

    /**
     * Get the conditions for this rule
     */
    public function conditions(): HasMany
    {
        return $this->hasMany(RuleCondition::class, 'rule_id')->orderBy('group_index');
    }

    /**
     * Get the actions for this rule
     */
    public function actions(): HasMany
    {
        return $this->hasMany(RuleAction::class, 'rule_id');
    }

    /**
     * Scope to get enabled rules ordered by priority
     */
    public function scopeEnabled($query)
    {
        return $query->where('enabled', true);
    }

    /**
     * Scope to get rules for a specific store
     */
    public function scopeForStore($query, $storeId)
    {
        return $query->where('store_id', $storeId);
    }

    /**
     * Scope to order by priority (lower first)
     */
    public function scopeByPriority($query)
    {
        return $query->orderBy('priority', 'asc');
    }

    /**
     * Get conditions grouped by group_index for OR logic
     */
    public function getConditionGroups(): array
    {
        return $this->conditions->groupBy('group_index')->toArray();
    }

    /**
     * Check if this rule matches all items (vs any item)
     */
    public function matchesAllItems(): bool
    {
        return $this->match_type === 'all';
    }

    /**
     * Check if conditions use AND logic (vs OR)
     */
    public function usesAndLogic(): bool
    {
        return $this->condition_logic === 'and';
    }
}
