<!DOCTYPE html>
<html lang="en" class="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Shipping Rules - User Manual</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: {
                extend: {
                    colors: {
                        zinc: {
                            850: '#1f1f23',
                        }
                    }
                }
            }
        }
    </script>
    <style>
        html { scroll-behavior: smooth; }
        .prose h2 { scroll-margin-top: 100px; }
        .prose h3 { scroll-margin-top: 100px; }
    </style>
</head>
<body class="bg-zinc-900 text-zinc-100 min-h-screen">
    <!-- Header -->
    <header class="sticky top-0 z-50 bg-zinc-900/95 backdrop-blur border-b border-zinc-800">
        <div class="max-w-6xl mx-auto px-6 py-4">
            <div class="flex items-center justify-between">
                <div class="flex items-center gap-3">
                    <div class="w-10 h-10 bg-blue-600 rounded-lg flex items-center justify-center">
                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-6 h-6 text-white">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M8.25 18.75a1.5 1.5 0 01-3 0m3 0a1.5 1.5 0 00-3 0m3 0h6m-9 0H3.375a1.125 1.125 0 01-1.125-1.125V14.25m17.25 4.5a1.5 1.5 0 01-3 0m3 0a1.5 1.5 0 00-3 0m3 0h1.125c.621 0 1.129-.504 1.09-1.124a17.902 17.902 0 00-3.213-9.193 2.056 2.056 0 00-1.58-.86H14.25M16.5 18.75h-2.25m0-11.177v-.958c0-.568-.422-1.048-.987-1.106a48.554 48.554 0 00-10.026 0 1.106 1.106 0 00-.987 1.106v7.635m12-6.677v6.677m0 4.5v-4.5m0 0h-12" />
                        </svg>
                    </div>
                    <div>
                        <h1 class="text-xl font-bold text-white">Shipping Rules</h1>
                        <p class="text-xs text-zinc-400">User Manual</p>
                    </div>
                </div>
                <nav class="hidden md:flex items-center gap-6 text-sm">
                    <a href="#overview" class="text-zinc-400 hover:text-white transition-colors">Overview</a>
                    <a href="#getting-started" class="text-zinc-400 hover:text-white transition-colors">Getting Started</a>
                    <a href="#conditions" class="text-zinc-400 hover:text-white transition-colors">Conditions</a>
                    <a href="#actions" class="text-zinc-400 hover:text-white transition-colors">Actions</a>
                    <a href="#examples" class="text-zinc-400 hover:text-white transition-colors">Examples</a>
                </nav>
            </div>
        </div>
    </header>

    <div class="max-w-6xl mx-auto px-6 py-12">
        <div class="grid grid-cols-1 lg:grid-cols-4 gap-12">
            <!-- Sidebar Navigation -->
            <aside class="hidden lg:block">
                <nav class="sticky top-24 space-y-1">
                    <p class="text-xs font-semibold text-zinc-500 uppercase tracking-wider mb-3">Contents</p>
                    <a href="#overview" class="block px-3 py-2 text-sm text-zinc-400 hover:text-white hover:bg-zinc-800 rounded-lg transition-colors">Overview</a>
                    <a href="#getting-started" class="block px-3 py-2 text-sm text-zinc-400 hover:text-white hover:bg-zinc-800 rounded-lg transition-colors">Getting Started</a>
                    <a href="#rules-list" class="block px-3 py-2 text-sm text-zinc-400 hover:text-white hover:bg-zinc-800 rounded-lg transition-colors pl-6">Rules List</a>
                    <a href="#creating-rule" class="block px-3 py-2 text-sm text-zinc-400 hover:text-white hover:bg-zinc-800 rounded-lg transition-colors pl-6">Creating a Rule</a>
                    <a href="#conditions" class="block px-3 py-2 text-sm text-zinc-400 hover:text-white hover:bg-zinc-800 rounded-lg transition-colors">Conditions</a>
                    <a href="#field-sources" class="block px-3 py-2 text-sm text-zinc-400 hover:text-white hover:bg-zinc-800 rounded-lg transition-colors pl-6">Field Sources</a>
                    <a href="#operators" class="block px-3 py-2 text-sm text-zinc-400 hover:text-white hover:bg-zinc-800 rounded-lg transition-colors pl-6">Operators</a>
                    <a href="#and-or-logic" class="block px-3 py-2 text-sm text-zinc-400 hover:text-white hover:bg-zinc-800 rounded-lg transition-colors pl-6">AND/OR Logic</a>
                    <a href="#actions" class="block px-3 py-2 text-sm text-zinc-400 hover:text-white hover:bg-zinc-800 rounded-lg transition-colors">Actions</a>
                    <a href="#examples" class="block px-3 py-2 text-sm text-zinc-400 hover:text-white hover:bg-zinc-800 rounded-lg transition-colors">Examples</a>
                    <a href="#troubleshooting" class="block px-3 py-2 text-sm text-zinc-400 hover:text-white hover:bg-zinc-800 rounded-lg transition-colors">Troubleshooting</a>
                </nav>
            </aside>

            <!-- Main Content -->
            <main class="lg:col-span-3 prose prose-invert max-w-none">
                <!-- Overview Section -->
                <section id="overview" class="mb-16">
                    <h2 class="text-3xl font-bold text-white mb-4 flex items-center gap-3">
                        <span class="w-10 h-10 bg-blue-600/20 rounded-lg flex items-center justify-center">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-5 h-5 text-blue-400">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M11.25 11.25l.041-.02a.75.75 0 011.063.852l-.708 2.836a.75.75 0 001.063.853l.041-.021M21 12a9 9 0 11-18 0 9 9 0 0118 0zm-9-3.75h.008v.008H12V8.25z" />
                            </svg>
                        </span>
                        Overview
                    </h2>
                    <p class="text-zinc-300 text-lg leading-relaxed mb-6">
                        The Shipping Rules system allows you to dynamically control which shipping rates are shown to customers at checkout based on what's in their cart. Instead of hardcoding shipping logic, you can create flexible rules that automatically include or exclude specific shipping options.
                    </p>

                    <div class="bg-zinc-800/50 border border-zinc-700 rounded-xl p-6 mb-6">
                        <h3 class="text-lg font-semibold text-white mb-4">Key Features</h3>
                        <ul class="space-y-3">
                            <li class="flex items-start gap-3">
                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-5 h-5 text-green-400 mt-0.5 flex-shrink-0">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 12.75L11.25 15 15 9.75M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                </svg>
                                <span class="text-zinc-300"><strong class="text-white">Dynamic Filtering</strong> - Show or hide shipping rates based on cart contents</span>
                            </li>
                            <li class="flex items-start gap-3">
                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-5 h-5 text-green-400 mt-0.5 flex-shrink-0">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 12.75L11.25 15 15 9.75M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                </svg>
                                <span class="text-zinc-300"><strong class="text-white">Complex Logic</strong> - Combine multiple conditions with AND/OR operators</span>
                            </li>
                            <li class="flex items-start gap-3">
                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-5 h-5 text-green-400 mt-0.5 flex-shrink-0">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 12.75L11.25 15 15 9.75M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                </svg>
                                <span class="text-zinc-300"><strong class="text-white">Priority System</strong> - Control which rule takes precedence when multiple rules match</span>
                            </li>
                            <li class="flex items-start gap-3">
                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-5 h-5 text-green-400 mt-0.5 flex-shrink-0">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 12.75L11.25 15 15 9.75M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                </svg>
                                <span class="text-zinc-300"><strong class="text-white">Multiple Data Sources</strong> - Check cart data, product options, custom fields, and more</span>
                            </li>
                        </ul>
                    </div>

                    <div class="bg-amber-900/20 border border-amber-700/50 rounded-xl p-6">
                        <div class="flex items-start gap-3">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-6 h-6 text-amber-400 flex-shrink-0">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M12 9v3.75m-9.303 3.376c-.866 1.5.217 3.374 1.948 3.374h14.71c1.73 0 2.813-1.874 1.948-3.374L13.949 3.378c-.866-1.5-3.032-1.5-3.898 0L2.697 16.126zM12 15.75h.007v.008H12v-.008z" />
                            </svg>
                            <div>
                                <h4 class="font-semibold text-amber-300 mb-1">Important</h4>
                                <p class="text-amber-200/80 text-sm">Rules are evaluated in priority order (lowest number first). The <strong>first matching rule</strong> determines which shipping rates are shown. If no rules match, all rates are displayed.</p>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- Getting Started Section -->
                <section id="getting-started" class="mb-16">
                    <h2 class="text-3xl font-bold text-white mb-4 flex items-center gap-3">
                        <span class="w-10 h-10 bg-green-600/20 rounded-lg flex items-center justify-center">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-5 h-5 text-green-400">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M5.25 5.653c0-.856.917-1.398 1.667-.986l11.54 6.348a1.125 1.125 0 010 1.971l-11.54 6.347a1.125 1.125 0 01-1.667-.985V5.653z" />
                            </svg>
                        </span>
                        Getting Started
                    </h2>

                    <h3 id="rules-list" class="text-xl font-semibold text-white mt-8 mb-4">Rules List</h3>
                    <p class="text-zinc-300 mb-4">
                        Access the Rules page from the sidebar navigation by clicking on <strong class="text-white">"Rules"</strong> under the BigCommerce section.
                    </p>

                    <div class="bg-zinc-800 border border-zinc-700 rounded-xl overflow-hidden mb-6">
                        <div class="bg-zinc-850 px-4 py-3 border-b border-zinc-700">
                            <span class="text-sm font-medium text-zinc-400">Rules List Page</span>
                        </div>
                        <div class="p-6">
                            <p class="text-zinc-300 mb-4">The rules list shows all your shipping rules with:</p>
                            <ul class="space-y-2 text-zinc-300">
                                <li class="flex items-center gap-2">
                                    <span class="w-2 h-2 bg-blue-400 rounded-full"></span>
                                    <strong class="text-white">Name</strong> - The rule's display name
                                </li>
                                <li class="flex items-center gap-2">
                                    <span class="w-2 h-2 bg-blue-400 rounded-full"></span>
                                    <strong class="text-white">Priority</strong> - Lower numbers are evaluated first
                                </li>
                                <li class="flex items-center gap-2">
                                    <span class="w-2 h-2 bg-blue-400 rounded-full"></span>
                                    <strong class="text-white">Status</strong> - Enabled or disabled
                                </li>
                                <li class="flex items-center gap-2">
                                    <span class="w-2 h-2 bg-blue-400 rounded-full"></span>
                                    <strong class="text-white">Conditions Summary</strong> - Quick preview of rule conditions
                                </li>
                                <li class="flex items-center gap-2">
                                    <span class="w-2 h-2 bg-blue-400 rounded-full"></span>
                                    <strong class="text-white">Actions</strong> - Edit, delete, or reorder rules
                                </li>
                            </ul>
                        </div>
                    </div>

                    <div class="bg-blue-900/20 border border-blue-700/50 rounded-xl p-6 mb-8">
                        <div class="flex items-start gap-3">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-6 h-6 text-blue-400 flex-shrink-0">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M11.25 11.25l.041-.02a.75.75 0 011.063.852l-.708 2.836a.75.75 0 001.063.853l.041-.021M21 12a9 9 0 11-18 0 9 9 0 0118 0zm-9-3.75h.008v.008H12V8.25z" />
                            </svg>
                            <div>
                                <h4 class="font-semibold text-blue-300 mb-1">Tip: Drag to Reorder</h4>
                                <p class="text-blue-200/80 text-sm">You can drag and drop rules to change their priority order. Rules at the top (lower priority number) are evaluated first.</p>
                            </div>
                        </div>
                    </div>

                    <h3 id="creating-rule" class="text-xl font-semibold text-white mt-8 mb-4">Creating a Rule</h3>
                    <p class="text-zinc-300 mb-4">
                        Click the <strong class="text-white">"Add Rule"</strong> button to create a new shipping rule. The rule builder has three main sections:
                    </p>

                    <div class="grid gap-4 mb-6">
                        <div class="bg-zinc-800 border border-zinc-700 rounded-xl p-5">
                            <div class="flex items-center gap-3 mb-3">
                                <span class="w-8 h-8 bg-zinc-700 rounded-lg flex items-center justify-center text-sm font-bold text-white">1</span>
                                <h4 class="font-semibold text-white">Basic Information</h4>
                            </div>
                            <ul class="space-y-2 text-zinc-300 text-sm ml-11">
                                <li><strong class="text-white">Name</strong> - A descriptive name for your rule</li>
                                <li><strong class="text-white">Priority</strong> - The order in which rules are evaluated (0 = highest priority)</li>
                                <li><strong class="text-white">Enabled</strong> - Toggle to activate or deactivate the rule</li>
                                <li><strong class="text-white">Description</strong> - Optional notes about what this rule does</li>
                                <li><strong class="text-white">Match Type</strong> - Whether ALL or ANY cart items must match conditions</li>
                            </ul>
                        </div>

                        <div class="bg-zinc-800 border border-zinc-700 rounded-xl p-5">
                            <div class="flex items-center gap-3 mb-3">
                                <span class="w-8 h-8 bg-zinc-700 rounded-lg flex items-center justify-center text-sm font-bold text-white">2</span>
                                <h4 class="font-semibold text-white">WHEN (Conditions)</h4>
                            </div>
                            <p class="text-zinc-300 text-sm ml-11">Define the criteria that must be met for this rule to apply. Conditions check cart data, product options, custom fields, and more.</p>
                        </div>

                        <div class="bg-zinc-800 border border-zinc-700 rounded-xl p-5">
                            <div class="flex items-center gap-3 mb-3">
                                <span class="w-8 h-8 bg-zinc-700 rounded-lg flex items-center justify-center text-sm font-bold text-white">3</span>
                                <h4 class="font-semibold text-white">THEN (Actions)</h4>
                            </div>
                            <p class="text-zinc-300 text-sm ml-11">Specify which shipping rates to include or exclude when the conditions are met.</p>
                        </div>
                    </div>
                </section>

                <!-- Conditions Section -->
                <section id="conditions" class="mb-16">
                    <h2 class="text-3xl font-bold text-white mb-4 flex items-center gap-3">
                        <span class="w-10 h-10 bg-purple-600/20 rounded-lg flex items-center justify-center">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-5 h-5 text-purple-400">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M10.5 6h9.75M10.5 6a1.5 1.5 0 11-3 0m3 0a1.5 1.5 0 10-3 0M3.75 6H7.5m3 12h9.75m-9.75 0a1.5 1.5 0 01-3 0m3 0a1.5 1.5 0 00-3 0m-3.75 0H7.5m9-6h3.75m-3.75 0a1.5 1.5 0 01-3 0m3 0a1.5 1.5 0 00-3 0m-9.75 0h9.75" />
                            </svg>
                        </span>
                        Conditions
                    </h2>
                    <p class="text-zinc-300 text-lg mb-6">
                        Conditions define when a rule should be applied. Each condition checks a specific field against a value using an operator.
                    </p>

                    <h3 id="field-sources" class="text-xl font-semibold text-white mt-8 mb-4">Field Sources</h3>
                    <p class="text-zinc-300 mb-4">Choose where to look for data when evaluating conditions:</p>

                    <div class="overflow-x-auto mb-8">
                        <table class="w-full text-sm">
                            <thead>
                                <tr class="border-b border-zinc-700">
                                    <th class="text-left py-3 px-4 font-semibold text-white">Source</th>
                                    <th class="text-left py-3 px-4 font-semibold text-white">Description</th>
                                    <th class="text-left py-3 px-4 font-semibold text-white">Available Fields</th>
                                </tr>
                            </thead>
                            <tbody class="text-zinc-300">
                                <tr class="border-b border-zinc-800">
                                    <td class="py-3 px-4">
                                        <span class="inline-flex items-center gap-1.5 px-2.5 py-1 bg-blue-900/30 text-blue-300 rounded-md text-xs font-medium">
                                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-3.5 h-3.5">
                                                <path stroke-linecap="round" stroke-linejoin="round" d="M2.25 3h1.386c.51 0 .955.343 1.087.835l.383 1.437M7.5 14.25a3 3 0 00-3 3h15.75m-12.75-3h11.218c1.121-2.3 2.1-4.684 2.924-7.138a60.114 60.114 0 00-16.536-1.84M7.5 14.25L5.106 5.272M6 20.25a.75.75 0 11-1.5 0 .75.75 0 011.5 0zm12.75 0a.75.75 0 11-1.5 0 .75.75 0 011.5 0z" />
                                            </svg>
                                            Cart
                                        </span>
                                    </td>
                                    <td class="py-3 px-4">Cart-level information including shipping destination</td>
                                    <td class="py-3 px-4">
                                        <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded">country_iso2</code>
                                        <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded ml-1">state</code>
                                        <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded ml-1">city</code>
                                        <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded ml-1">postal_code</code>
                                    </td>
                                </tr>
                                <tr class="border-b border-zinc-800">
                                    <td class="py-3 px-4">
                                        <span class="inline-flex items-center gap-1.5 px-2.5 py-1 bg-green-900/30 text-green-300 rounded-md text-xs font-medium">
                                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-3.5 h-3.5">
                                                <path stroke-linecap="round" stroke-linejoin="round" d="M20.25 7.5l-.625 10.632a2.25 2.25 0 01-2.247 2.118H6.622a2.25 2.25 0 01-2.247-2.118L3.75 7.5M10 11.25h4M3.375 7.5h17.25c.621 0 1.125-.504 1.125-1.125v-1.5c0-.621-.504-1.125-1.125-1.125H3.375c-.621 0-1.125.504-1.125 1.125v1.5c0 .621.504 1.125 1.125 1.125z" />
                                            </svg>
                                            Item
                                        </span>
                                    </td>
                                    <td class="py-3 px-4">Line item data from the checkout</td>
                                    <td class="py-3 px-4">
                                        <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded">quantity</code>
                                        <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded ml-1">sku</code>
                                        <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded ml-1">name</code>
                                        <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded ml-1">price</code>
                                    </td>
                                </tr>
                                <tr class="border-b border-zinc-800">
                                    <td class="py-3 px-4">
                                        <span class="inline-flex items-center gap-1.5 px-2.5 py-1 bg-amber-900/30 text-amber-300 rounded-md text-xs font-medium">
                                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-3.5 h-3.5">
                                                <path stroke-linecap="round" stroke-linejoin="round" d="M21 7.5l-9-5.25L3 7.5m18 0l-9 5.25m9-5.25v9l-9 5.25M3 7.5l9 5.25M3 7.5v9l9 5.25m0-9v9" />
                                            </svg>
                                            Product
                                        </span>
                                    </td>
                                    <td class="py-3 px-4">Product catalog data</td>
                                    <td class="py-3 px-4">
                                        <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded">categories</code>
                                        <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded ml-1">brand_id</code>
                                        <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded ml-1">weight</code>
                                    </td>
                                </tr>
                                <tr class="border-b border-zinc-800">
                                    <td class="py-3 px-4">
                                        <span class="inline-flex items-center gap-1.5 px-2.5 py-1 bg-cyan-900/30 text-cyan-300 rounded-md text-xs font-medium">
                                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-3.5 h-3.5">
                                                <path stroke-linecap="round" stroke-linejoin="round" d="M9.594 3.94c.09-.542.56-.94 1.11-.94h2.593c.55 0 1.02.398 1.11.94l.213 1.281c.063.374.313.686.645.87.074.04.147.083.22.127.324.196.72.257 1.075.124l1.217-.456a1.125 1.125 0 011.37.49l1.296 2.247a1.125 1.125 0 01-.26 1.431l-1.003.827c-.293.24-.438.613-.431.992a6.759 6.759 0 010 .255c-.007.378.138.75.43.99l1.005.828c.424.35.534.954.26 1.43l-1.298 2.247a1.125 1.125 0 01-1.369.491l-1.217-.456c-.355-.133-.75-.072-1.076.124a6.57 6.57 0 01-.22.128c-.331.183-.581.495-.644.869l-.213 1.28c-.09.543-.56.941-1.11.941h-2.594c-.55 0-1.02-.398-1.11-.94l-.213-1.281c-.062-.374-.312-.686-.644-.87a6.52 6.52 0 01-.22-.127c-.325-.196-.72-.257-1.076-.124l-1.217.456a1.125 1.125 0 01-1.369-.49l-1.297-2.247a1.125 1.125 0 01.26-1.431l1.004-.827c.292-.24.437-.613.43-.992a6.932 6.932 0 010-.255c.007-.378-.138-.75-.43-.99l-1.004-.828a1.125 1.125 0 01-.26-1.43l1.297-2.247a1.125 1.125 0 011.37-.491l1.216.456c.356.133.751.072 1.076-.124.072-.044.146-.087.22-.128.332-.183.582-.495.644-.869l.214-1.281z" />
                                                <path stroke-linecap="round" stroke-linejoin="round" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                                            </svg>
                                            Option
                                        </span>
                                    </td>
                                    <td class="py-3 px-4">Product options selected by customer</td>
                                    <td class="py-3 px-4">
                                        <span class="text-zinc-400 italic">Enter option name (e.g., "free sample", "color")</span>
                                    </td>
                                </tr>
                                <tr>
                                    <td class="py-3 px-4">
                                        <span class="inline-flex items-center gap-1.5 px-2.5 py-1 bg-pink-900/30 text-pink-300 rounded-md text-xs font-medium">
                                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-3.5 h-3.5">
                                                <path stroke-linecap="round" stroke-linejoin="round" d="M9.568 3H5.25A2.25 2.25 0 003 5.25v4.318c0 .597.237 1.17.659 1.591l9.581 9.581c.699.699 1.78.872 2.607.33a18.095 18.095 0 005.223-5.223c.542-.827.369-1.908-.33-2.607L11.16 3.66A2.25 2.25 0 009.568 3z" />
                                                <path stroke-linecap="round" stroke-linejoin="round" d="M6 6h.008v.008H6V6z" />
                                            </svg>
                                            Custom Field
                                        </span>
                                    </td>
                                    <td class="py-3 px-4">Product custom fields from BigCommerce</td>
                                    <td class="py-3 px-4">
                                        <span class="text-zinc-400 italic">Enter custom field name (e.g., "item type", "pallet")</span>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>

                    <h3 id="operators" class="text-xl font-semibold text-white mt-8 mb-4">Operators</h3>
                    <p class="text-zinc-300 mb-4">Choose how to compare the field value:</p>

                    <div class="grid md:grid-cols-2 gap-4 mb-8">
                        <div class="bg-zinc-800 border border-zinc-700 rounded-xl p-4">
                            <h4 class="font-semibold text-white mb-3">Text Operators</h4>
                            <ul class="space-y-2 text-sm text-zinc-300">
                                <li><code class="text-blue-300">equals</code> - Exact match</li>
                                <li><code class="text-blue-300">not_equals</code> - Does not match</li>
                                <li><code class="text-blue-300">contains</code> - Contains the text</li>
                                <li><code class="text-blue-300">not_contains</code> - Does not contain</li>
                                <li><code class="text-blue-300">in_list</code> - Matches any value in comma-separated list</li>
                                <li><code class="text-blue-300">regex</code> - Regular expression match</li>
                            </ul>
                        </div>
                        <div class="bg-zinc-800 border border-zinc-700 rounded-xl p-4">
                            <h4 class="font-semibold text-white mb-3">Numeric Operators</h4>
                            <ul class="space-y-2 text-sm text-zinc-300">
                                <li><code class="text-green-300">greater_than</code> - Value is greater than</li>
                                <li><code class="text-green-300">less_than</code> - Value is less than</li>
                                <li><code class="text-green-300">between</code> - Value is within range (inclusive)</li>
                            </ul>
                        </div>
                    </div>

                    <h3 id="and-or-logic" class="text-xl font-semibold text-white mt-8 mb-4">AND/OR Logic</h3>
                    <p class="text-zinc-300 mb-4">
                        Combine multiple conditions using AND/OR logic to create complex rules:
                    </p>

                    <div class="bg-zinc-800 border border-zinc-700 rounded-xl p-6 mb-6">
                        <div class="space-y-4">
                            <div>
                                <div class="flex items-center gap-2 mb-2">
                                    <span class="px-2 py-0.5 text-xs font-bold text-blue-400 bg-blue-900/30 rounded">AND</span>
                                    <span class="text-white font-medium">All conditions in a group must be true</span>
                                </div>
                                <p class="text-zinc-400 text-sm ml-12">Add conditions within the same group using "Add AND Condition"</p>
                            </div>
                            <div class="border-t border-zinc-700 pt-4">
                                <div class="flex items-center gap-2 mb-2">
                                    <span class="px-2 py-0.5 text-xs font-bold text-orange-400 bg-orange-900/30 rounded">OR</span>
                                    <span class="text-white font-medium">Any one of the groups must be true</span>
                                </div>
                                <p class="text-zinc-400 text-sm ml-12">Create separate condition groups using "Add OR Group"</p>
                            </div>
                        </div>
                    </div>

                    <div class="bg-zinc-850 border border-zinc-700 rounded-xl overflow-hidden">
                        <div class="bg-zinc-800 px-4 py-3 border-b border-zinc-700">
                            <span class="text-sm font-medium text-zinc-400">Example: Complex Logic</span>
                        </div>
                        <div class="p-5 font-mono text-sm">
                            <div class="text-zinc-300">
                                <span class="text-zinc-500">// This rule matches if:</span><br>
                                <span class="text-blue-400">(</span> option "free sample" = "yes" <span class="text-blue-400">AND</span> quantity &gt; 0 <span class="text-blue-400">)</span><br>
                                <span class="text-orange-400 ml-4">OR</span><br>
                                <span class="text-blue-400">(</span> custom field "item type" = "accessory" <span class="text-blue-400">)</span>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- Actions Section -->
                <section id="actions" class="mb-16">
                    <h2 class="text-3xl font-bold text-white mb-4 flex items-center gap-3">
                        <span class="w-10 h-10 bg-orange-600/20 rounded-lg flex items-center justify-center">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-5 h-5 text-orange-400">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M3.75 13.5l10.5-11.25L12 10.5h8.25L9.75 21.75 12 13.5H3.75z" />
                            </svg>
                        </span>
                        Actions
                    </h2>
                    <p class="text-zinc-300 text-lg mb-6">
                        Actions determine which shipping rates to show when a rule's conditions are met. You can include specific rates or exclude unwanted ones.
                    </p>

                    <div class="grid md:grid-cols-2 gap-4 mb-8">
                        <div class="bg-green-900/20 border border-green-700/50 rounded-xl p-5">
                            <div class="flex items-center gap-2 mb-3">
                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-5 h-5 text-green-400">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 12.75L11.25 15 15 9.75M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                </svg>
                                <h4 class="font-semibold text-green-300">Include Rates</h4>
                            </div>
                            <p class="text-green-200/70 text-sm">Only show rates that match the specified pattern. All other rates will be hidden.</p>
                        </div>
                        <div class="bg-red-900/20 border border-red-700/50 rounded-xl p-5">
                            <div class="flex items-center gap-2 mb-3">
                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-5 h-5 text-red-400">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M18.364 18.364A9 9 0 005.636 5.636m12.728 12.728A9 9 0 015.636 5.636m12.728 12.728L5.636 5.636" />
                                </svg>
                                <h4 class="font-semibold text-red-300">Exclude Rates</h4>
                            </div>
                            <p class="text-red-200/70 text-sm">Hide rates that match the specified pattern. All other rates will be shown.</p>
                        </div>
                    </div>

                    <h3 class="text-xl font-semibold text-white mt-8 mb-4">Rate Pattern Matching</h3>
                    <p class="text-zinc-300 mb-4">
                        Specify which rates to include or exclude by matching against the rate code:
                    </p>

                    <div class="overflow-x-auto mb-6">
                        <table class="w-full text-sm">
                            <thead>
                                <tr class="border-b border-zinc-700">
                                    <th class="text-left py-3 px-4 font-semibold text-white">Match Type</th>
                                    <th class="text-left py-3 px-4 font-semibold text-white">Description</th>
                                    <th class="text-left py-3 px-4 font-semibold text-white">Example</th>
                                </tr>
                            </thead>
                            <tbody class="text-zinc-300">
                                <tr class="border-b border-zinc-800">
                                    <td class="py-3 px-4"><code class="text-blue-300">contains</code></td>
                                    <td class="py-3 px-4">Rate code contains the pattern</td>
                                    <td class="py-3 px-4">"sample" matches <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded">free_sample</code>, <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded">sample_delivery</code></td>
                                </tr>
                                <tr class="border-b border-zinc-800">
                                    <td class="py-3 px-4"><code class="text-blue-300">equals</code></td>
                                    <td class="py-3 px-4">Rate code exactly matches</td>
                                    <td class="py-3 px-4">"std_delivery" matches only <code class="text-xs bg-zinc-800 px-1.5 py-0.5 rounded">std_delivery</code></td>
                                </tr>
                                <tr>
                                    <td class="py-3 px-4"><code class="text-blue-300">regex</code></td>
                                    <td class="py-3 px-4">Regular expression match</td>
                                    <td class="py-3 px-4">"^pallet_" matches codes starting with "pallet_"</td>
                                </tr>
                            </tbody>
                        </table>
                    </div>

                    <div class="bg-blue-900/20 border border-blue-700/50 rounded-xl p-6">
                        <div class="flex items-start gap-3">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-6 h-6 text-blue-400 flex-shrink-0">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M11.25 11.25l.041-.02a.75.75 0 011.063.852l-.708 2.836a.75.75 0 001.063.853l.041-.021M21 12a9 9 0 11-18 0 9 9 0 0118 0zm-9-3.75h.008v.008H12V8.25z" />
                            </svg>
                            <div>
                                <h4 class="font-semibold text-blue-300 mb-1">Tip: Select Available Rates</h4>
                                <p class="text-blue-200/80 text-sm">The Rate Pattern dropdown shows all available rates from your carriers. You can select a specific rate or enter a custom pattern to match multiple rates.</p>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- Examples Section -->
                <section id="examples" class="mb-16">
                    <h2 class="text-3xl font-bold text-white mb-4 flex items-center gap-3">
                        <span class="w-10 h-10 bg-cyan-600/20 rounded-lg flex items-center justify-center">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-5 h-5 text-cyan-400">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M12 6.042A8.967 8.967 0 006 3.75c-1.052 0-2.062.18-3 .512v14.25A8.987 8.987 0 016 18c2.305 0 4.408.867 6 2.292m0-14.25a8.966 8.966 0 016-2.292c1.052 0 2.062.18 3 .512v14.25A8.987 8.987 0 0018 18a8.967 8.967 0 00-6 2.292m0-14.25v14.25" />
                            </svg>
                        </span>
                        Examples
                    </h2>
                    <p class="text-zinc-300 text-lg mb-6">
                        Here are some common shipping rule scenarios:
                    </p>

                    <!-- Example 1 -->
                    <div class="bg-zinc-800 border border-zinc-700 rounded-xl overflow-hidden mb-6">
                        <div class="bg-zinc-850 px-5 py-4 border-b border-zinc-700">
                            <h3 class="font-semibold text-white">Example 1: Free Sample Shipping</h3>
                            <p class="text-sm text-zinc-400 mt-1">Only show sample shipping rates when all items are free samples</p>
                        </div>
                        <div class="p-5 space-y-4">
                            <div>
                                <span class="text-xs font-semibold text-zinc-500 uppercase tracking-wider">Conditions</span>
                                <div class="mt-2 bg-zinc-900 rounded-lg p-3 text-sm">
                                    <span class="text-cyan-300">Option</span> "free sample" <span class="text-blue-300">equals</span> "yes"
                                </div>
                            </div>
                            <div>
                                <span class="text-xs font-semibold text-zinc-500 uppercase tracking-wider">Actions</span>
                                <div class="mt-2 bg-zinc-900 rounded-lg p-3 text-sm">
                                    <span class="text-green-300">Include</span> rates <span class="text-blue-300">containing</span> "sample"
                                </div>
                            </div>
                            <div>
                                <span class="text-xs font-semibold text-zinc-500 uppercase tracking-wider">Match Type</span>
                                <div class="mt-2 text-sm text-zinc-300">ALL items must match</div>
                            </div>
                        </div>
                    </div>

                    <!-- Example 2 -->
                    <div class="bg-zinc-800 border border-zinc-700 rounded-xl overflow-hidden mb-6">
                        <div class="bg-zinc-850 px-5 py-4 border-b border-zinc-700">
                            <h3 class="font-semibold text-white">Example 2: Pallet Shipping for Heavy Items</h3>
                            <p class="text-sm text-zinc-400 mt-1">Show only pallet shipping when any item requires a pallet</p>
                        </div>
                        <div class="p-5 space-y-4">
                            <div>
                                <span class="text-xs font-semibold text-zinc-500 uppercase tracking-wider">Conditions</span>
                                <div class="mt-2 bg-zinc-900 rounded-lg p-3 text-sm">
                                    <span class="text-pink-300">Custom Field</span> "pallet" <span class="text-blue-300">equals</span> "T"
                                </div>
                            </div>
                            <div>
                                <span class="text-xs font-semibold text-zinc-500 uppercase tracking-wider">Actions</span>
                                <div class="mt-2 bg-zinc-900 rounded-lg p-3 text-sm space-y-1">
                                    <div><span class="text-green-300">Include</span> rates <span class="text-blue-300">containing</span> "pallet"</div>
                                    <div><span class="text-red-300">Exclude</span> rates <span class="text-blue-300">containing</span> "free"</div>
                                    <div><span class="text-red-300">Exclude</span> rates <span class="text-blue-300">containing</span> "sample"</div>
                                </div>
                            </div>
                            <div>
                                <span class="text-xs font-semibold text-zinc-500 uppercase tracking-wider">Match Type</span>
                                <div class="mt-2 text-sm text-zinc-300">ANY item must match</div>
                            </div>
                        </div>
                    </div>

                    <!-- Example 3 -->
                    <div class="bg-zinc-800 border border-zinc-700 rounded-xl overflow-hidden mb-6">
                        <div class="bg-zinc-850 px-5 py-4 border-b border-zinc-700">
                            <h3 class="font-semibold text-white">Example 3: Country-Specific Rates</h3>
                            <p class="text-sm text-zinc-400 mt-1">Show international shipping only for non-UK destinations</p>
                        </div>
                        <div class="p-5 space-y-4">
                            <div>
                                <span class="text-xs font-semibold text-zinc-500 uppercase tracking-wider">Conditions</span>
                                <div class="mt-2 bg-zinc-900 rounded-lg p-3 text-sm">
                                    <span class="text-blue-300">Cart</span> "country_iso2" <span class="text-blue-300">not_equals</span> "GB"
                                </div>
                            </div>
                            <div>
                                <span class="text-xs font-semibold text-zinc-500 uppercase tracking-wider">Actions</span>
                                <div class="mt-2 bg-zinc-900 rounded-lg p-3 text-sm">
                                    <span class="text-green-300">Include</span> rates <span class="text-blue-300">containing</span> "international"
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Example 4 -->
                    <div class="bg-zinc-800 border border-zinc-700 rounded-xl overflow-hidden">
                        <div class="bg-zinc-850 px-5 py-4 border-b border-zinc-700">
                            <h3 class="font-semibold text-white">Example 4: Standard Shipping (Default)</h3>
                            <p class="text-sm text-zinc-400 mt-1">Show standard rates and hide special shipping options</p>
                        </div>
                        <div class="p-5 space-y-4">
                            <div>
                                <span class="text-xs font-semibold text-zinc-500 uppercase tracking-wider">Conditions</span>
                                <div class="mt-2 bg-zinc-900 rounded-lg p-3 text-sm text-zinc-400 italic">
                                    (No conditions - always matches as a fallback)
                                </div>
                            </div>
                            <div>
                                <span class="text-xs font-semibold text-zinc-500 uppercase tracking-wider">Actions</span>
                                <div class="mt-2 bg-zinc-900 rounded-lg p-3 text-sm space-y-1">
                                    <div><span class="text-red-300">Exclude</span> rates <span class="text-blue-300">containing</span> "free"</div>
                                    <div><span class="text-red-300">Exclude</span> rates <span class="text-blue-300">containing</span> "sample"</div>
                                    <div><span class="text-red-300">Exclude</span> rates <span class="text-blue-300">containing</span> "accessory"</div>
                                </div>
                            </div>
                            <div>
                                <span class="text-xs font-semibold text-zinc-500 uppercase tracking-wider">Priority</span>
                                <div class="mt-2 text-sm text-zinc-300">99 (lowest priority - evaluated last)</div>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- Troubleshooting Section -->
                <section id="troubleshooting" class="mb-16">
                    <h2 class="text-3xl font-bold text-white mb-4 flex items-center gap-3">
                        <span class="w-10 h-10 bg-red-600/20 rounded-lg flex items-center justify-center">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="w-5 h-5 text-red-400">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M11.42 15.17L17.25 21A2.652 2.652 0 0021 17.25l-5.877-5.877M11.42 15.17l2.496-3.03c.317-.384.74-.626 1.208-.766M11.42 15.17l-4.655 5.653a2.548 2.548 0 11-3.586-3.586l6.837-5.63m5.108-.233c.55-.164 1.163-.188 1.743-.14a4.5 4.5 0 004.486-6.336l-3.276 3.277a3.004 3.004 0 01-2.25-2.25l3.276-3.276a4.5 4.5 0 00-6.336 4.486c.091 1.076-.071 2.264-.904 2.95l-.102.085m-1.745 1.437L5.909 7.5H4.5L2.25 3.75l1.5-1.5L7.5 4.5v1.409l4.26 4.26m-1.745 1.437l1.745-1.437m6.615 8.206L15.75 15.75M4.867 19.125h.008v.008h-.008v-.008z" />
                            </svg>
                        </span>
                        Troubleshooting
                    </h2>

                    <div class="space-y-4">
                        <div class="bg-zinc-800 border border-zinc-700 rounded-xl p-5">
                            <h4 class="font-semibold text-white mb-2">No shipping rates appear at checkout</h4>
                            <ul class="text-sm text-zinc-300 space-y-1">
                                <li>• Check that at least one rule matches the cart contents</li>
                                <li>• Verify the actions aren't excluding all available rates</li>
                                <li>• Ensure you have carriers and rates configured in the Carrier section</li>
                            </ul>
                        </div>

                        <div class="bg-zinc-800 border border-zinc-700 rounded-xl p-5">
                            <h4 class="font-semibold text-white mb-2">Wrong shipping rates showing</h4>
                            <ul class="text-sm text-zinc-300 space-y-1">
                                <li>• Check rule priorities - lower numbers are evaluated first</li>
                                <li>• Verify conditions are using the correct field source and operator</li>
                                <li>• Remember that only the FIRST matching rule's actions are applied</li>
                            </ul>
                        </div>

                        <div class="bg-zinc-800 border border-zinc-700 rounded-xl p-5">
                            <h4 class="font-semibold text-white mb-2">Condition not matching</h4>
                            <ul class="text-sm text-zinc-300 space-y-1">
                                <li>• Check if "Case sensitive" is enabled when it shouldn't be</li>
                                <li>• For options and custom fields, verify the exact name spelling</li>
                                <li>• Use "contains" instead of "equals" for partial matches</li>
                            </ul>
                        </div>

                        <div class="bg-zinc-800 border border-zinc-700 rounded-xl p-5">
                            <h4 class="font-semibold text-white mb-2">Rule not being evaluated</h4>
                            <ul class="text-sm text-zinc-300 space-y-1">
                                <li>• Ensure the rule is enabled (toggle is on)</li>
                                <li>• A higher-priority rule might be matching first</li>
                                <li>• Check that conditions aren't empty or invalid</li>
                            </ul>
                        </div>
                    </div>
                </section>

                <!-- Footer -->
                <footer class="border-t border-zinc-800 pt-8 mt-16">
                    <div class="flex items-center justify-between text-sm text-zinc-500">
                        <p>BigCommerce Shipping Rates - MTC Media</p>
                        <p>Version {{ config('bigcommerce-rates.version', '0.0.14') }}</p>
                    </div>
                </footer>
            </main>
        </div>
    </div>

    <script>
        // Smooth scroll for anchor links
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function (e) {
                e.preventDefault();
                const target = document.querySelector(this.getAttribute('href'));
                if (target) {
                    target.scrollIntoView({ behavior: 'smooth', block: 'start' });
                }
            });
        });

        // Highlight active section in sidebar
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    document.querySelectorAll('aside a').forEach(link => {
                        link.classList.remove('text-white', 'bg-zinc-800');
                        link.classList.add('text-zinc-400');
                    });
                    const activeLink = document.querySelector(`aside a[href="#${entry.target.id}"]`);
                    if (activeLink) {
                        activeLink.classList.remove('text-zinc-400');
                        activeLink.classList.add('text-white', 'bg-zinc-800');
                    }
                }
            });
        }, { rootMargin: '-100px 0px -66% 0px' });

        document.querySelectorAll('section[id]').forEach(section => {
            observer.observe(section);
        });
    </script>
</body>
</html>
