<?php

namespace Mtc\BigCommerceRates\Database\Seeders;

use Illuminate\Database\Seeder;
use Mtc\BigCommerceRates\Models\BigCommerce\Rules\ShippingRule;
use Mtc\BigCommerceRates\Models\BigCommerce\Rules\RuleCondition;
use Mtc\BigCommerceRates\Models\BigCommerce\Rules\RuleAction;
use Mtc\BigCommerceRates\Models\StoreInstallation;

class SfdShippingRulesSeeder extends Seeder
{
    /**
     * Seed the default SFD shipping rules for all stores.
     * These rules replicate the original hardcoded logic from CarrierController.
     */
    public function run(): void
    {
        $stores = StoreInstallation::all();

        foreach ($stores as $store) {
            $this->seedRulesForStore($store->id);
        }
    }

    /**
     * Seed rules for a specific store.
     */
    public function seedRulesForStore(int $storeId): void
    {
        // Skip if rules already exist for this store
        if (ShippingRule::forStore($storeId)->exists()) {
            return;
        }

        // Rule 1: Free Sample Only
        // When ALL items have option "free sample" = "yes", show only sample rates
        $rule1 = ShippingRule::create([
            'store_id' => $storeId,
            'name' => 'Free Sample Only',
            'description' => 'Show only sample shipping when all items are free samples',
            'enabled' => true,
            'priority' => 1,
            'match_type' => 'all',
            'condition_logic' => 'and',
        ]);

        RuleCondition::create([
            'rule_id' => $rule1->id,
            'group_index' => 0,
            'field_source' => 'option',
            'field_name' => 'free sample',
            'field_operator' => 'equals',
            'field_value' => 'yes',
            'case_sensitive' => false,
        ]);

        RuleAction::create([
            'rule_id' => $rule1->id,
            'action_type' => 'include',
            'rate_pattern' => 'sample',
            'pattern_operator' => 'contains',
        ]);

        // Rule 2: All Accessories
        // When ALL items have custom field "item type" = "accessory", show accessory rates
        $rule2 = ShippingRule::create([
            'store_id' => $storeId,
            'name' => 'All Accessories',
            'description' => 'Show accessory shipping when all items are accessories',
            'enabled' => true,
            'priority' => 2,
            'match_type' => 'all',
            'condition_logic' => 'and',
        ]);

        RuleCondition::create([
            'rule_id' => $rule2->id,
            'group_index' => 0,
            'field_source' => 'custom_field',
            'field_name' => 'item type',
            'field_operator' => 'equals',
            'field_value' => 'accessory',
            'case_sensitive' => false,
        ]);

        RuleAction::create([
            'rule_id' => $rule2->id,
            'action_type' => 'include',
            'rate_pattern' => 'accessory',
            'pattern_operator' => 'contains',
        ]);

        // Rule 3: Free Samples + Accessories Mix
        // When ALL items are either free samples OR accessories, show free shipping
        // This uses OR groups: Group 0 = free sample check, Group 1 = accessory check
        // But for "ALL items must match one of these", we need a different approach
        // Since this is complex (each item can be sample OR accessory), we create conditions
        // that check both possibilities per item
        $rule3 = ShippingRule::create([
            'store_id' => $storeId,
            'name' => 'Free Samples + Accessories Mix',
            'description' => 'Show free shipping when all items are either free samples or accessories',
            'enabled' => true,
            'priority' => 3,
            'match_type' => 'all', // All items must satisfy one of the OR groups
            'condition_logic' => 'and',
        ]);

        // Group 0: Free sample condition
        RuleCondition::create([
            'rule_id' => $rule3->id,
            'group_index' => 0,
            'field_source' => 'option',
            'field_name' => 'free sample',
            'field_operator' => 'equals',
            'field_value' => 'yes',
            'case_sensitive' => false,
        ]);

        // Group 1: Accessory condition (OR with Group 0)
        RuleCondition::create([
            'rule_id' => $rule3->id,
            'group_index' => 1,
            'field_source' => 'custom_field',
            'field_name' => 'item type',
            'field_operator' => 'equals',
            'field_value' => 'accessory',
            'case_sensitive' => false,
        ]);

        RuleAction::create([
            'rule_id' => $rule3->id,
            'action_type' => 'include',
            'rate_pattern' => 'free',
            'pattern_operator' => 'contains',
        ]);

        // Rule 4: Pallet Shipping
        // When ANY item has custom field containing "pallet" = "T", show pallet rates
        $rule4 = ShippingRule::create([
            'store_id' => $storeId,
            'name' => 'Pallet Shipping',
            'description' => 'Show pallet shipping options when any item requires pallet delivery',
            'enabled' => true,
            'priority' => 4,
            'match_type' => 'any', // ANY item can trigger this
            'condition_logic' => 'and',
        ]);

        RuleCondition::create([
            'rule_id' => $rule4->id,
            'group_index' => 0,
            'field_source' => 'custom_field',
            'field_name' => 'pallet',
            'field_operator' => 'equals',
            'field_value' => 'T',
            'case_sensitive' => false,
        ]);

        RuleAction::create([
            'rule_id' => $rule4->id,
            'action_type' => 'include',
            'rate_pattern' => 'pallet',
            'pattern_operator' => 'contains',
        ]);

        RuleAction::create([
            'rule_id' => $rule4->id,
            'action_type' => 'exclude',
            'rate_pattern' => 'free',
            'pattern_operator' => 'contains',
        ]);

        RuleAction::create([
            'rule_id' => $rule4->id,
            'action_type' => 'exclude',
            'rate_pattern' => 'sample',
            'pattern_operator' => 'contains',
        ]);

        // Rule 5: Standard Shipping (Default/Fallback)
        // No conditions - always matches as fallback. Excludes free, sample, and accessory rates.
        $rule5 = ShippingRule::create([
            'store_id' => $storeId,
            'name' => 'Standard Shipping',
            'description' => 'Default shipping options - excludes special rates',
            'enabled' => true,
            'priority' => 99,
            'match_type' => 'all',
            'condition_logic' => 'and',
        ]);

        // No conditions - this rule always matches as a fallback

        RuleAction::create([
            'rule_id' => $rule5->id,
            'action_type' => 'exclude',
            'rate_pattern' => 'free',
            'pattern_operator' => 'contains',
        ]);

        RuleAction::create([
            'rule_id' => $rule5->id,
            'action_type' => 'exclude',
            'rate_pattern' => 'sample',
            'pattern_operator' => 'contains',
        ]);

        RuleAction::create([
            'rule_id' => $rule5->id,
            'action_type' => 'exclude',
            'rate_pattern' => 'accessory',
            'pattern_operator' => 'contains',
        ]);
    }
}
