<?php

namespace Mtc\BigCommerceRates\Livewire\BigCommerce\Product;

use Mtc\BigCommerceRates\Models\BigCommerce\Product;
use Mtc\BigCommerceRates\Models\StoreInstallation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Pagination\LengthAwarePaginator;
use Tymon\JWTAuth\Facades\JWTAuth;


class Import extends Component
{
    use WithPagination;

    public $storeHash;

    // Import progress tracking
    public $isImporting = false;
    public $importedCount = 0;
    public $currentPage = 0;
    public $totalPages = 0;
    public $totalProducts = 0;


    public function mount($storeHash)
    {
        // Try to get from parameter, then from query, then from session, then from the authenticated user
        $this->storeHash = $storeHash
            ?? (auth()->user()?->storeInstallation?->store_hash ?? null);

    }

    public function getUserForStore()
    {
        // First check if user is session-authenticated (for local development)
        if (Auth::check()) {
            return Auth::user();
        }

        // Otherwise try JWT from cookie (for BigCommerce iframe)
        $storeHash = request()->route('storeHash');
        Log::info('Store hash:', ['store_hash' => $storeHash]);
        $cookieName = 'bc_jwt_' . $storeHash;
        $token = request()->cookie($cookieName) ?? ($_COOKIE[$cookieName] ?? null);
        Log::info('JWT cookie value:', ['cookie' => $cookieName, 'token' => $token ? 'found' : 'not found']);
        if ($token) {
            try {
                $user = \Tymon\JWTAuth\Facades\JWTAuth::setToken($token)->authenticate();
                Log::info('User from JWT:', ['user' => $user]);
                return $user;
            } catch (\Exception $e) {
                Log::error('JWT error:', ['message' => $e->getMessage()]);
                return null;
            }
        }
        return null;
    }

    public function importProducts()
    {
        // Get the current user
        $user = $this->getUserForStore();
        if (!$user) {
            session()->flash('error', 'No authenticated user found.');
            return;
        }

        $store = $user->storeInstallation;

        if (!$store) {
            session()->flash('error', 'No store installation found for this user.');
            return;
        }

        // Reset progress tracking
        $this->isImporting = true;
        $this->importedCount = 0;
        $this->currentPage = 0;
        $this->totalPages = 0;
        $this->totalProducts = 0;

        $storeHash = $store->store_hash;

        // First, get total count
        $this->fetchTotalProductCount($storeHash);

        $importedCount = $this->importProductsForStore($storeHash, $store, 1);

        $this->isImporting = false;
        session()->flash('success', "Successfully imported or updated {$importedCount} products for store: {$storeHash}");

        // Reset pagination to show the new data
        $this->resetPage();
    }

    protected function fetchTotalProductCount($storeHash)
    {
        $endpointUrl = env('APP_URL')."bc-api/{$storeHash}/v3/catalog/products";
        $queryParams = [
            "include_fields" => "id",
            "limit" => 1,
            "page" => 1
        ];

        try {
            $response = Http::timeout(30)
                ->connectTimeout(5)
                ->get($endpointUrl, $queryParams)
                ->json();

            $this->totalProducts = data_get($response, 'meta.pagination.total', 0);
            $this->totalPages = data_get($response, 'meta.pagination.total_pages', 1);
        } catch (\Exception $e) {
            Log::error('Failed to fetch product count: ' . $e->getMessage());
        }
    }

    protected function importProductsForStore($selectedStore, $store, $page = 1)
    {
        $endpointUrl = env('APP_URL')."bc-api/{$selectedStore}/v3/catalog/products";
        $queryParams = [
            "include" => "custom_fields",
            "include_fields" => "id,name,sku,price",
            "limit" => 250,
            "direction" => "desc",
            "page" => $page
        ];

        // Update current page for progress display
        $this->currentPage = $page;

        try {
            $response = Http::timeout(400)
                ->connectTimeout(10)
                ->get($endpointUrl, $queryParams)
                ->collect();
        } catch (\Illuminate\Http\Client\HttpClientException $clientException) {
            $this->isImporting = false;
            session()->flash('error', "HTTP Client Exception: " . $clientException->getMessage());
            return 0;
        }

        if ($response->isEmpty()) {
            $this->isImporting = false;
            session()->flash('error', "No products returned from API.");
            return 0;
        }

        $importedCount = 0;
        foreach ($response["data"] as $productData) {
            $product = Product::updateOrCreate(
                [
                    'product_id' => $productData['id'],
                    'store_id' => $store->id,
                ],
                [
                    'sku' => $productData['sku'] ?? null,
                    'name' => $productData['name'] ?? '',
                    'price' => $productData['price'] ?? '0',
                    'custom_fields' => $productData['custom_fields'] ?? null,
                ]
            );
            // Force updated_at refresh even if no data changed
            $product->touch();
            $importedCount++;
            $this->importedCount++;
        }

        $currentPage = data_get($response, 'meta.pagination.current_page', 1);
        $totalPages = data_get($response, 'meta.pagination.total_pages', 1);
        $this->totalPages = $totalPages;

        if ($currentPage < $totalPages) {
            $importedCount += $this->importProductsForStore($selectedStore, $store, $currentPage + 1);
        }

        return $importedCount;
    }

    public function render()
    {
        $user = $this->getUserForStore();
        $totalProductsInDb = 0;
        $lastImportAt = null;

        if ($user && $user->storeInstallation) {
            $store = $user->storeInstallation;
            $products = Product::where('store_id', $store->id)
                ->latest()
                ->paginate(10);

            // Get stats
            $totalProductsInDb = Product::where('store_id', $store->id)->count();
            $lastProduct = Product::where('store_id', $store->id)->latest('updated_at')->first();
            $lastImportAt = $lastProduct?->updated_at;
        } else {
            // Always return a paginator, even if empty
            $products = new LengthAwarePaginator([], 0, 10);
        }

        return view('bigcommerce::livewire.bigcommerce.product.import', [
            'products' => $products,
            'totalProductsInDb' => $totalProductsInDb,
            'lastImportAt' => $lastImportAt,
        ]);
    }
}
