<?php

namespace Mtc\BigCommerceRates\Livewire\BigCommerce;

use Livewire\Component;
use Livewire\WithPagination;
use Mtc\BigCommerceRates\Models\BigCommerce\Rules\ShippingRule;
use Mtc\BigCommerceRates\Models\StoreInstallation;

class Rules extends Component
{
    use WithPagination;

    public $storeHash;
    public $showDeleteModal = false;
    public $ruleToDelete = null;

    protected $listeners = [
        'refreshRules' => '$refresh',
        'updateRulePriorities' => 'updatePriorities',
    ];

    public function mount($storeHash = null)
    {
        $this->storeHash = $storeHash
            ?? (auth()->user()?->storeInstallation?->store_hash ?? null);
    }

    public function getRulesProperty()
    {
        $store = StoreInstallation::findByStoreHash($this->storeHash);

        if (!$store) {
            return collect();
        }

        return ShippingRule::forStore($store->id)
            ->byPriority()
            ->with(['conditions', 'actions'])
            ->get();
    }

    public function toggleEnabled($ruleId)
    {
        $rule = ShippingRule::find($ruleId);
        if ($rule) {
            $rule->enabled = !$rule->enabled;
            $rule->save();
        }
    }

    public function updatePriorities($orderedIds)
    {
        foreach ($orderedIds as $priority => $ruleId) {
            ShippingRule::where('id', $ruleId)->update(['priority' => $priority]);
        }
        $this->dispatch('notify', message: 'Rule priorities updated');
    }

    public function confirmDelete($ruleId)
    {
        $this->ruleToDelete = $ruleId;
        $this->showDeleteModal = true;
    }

    public function deleteRule()
    {
        if ($this->ruleToDelete) {
            $rule = ShippingRule::find($this->ruleToDelete);
            if ($rule) {
                $rule->delete();
                $this->dispatch('notify', message: 'Rule deleted successfully');
            }
        }
        $this->showDeleteModal = false;
        $this->ruleToDelete = null;
    }

    public function cancelDelete()
    {
        $this->showDeleteModal = false;
        $this->ruleToDelete = null;
    }

    public function duplicateRule($ruleId)
    {
        $rule = ShippingRule::with(['conditions', 'actions'])->find($ruleId);
        if (!$rule) return;

        // Create a copy of the rule
        $newRule = $rule->replicate();
        $newRule->name = $rule->name . ' (Copy)';
        $newRule->priority = ShippingRule::forStore($rule->store_id)->max('priority') + 1;
        $newRule->save();

        // Duplicate conditions
        foreach ($rule->conditions as $condition) {
            $newCondition = $condition->replicate();
            $newCondition->rule_id = $newRule->id;
            $newCondition->save();
        }

        // Duplicate actions
        foreach ($rule->actions as $action) {
            $newAction = $action->replicate();
            $newAction->rule_id = $newRule->id;
            $newAction->save();
        }

        $this->dispatch('notify', message: 'Rule duplicated successfully');
    }

    public function render()
    {
        return view('bigcommerce::livewire.bigcommerce.rules.index', [
            'rules' => $this->rules,
        ]);
    }
}
