<?php

namespace Mtc\BigCommerceRates\Http\Controllers\BigCommerce;

use Illuminate\Http\Request;
use Mtc\BigCommerceRates\Models\StoreInstallation;
use Mtc\BigCommerceRates\Configuration\Configuration;
use App\Http\Controllers\Controller as BaseController;
use Illuminate\Support\Facades\Http;

class Proxy extends BaseController
{

    public function __construct(
        public Request $request,
        protected Configuration $configuration,
    )
    {
        $this->baseURL = env('APP_URL');

    }

    protected function buildHeaders( string $accessToken): array
    {

        return [
            'X-Auth-Client' => $this->configuration->getAppClientId(),
            'X-Auth-Token'  => $accessToken,
            'Content-Type'  => 'application/json',
        ];
    }

    public function makeBigCommerceAPIRequest(string $storeHash, string $endpoint)
    {

        $installation = StoreInstallation::where('store_hash', $storeHash)->first();
        if (!$installation || !$installation->access_token) {
            abort(404, 'Store installation not found or missing access token');
        }

        $queryString = $this->request->getQueryString();
        if ($queryString) {
            $endpoint .= '?' . $queryString;
        }
        $url = sprintf(
            'https://api.bigcommerce.com/stores/%s/%s',
            $storeHash,
            $endpoint
        );

        $httpRequest = Http::withHeaders($this->buildHeaders($installation->access_token));

        // Add body for certain HTTP methods
        if (in_array($this->request->method(), ['POST', 'PUT', 'PATCH', 'DELETE'])) {
            $httpRequest = $httpRequest->withBody($this->request->getContent(), 'application/json');
        }

        try {
            $response = $httpRequest->withOptions(['verify' => false])->send($this->request->method(), $url);

            // Throw an exception if the response status code is >= 400
            $response->throw();

            // Return the response as a fluent object
            return $response->collect();
        } catch (\Exception $e) {
            abort(500, 'BigCommerce API request failed: ' . $e->getMessage());
        }
    }

    public function proxyBigCommerceAPIRequest(string $storeHash, string $endpoint)
    {
        // For v2 endpoints, normalize by adding .json
        if (str_contains($endpoint, 'v2')) {
            $endpoint .= '.json';
        }

        $response = $this->makeBigCommerceAPIRequest($storeHash, $endpoint);
        return response($response->toJson(), 200)
            ->header('Content-Type', 'application/json');
    }

}
