<?php

namespace Mtc\ContentManager\Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Mtc\ContentManager\Models\Media;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\ContentManager\Tests\AsUser;
use Mtc\ContentManager\Tests\TestCase;

class MediaControllerTest extends TestCase
{
    use RefreshDatabase;
    use AsUser;

    public function testIndex()
    {
        $this->asUser();
        /** @var JsonResponse $response */
        $response = $this->get(route('content.media.index'));

        $this->assertEquals(200, $response->status());
        $body = $response->getData(true);
        $this->assertArrayHasKey('current_page', $body);
        $this->assertArrayHasKey('data', $body);
        $this->assertEquals(0, count($body['data']));

        Media::factory()->create(['src' => 'foo.jpg']);
        Media::factory()->create(['src' => 'baz.jpg']);
        Media::factory()->create(['src' => 'bar.png']);

        $response = $this->get(route('content.media.index'));

        $body = $response->getData(true);
        $this->assertEquals(3, count($body['data']));
    }

    public function testStoreImage()
    {
        $this->asUser();
        Storage::fake('media');

        $response = $this->postJson(route('content.media.store'), [
            'file' => UploadedFile::fake()->image('avatar.jpg', 110, 110),
        ]);

        $response->assertStatus(201);
        $data = $response->getData(true);
        $this->assertArrayHasKey('id', $data['data']);
        $this->assertArrayHasKey('path', $data['data']);
        $this->assertArrayHasKey('src', $data['data']);
        Storage::disk('media')->assertExists($data['data']['path'] . '/' . $data['data']['src']);
    }

    public function testStoreFile()
    {
        $this->asUser();
        Storage::fake('media');

        $response = $this->postJson(route('content.media.store'), [
            'file' => UploadedFile::fake()->create('test.pdf', 110, 'application/pdf'),
        ]);

        $response->assertStatus(201);
        $data = $response->getData(true);
        $this->assertArrayHasKey('id', $data['data']);
        $this->assertArrayHasKey('path', $data['data']);
        $this->assertArrayHasKey('src', $data['data']);
        $this->assertTrue(Str::contains($data['data']['src'], 'test.pdf'));
        Storage::disk('media')->assertExists($data['data']['path'] . '/' . $data['data']['src']);
    }

    public function testShow()
    {
        $this->asUser();

        $media_file = Media::factory()->create(['src' => 'avatar.jpg']);
        $response = $this->get(route('content.media.show', $media_file->id));

        $response->assertStatus(200);
        $data = $response->getData(true);
        $this->assertArrayHasKey('id', $data['data']);
        $this->assertArrayHasKey('path', $data['data']);
        $this->assertArrayHasKey('src', $data['data']);
        $this->assertTrue(Str::contains($data['data']['src'], 'avatar.jpg'));
    }

    public function testUpdate()
    {
        $this->asUser();
        $media_file = Media::factory()->create(['src' => 'avatar.jpg']);
        MediaUse::factory()->create(['media_id' => $media_file->id]);

        $response = $this->putJson(route('content.media.update', $media_file->id), [
            'alt_text' => 'lorem ipsum',
            'title' => 'foo',
            'caption' => 'baz',
            'description' => 'bar',
        ]);

        $response->assertStatus(200);
        $media_file->refresh();
        $this->assertEquals('lorem ipsum', $media_file->alt_text);
        $this->assertEquals('foo', $media_file->title);
        $this->assertEquals('baz', $media_file->caption);
        $this->assertEquals('bar', $media_file->description);
    }

    public function testResize()
    {
        $this->asUser();
        Storage::fake('media');

        $response = $this->postJson(route('content.media.store'), [
            'file' => UploadedFile::fake()->image('avatar.jpg', 110, 110),
            'model' => 'vehicle',
        ]);

        $mediaUse = MediaUse::factory()->create(['media_id' => $response->json('data.id')]);

        $response = $this->postJson(route('content.media.resize'), [
            'size' => '120x120',
            'media_use_id' => $mediaUse->id,
            'coordinates' => [
                0,
                0,
                120,
                120
            ]
        ]);

        $response->assertStatus(200);
        $this->assertIsArray($response->json());
        $this->assertArrayHasKey('url', $response->json());
    }

    public function testDestroyRemovesRecord()
    {
        $this->asUser();
        $media_file = Media::factory()->create(['src' => 'avatar.jpg']);
        $response = $this->deleteJson(route('content.media.destroy', $media_file->id));
        $response->assertStatus(200);
    }

    public function testDestroyRemovesFileOriginal()
    {
        $this->asUser();
        Storage::fake('media');

        $response = $this->postJson(route('content.media.store'), [
            'file' => UploadedFile::fake()->image('avatar.jpg', 110, 110)
        ]);
        $response->assertStatus(201);
        $data = $response->getData(true);
        Storage::disk('media')->assertExists($data['data']['path'] . '/' . $data['data']['src']);

        $response = $this->deleteJson(route('content.media.destroy', $data['data']['id']));
        $response->assertStatus(200);
        Storage::disk('media')->assertMissing($data['data']['path'] . '/' . $data['data']['src']);
    }
}
