<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\CommentRepository;
use Mtc\ContentManager\Contracts\Template;
use Mtc\ContentManager\Contracts\TemplateElement;
use Mtc\ContentManager\ElementRepository;
use Mtc\ContentManager\Facades\Template as TemplateFacade;
use Mtc\ContentManager\Http\Requests\CopyTemplateRequest;
use Mtc\ContentManager\Http\Requests\StoreTemplateRequest;
use Mtc\ContentManager\Http\Requests\UpdateTemplateRequest;
use Mtc\ContentManager\Http\Resources\TemplateListResource;
use Mtc\ContentManager\Traits\EnsuresSlug;

class TemplateController
{
    use ValidatesRequests;
    use EnsuresSlug;

    /**
     * List available template records
     *
     * @param Request $request
     * @param Template $layout
     * @return TemplateListResource
     */
    public function index(Request $request, Template $layout)
    {
        $perPage = min((int) $request->input('per_page', 15), 200);

        $results = $layout->newQuery()
            ->orderByDesc($request->input('sortBy', 'updated_at'))
            ->withInactivePageCount()
            ->withCommentCount()
            ->withActivePageCount()
            ->with('updatedBy')
            ->setSortBy($request->input('sort_by'))
            ->setFilters($request->input('filters') ?? '')
            ->paginate($perPage);

        $resourceClass = Config::get('pages.template_list_resource');
        return new $resourceClass($results);
    }

    /**
     * Store a new template record
     *
     * @param StoreTemplateRequest $request
     * @param Template $template
     * @return Template|Model
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(StoreTemplateRequest $request, Template $template): Template
    {
        return $template->newQuery()
            ->create([
                'name' => $request->input('name'),
                'slug' => $this->ensureSlug($request->input('name'), $template)
            ]);
    }

    /**
     * Show details of template
     *
     * @param Template $template
     * @param ElementRepository $repository
     * @param CommentRepository $comments
     * @return array
     */
    public function show(Template $template)
    {
        $resourceClass = Config::get('pages.template_view_resource');
        return  new $resourceClass($template);
    }

    /**
     * Update the template record
     *
     * @param Request $request
     * @param Template $template
     * @return array
     */
    public function update(UpdateTemplateRequest $request, Template $template, TemplateElement $templateElement)
    {
        TemplateFacade::update($request, $template, $templateElement);
        $template->refresh();
        $template->load('elements');
        return [
            'template' => $template,
            'template_elements' => $template->elements,
            'success' => true
        ];
    }

    /**
     * Remove Template
     *
     * @param Template $template
     * @return Response
     */
    public function destroy(Template $template)
    {

        if (TemplateFacade::canRemove($template)) {
            return response([
                'success' => $template->delete(),
            ]);
        }

        return response([
            'message' => __('validation.not_allowed_to_delete')
        ], 422);
    }

    /**
     * List unused templates (safe-to-remove)
     *
     * @param Request $request
     * @param Template $template
     */
    public function unused(Template $template): TemplateListResource
    {
        $templates = $template->newQuery()
            ->whereDoesntHave('pages')
            ->paginate();

        return new TemplateListResource($templates);
    }


    /**
     * Create a new copy of an existing global content element
     *
     * @param CopyTemplateRequest $request
     * @param int $id_to_copy
     * @return Template|Model
     */
    public function copy(CopyTemplateRequest $request, Template $template): Template
    {
        /** @var Template $copy */
        $copy = $template->newQuery()->create([
            'name' => $request->input('name'),
            'description' => $template->description,
            'seo_defaults' => $template->seo_defaults,
        ]);

        $template->elements
            ->each(fn(TemplateElement $original) => $copy->elements()->create($original->toArray()));

        return $copy;
    }
}
