<?php

namespace Mtc\ContentManager\Tests\Unit;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Validator;
use Mtc\ContentManager\Models\Page;
use Mtc\ContentManager\Rules\PageSlugDoesNotConflict;
use Mtc\ContentManager\Tests\TestCase;

class PageSlugDoesNotConflictTest extends TestCase
{
    use RefreshDatabase;

    public function testSlugAvailableNoParent()
    {
        $validator = Validator::make(
            ['slug' => 'foo'],
            ['slug' => [new PageSlugDoesNotConflict()]]
        );
        $this->assertTrue($validator->passes());
    }

    public function testSlugAvailableWithParent()
    {
        $parent = Page::factory()->create(['page_slug' => 'baz']);
        $validator = Validator::make(
            ['slug' => 'foo'],
            ['slug' => [new PageSlugDoesNotConflict(null, $parent->id)]]
        );
        $this->assertTrue($validator->passes());
    }

    public function testSlugTakenNoParent()
    {
        Page::factory()->create(['page_slug' => 'foo']);
        $validator = Validator::make(
            ['slug' => 'foo'],
            ['slug' => [new PageSlugDoesNotConflict()]]
        );
        $this->assertFalse($validator->passes());
    }

    public function testSlugTakenWithParent()
    {
        $parent = Page::factory()->create(['page_slug' => 'baz']);
        Page::factory()->create(['page_slug' => 'foo', 'parent_id' => $parent->id]);
        $validator = Validator::make(
            ['slug' => 'foo'],
            ['slug' => [new PageSlugDoesNotConflict(null, $parent->id)]]
        );
        $this->assertFalse($validator->passes());
    }

    public function testSlugExistsForDifferentParent()
    {
        $parent = Page::factory()->create(['page_slug' => 'baz']);
        $parent2 = Page::factory()->create(['page_slug' => 'bar']);
        Page::factory()->create(['page_slug' => 'foo', 'parent_id' => $parent->id]);
        $validator = Validator::make(
            ['slug' => 'foo'],
            ['slug' => [new PageSlugDoesNotConflict(null, $parent2->id)]]
        );
        $this->assertTrue($validator->passes());
    }

    public function testSlugExistsOnNoParent()
    {
        $parent = Page::factory()->create(['page_slug' => 'baz']);
        Page::factory()->create(['page_slug' => 'foo']);
        $validator = Validator::make(
            ['slug' => 'foo'],
            ['slug' => [new PageSlugDoesNotConflict(null, $parent->id)]]
        );
        $this->assertTrue($validator->passes());
    }
}
