<?php

namespace Mtc\Crm\Http\Controllers;

use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Http\Resources\Json\JsonResource;
use Mtc\Crm\Contracts\Customer;
use Mtc\Crm\Contracts\CustomerSavedSearch;
use Mtc\Crm\Http\Requests\StoreCustomerSavedSearchRequest;
use Mtc\Crm\Http\Requests\UpdateCustomerSavedSearchRequest;
use Mtc\Crm\Http\Resources\CustomerSavedSearchList;
use Mtc\Crm\Http\Resources\CustomerSavedSearchView;

class CustomerSavedSearchController
{
    use ValidatesRequests;

    public function index(Customer $customer): AnonymousResourceCollection
    {
        $savedSearches = $customer->savedSearches()
            ->withCount('filters')
            ->latest()
            ->get();

        return CustomerSavedSearchList::collection($savedSearches);
    }

    public function store(StoreCustomerSavedSearchRequest $request, Customer $customer): JsonResource
    {
        $savedSearch = $customer->savedSearches()->create([
            'name' => $request->input('name'),
            'sort_by' => $request->input('sort_by'),
            'notifications_enabled' => $request->boolean('notifications_enabled', false),
        ]);

        foreach ($request->input('filters') as $filter) {
            $savedSearch->filters()->create([
                'type' => $filter['type'],
                'value' => $filter['value'],
            ]);
        }

        return new CustomerSavedSearchView($savedSearch->load('filters'));
    }

    public function show(Customer $customer, CustomerSavedSearch $savedSearch): JsonResource
    {
        return new CustomerSavedSearchView($savedSearch->load('filters'));
    }

    public function update(
        UpdateCustomerSavedSearchRequest $request,
        Customer $customer,
        CustomerSavedSearch $savedSearch
    ): JsonResource {
        $savedSearch->update($request->only(['name', 'sort_by', 'notifications_enabled']));

        if ($request->has('filters')) {
            $savedSearch->filters()->delete();
            foreach ($request->input('filters') as $filter) {
                $savedSearch->filters()->create([
                    'type' => $filter['type'],
                    'value' => $filter['value'],
                ]);
            }
        }

        return new CustomerSavedSearchView($savedSearch->load('filters'));
    }

    public function destroy(Customer $customer, CustomerSavedSearch $savedSearch): bool
    {
        return $savedSearch->delete();
    }
}
