<?php

namespace Mtc\Crm\Tests\Unit;

use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Notification;
use Mtc\Crm\EnquiryRepository;
use Mtc\Crm\Events\UserAssignedToEnquiry;
use Mtc\Crm\Events\UserSubscribedToEnquiry;
use Mtc\Crm\Events\UserUnsubscribedFromEnquiry;
use Mtc\Crm\Facades\Enquiries;
use Mtc\Crm\Models\Enquiry;
use Mtc\Crm\Models\EnquiryStatus;
use Mtc\Crm\Models\EnquiryStatusHistory;
use Mtc\Crm\Models\EnquiryTag;
use Mtc\Crm\Notifications\EnquiryAssignedToUser;
use Mtc\Crm\Notifications\UserSubscribedToEnquiryNotification;
use Mtc\Crm\Tests\TestCase;
use Mtc\Crm\Tests\User;
use Mtc\Crm\Models\EnquiryAction;

class EnquiryRepositoryTest extends TestCase
{
    use RefreshDatabase;

    public function testHasFailedActionsScope()
    {
        // Create enquiry without failed actions
        $enquiryWithoutFailed = Enquiry::factory()->create([]);
        EnquiryAction::factory()->create([
            'enquiry_id' => $enquiryWithoutFailed->id,
            'action_name' => 'test-action',
            'processed_at' => now(),
            'failed_at' => null,
        ]);

        // Create enquiry with failed action
        $enquiryWithFailed = Enquiry::factory()->create([]);
        EnquiryAction::factory()->create([
            'enquiry_id' => $enquiryWithFailed->id,
            'action_name' => 'test-action',
            'processed_at' => null,
            'failed_at' => now(),
        ]);

        // Create enquiry with pending action (no failed_at, no processed_at)
        $enquiryWithPending = Enquiry::factory()->create([]);
        EnquiryAction::factory()->create([
            'enquiry_id' => $enquiryWithPending->id,
            'action_name' => 'test-action',
            'processed_at' => null,
            'failed_at' => null,
        ]);

        $filtered = Enquiry::query()->hasFailedActions()->get();

        $this->assertCount(1, $filtered);
        $this->assertEquals($enquiryWithFailed->id, $filtered->first()->id);
    }

    public function testGetModel()
    {
        $this->assertInstanceOf(Enquiry::class, Enquiries::getModel());
    }

    public function testSetModel()
    {
        /** @var Enquiry $enquiry */
        $enquiry = Enquiry::query()->create([]);

        Enquiries::setModel($enquiry);
        $this->assertTrue(Enquiries::getModel()->is($enquiry));
    }

    public function testLoad()
    {
        /** @var Enquiry $enquiry */
        $enquiry = Enquiry::factory()->create([]);

        $this->assertEquals(null, Enquiries::getModel()->id);
        $this->assertInstanceOf(EnquiryRepository::class, Enquiries::load($enquiry->id));
        $this->assertEquals($enquiry->id, Enquiries::getModel()->id);
    }

    public function testFind()
    {
        /** @var Enquiry $enquiry */
        $enquiry = Enquiry::factory()->create([]);

        $found_entry = Enquiries::find($enquiry->id);
        $this->assertInstanceOf(Enquiry::class, $found_entry);
        $this->assertTrue($found_entry->is($enquiry));
    }

    public function testEnquiryNotFound()
    {
        $repository = new EnquiryRepository([], new Enquiry());

        $this->expectException(ModelNotFoundException::class);
        Enquiries::find(PHP_INT_MAX);
    }

    public function testArchive()
    {
        /** @var Enquiry $enquiry */
        $enquiry = Enquiry::factory()->create([]);

        Enquiries::archive($enquiry->id);
        $enquiry_archived = Enquiry::query()
            ->onlyTrashed()
            ->where('id', $enquiry->id)
            ->exists();

        $this->assertTrue($enquiry_archived);

        $enquiry2 = Enquiry::factory()->create([]);
        $enquiry3 = Enquiry::factory()->create([]);

        Enquiries::archive([$enquiry2->id, $enquiry3->id]);

        $enquiry2_archived = Enquiry::query()
            ->onlyTrashed()
            ->where('id', $enquiry2->id)
            ->exists();

        $this->assertTrue($enquiry2_archived);

        $enquiry3_archived = Enquiry::query()
            ->onlyTrashed()
            ->where('id', $enquiry3->id)
            ->exists();

        $this->assertTrue($enquiry3_archived);
    }

    public function testSetStatus()
    {
        $enquiry = Enquiry::factory()->create(['status_id' => 100]);
        $status = EnquiryStatus::query()->create(['name' => 'new-status']);
        Enquiries::load($enquiry->id)->setStatus($status->id);

        $this->assertEquals($status->id, Enquiries::getModel()->status_id);

        $history_tracked = EnquiryStatusHistory::query()
            ->where('enquiry_id', $enquiry->id)
            ->where('status_id', $status->id)
            ->exists();
        $this->assertTrue($history_tracked);
    }

    public function testAssign()
    {
        Notification::fake();
        Event::fake();
        /** @var User $user */
        $user = User::query()->create(['name' => 'john', 'email' => 'hello@example.com', 'password' => '']);

        /** @var Enquiry $enquiry */
        $enquiry = Enquiry::factory()->create([]);

        Notification::assertNothingSent();

        Enquiries::load($enquiry->id)->assign($user, 0);
        Event::assertDispatched(UserAssignedToEnquiry::class);

        $this->assertEquals($user->id, Enquiries::getModel()->assigned_user_id);
        Notification::assertSentTo($user, EnquiryAssignedToUser::class);
    }

    public function testAssignToSelf()
    {
        Notification::fake();
        /** @var User $user */
        $user = User::query()->create(['name' => 'john', 'email' => 'hello@example.com', 'password' => '']);

        /** @var Enquiry $enquiry */
        $enquiry = Enquiry::factory()->create([]);

        Enquiries::load($enquiry->id)->assign($user, $user->id);
        Notification::assertNothingSent();
    }

    public function testAddTag()
    {
        $enquiry = Enquiry::factory()->create([]);
        $tag = EnquiryTag::query()->create(['name' => 'hello-world']);

        Enquiries::addTag($enquiry->id, $tag->id);
        $data = Enquiries::find($enquiry->id);

        $this->assertEquals(1, $data->tags()->count());
        $this->assertTrue($data->tags()->first()->is($tag));
    }

    public function testRemoveTag()
    {
        $enquiry = Enquiry::factory()->create([]);
        $tag = EnquiryTag::query()->create(['name' => 'hello-world']);

        Enquiries::addTag($enquiry->id, $tag->id);
        $data = Enquiries::find($enquiry->id);
        $this->assertEquals(1, $data->tags()->count());

        // remove non-existing id doesn't remove unnecessary one
        Enquiries::removeTag($enquiry->id, PHP_INT_MAX);
        $this->assertEquals(1, $data->tags()->count());

        Enquiries::removeTag($enquiry->id, $tag->id);
        $this->assertEquals(0, $data->tags()->count());
    }

    public function testSubscribe()
    {
        Notification::fake();
        Event::fake();

        /** @var User $user */
        $user = User::query()->create(['name' => 'john', 'email' => 'hello@example.com', 'password' => '']);

        /** @var Enquiry $enquiry */
        $enquiry = Enquiry::factory()->create([]);

        Notification::assertNothingSent();

        Enquiries::load($enquiry->id)->subscribe($user);
        Event::assertDispatched(UserSubscribedToEnquiry::class);

        $this->assertTrue(Enquiries::getModel()->subscribers()->where('user_id', $user->id)->exists());
        Notification::assertSentTo($user, UserSubscribedToEnquiryNotification::class);
    }

    public function testUnsubscribe()
    {
        Notification::fake();
        Event::fake();
        /** @var User $user */
        $user = User::query()->create(['name' => 'john', 'email' => 'hello@example.com', 'password' => '']);

        /** @var Enquiry $enquiry */
        $enquiry = Enquiry::factory()->create([]);

        Enquiries::load($enquiry->id)->subscribe($user);
        $this->assertTrue(Enquiries::getModel()->subscribers()->where('user_id', $user->id)->exists());

        Enquiries::unsubscribe($user);
        Event::assertDispatched(UserUnsubscribedFromEnquiry::class);

        $this->assertFalse(Enquiries::getModel()->subscribers()->where('user_id', $user->id)->exists());
    }
}
