<?php

namespace Mtc\Crm;

use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Auth\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Notification;
use Mtc\Crm\Events\UserAssignedToEnquiry;
use Mtc\Crm\Events\UserSubscribedToEnquiry;
use Mtc\Crm\Events\UserUnsubscribedFromEnquiry;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\Crm\Models\EnquiryMessage;
use Mtc\Crm\Notifications\EnquiryAssignedToUser;
use Mtc\Crm\Notifications\EnquiryResponseNotification;
use Mtc\Crm\Notifications\UserSubscribedToEnquiryNotification;

class EnquiryRepository
{
    public function __construct(
        protected readonly array $config,
        protected EnquiryModel $enquiry
    ) {
        //
    }

    /**
     * Load enquiry and return self to allow chaining
     *
     * @param int $id
     * @param bool $withTrashed Include soft-deleted records
     * @return $this
     */
    public function load(int $id, bool $withTrashed = false): self
    {
        $this->enquiry = $this->enquiry->newQuery()
            ->when($withTrashed, fn($query) => $query->withTrashed())
            ->findOrFail($id);
        return $this;
    }

    /**
     * find and return enquiry
     * @param int $id
     * @param bool $withTrashed Include soft-deleted records
     * @return EnquiryModel|Model
     */
    public function find(int $id, bool $withTrashed = true): EnquiryModel
    {
        return $this->enquiry = $this->enquiry->newQuery()
            ->when($withTrashed, fn($query) => $query->withTrashed())
            ->findOrFail($id);
    }

    /**
     * Restore a soft-deleted enquiry
     *
     * @param int $id
     * @return EnquiryModel
     */
    public function restore(int $id): EnquiryModel
    {
        $this->enquiry = $this->enquiry->newQuery()
            ->withTrashed()
            ->findOrFail($id);
        $this->enquiry->restore();
        return $this->enquiry;
    }

    /**
     * Get the currently loaded enquiry model
     *
     * @return EnquiryModel
     */
    public function getModel(): EnquiryModel
    {
        return $this->enquiry;
    }

    /**
     * Set the enquiry object to repository
     *
     * @param EnquiryModel $enquiry
     * @return $this
     */
    public function setModel(EnquiryModel $enquiry): self
    {
        $this->enquiry = $enquiry;
        return $this;
    }


    /**
     * Add a new message to Enquiry
     *
     * @param string $message
     * @param int|null $new_status_id
     * @param bool $from_customer
     * @param int|null $user_id
     * @param bool $silent
     * @return EnquiryModel
     */
    public function addMessage(
        string $message,
        int|null $new_status_id,
        bool $from_customer = false,
        int $user_id = null,
        bool $silent = false
    ): EnquiryModel {
        /** @var EnquiryMessage $enquiry_message */
        $enquiry_message = $this->enquiry
            ->messages()
            ->create([
                'new_status_id' => $new_status_id,
                'message' => $message,
                'from_customer' => $from_customer,
                'user_id' => $user_id,
            ]);

        if ($new_status_id !== null) {
            $this->setStatus($new_status_id, $from_customer, $user_id);
        }

        if ($silent === false) {
            $this->notifyNewEnquiryMessage($enquiry_message, $from_customer, $user_id);
        }

        $this->enquiry->load('messages');
        return $this->enquiry;
    }

    /**
     * Archive enquiries
     *
     * @param int|array $ids
     * @return bool
     */
    public function archive(int|array $ids): bool
    {
        return $this->enquiry->newQuery()
            ->whereIn('id', (array)$ids)
            ->delete();
    }

    /**
     * Update status of enquiry
     *
     * @param int $new_status_id
     * @param bool $from_customer
     * @param int|null $user_id
     */
    public function setStatus(int $new_status_id, bool $from_customer = false, int $user_id = null)
    {
        $this->enquiry->status_id = $new_status_id;
        $this->enquiry->save();
        $this->enquiry->statusHistory()
            ->create([
                'status_id' => $new_status_id,
                'from_customer' => $from_customer,
                'user_id' => $user_id
            ]);
    }

    public function showList(Request $request): LengthAwarePaginator
    {
        return $this->enquiry->newQuery()
            ->with([
                'assignee'
            ])
            ->archived($request->input('archive', 0))
            ->setSortBy($request->input('sort_by', 'id_desc'))
            ->setFilters($request->input('filters', []))
            ->paginate()
            ->appends($request->input());
    }

    /**
     * Add tag to enquiry
     *
     * @param int $enquiry_id
     * @param int $tag_id
     * @return bool
     */
    public function addTag(int $enquiry_id, int $tag_id): bool
    {
        $this->load($enquiry_id);
        $this->enquiry->tags()->attach($tag_id);
        return true;
    }

    /**
     * Remove tag from enquiry
     *
     * @param int $enquiry_id
     * @param int $tag_id
     * @return bool
     */
    public function removeTag(int $enquiry_id, int $tag_id): bool
    {
        $this->load($enquiry_id);
        $this->enquiry->tags()->detach($tag_id);
        return true;
    }

    /**
     * Assign user to an enquiry
     *
     * @param User $user
     * @param int $current_user_id
     */
    public function assign(User $user, int $current_user_id)
    {
        $this->enquiry->assigned_user_id = $user->id;
        $this->enquiry->save();
        Event::dispatch(new UserAssignedToEnquiry($this->enquiry, $user->id, $current_user_id));

        if ($current_user_id !== $user->id) {
            Notification::send($user, new EnquiryAssignedToUser($this->enquiry, $user->id));
        }
        return $this;
    }

    /**
     * Subscribe user to an enquiry
     *
     * @param User $subscriber
     */
    public function subscribe(User $subscriber)
    {
        $this->enquiry->subscribers()->attach($subscriber->id);
        Event::dispatch(new UserSubscribedToEnquiry($this->enquiry, $subscriber->id, Auth::id() ?? 0));

        if (Auth::id() !== $subscriber->id) {
            Notification::send($subscriber, new UserSubscribedToEnquiryNotification($this->enquiry));
        }
        return $this;
    }

    /**
     * Unsubscribe user from an enquiry
     *
     * @param User $subscriber
     */
    public function unsubscribe(User $subscriber)
    {
        $this->enquiry->subscribers()->detach($subscriber->id);
        Event::dispatch(new UserUnsubscribedFromEnquiry($this->enquiry, $subscriber->id, Auth::id() ?? 0));
        return $this;
    }

    public function makeAction(string $action)
    {
        $handler = config('crm.form_actions.' . $action);
        if (class_exists($handler)) {
            return App::make($handler);
        }
        throw new Exception('Unknown Action ' . $action);
    }

    /**
     * Notify relevant people on new enquiry message
     *
     * @param EnquiryMessage $enquiry_message
     * @param bool $from_customer
     * @param int $user_id
     */
    protected function notifyNewEnquiryMessage(EnquiryMessage $enquiry_message, bool $from_customer, int $user_id)
    {
        // Subscribers
        $subscribers = $this->enquiry->subscribers()->get()
            ->reject(fn($entry) => $from_customer === false && $entry->id === $user_id);
        Notification::send($subscribers, new EnquiryResponseNotification($enquiry_message));

        // Customer
        if ($from_customer === false && !empty($this->enquiry->email)) {
            // TODO: refactor notification to mail as it cannot be tested
//            Mail::route('mail', $this->enquiry->email)
//                ->notify(new EnquiryResponseNotificationToCustomer($enquiry_message));
        }
    }
}
