# `mtcmedia/gp-addresses`

*Any issues/questions regarding this package, feel free to reach out to `jack.jefferson@mtc.co.uk`*

The GP Addresses package contains processes for pulling GP Addresses and contact information from the [NHS ODS](https://www.odsdatasearchandexport.nhs.uk/referenceDataCatalogue/index.html).

## Table of Contents

- [Installation](#installation)
- [Setting up Jobs](#setting-up-the-jobs)
- [Commands](#commands)
- [Integration Guide](#integration-guide)
  - [Backend Integration](#backend-integration)
  - [Frontend Integration](#frontend-integration)
- [API Endpoints](#api-endpoints)
- [Data Structure](#data-structure)

---

# Installation

To include the package onto a pharmacy site, run the following command:

```bash
composer require "mtcmedia/gp-addresses:^0.9.0"
```

The package should be registered by Laravel's package discovery. If the package is not auto-discovered, you may need to manually add the `GpAddressServiceProvider` to `config/app.php`.

After installing, run the migrations:

```bash
# Load new migrations
php artisan migrate

# Check that routes have been loaded correctly
php artisan route:list | grep "gp-addresses"
```

---

# Setting up the Jobs

If the pharmacy site you are working on does not contain a Job queue by default then you may have to setup jobs. More documentation about Laravel Job Queues can be found [here](https://laravel.com/docs/11.x/queues).

## Database queue driver

To setup the database queue driver:

```bash
# Build the jobs table
php artisan make:queue-table

# Build the failed jobs table
php artisan make:queue-failed-table
```

## Scheduling the queue to run

Include this at the top of `schedule()` in `app/Console/Kernel.php` to start the job queue:

```php
if (config('queue.default') === 'database') {
    $schedule->command('queue:work --timeout=3600 --stop-when-empty')
        ->everyMinute()
        ->withoutOverlapping();
}
```

## Manually running the queue

If you need to run the queue manually:

```bash
php artisan queue:work
```

---

# Commands

Below are the commands available for managing GP address data.

## Pulling Data

This command checks the current data for things like dates or total counts. If it finds any issues, it does a full data pull. Otherwise, it just syncs the changes. Full pulls can take a while, so this is mainly for initial setup or large updates.

```bash
php artisan gp:pull
```

## Syncing Data

Looks for a last date and then runs the command to fetch data since that date.

```bash
php artisan gp:sync
```

## Deleting Data

Checks to see if any records have been deleted from the NHS ODS API.

```bash
php artisan gp:delete
```

More details [here](https://digital.nhs.uk/services/organisation-data-service/organisation-data-service-apis/technical-guidance-ord/notifications-endpoint).

---

# Integration Guide

This section explains how to integrate the GpAddresses package into your pharmacy site, replacing the legacy `DoctorSurgery` module.

## Backend Integration

### Using the NhsOrganisation Model

The package provides the `NhsOrganisation` model with related `address` and `contact` relationships:

```php
use Mtc\GpAddresses\Models\NhsOrganisation;

// Search for GP practices by postcode
$practices = NhsOrganisation::query()
    ->with(['address', 'contact'])
    ->where('status', 'Active')
    ->whereHas('address', function ($query) use ($postcode) {
        $query->where('postcode', 'LIKE', "{$postcode}%");
    })
    ->limit(100)
    ->get();

// Access related data
foreach ($practices as $practice) {
    $name = $practice->name;
    $orgId = $practice->org_id;
    $addressLine1 = $practice->address?->address_line_1;
    $phone = $practice->contact?->telephone_number;
}
```

### Mapping Legacy Data

When migrating from the legacy `DoctorSurgery` format, map the data to the new structure:

```php
// Legacy doctor_surgery data stored in member record
$doctor_surgery = json_decode($member->doctor_surgery);

// Map to GpAddresses format
$gp_practice = (object) [
    'manual_gp_practice' => $doctor_surgery->manual_doctor_surgery ?? 0,
    'id' => $doctor_surgery->id ?? 0,
    'name' => $doctor_surgery->practice_name ?? '',
    'org_id' => $doctor_surgery->ods_code ?? '',
    'address_line_1' => $doctor_surgery->address_1 ?? '',
    'address_line_2' => $doctor_surgery->address_2 ?? '',
    'town' => $doctor_surgery->city ?? '',
    'county' => $doctor_surgery->address_3 ?? '',
    'postcode' => $doctor_surgery->postcode ?? '',
    'telephone_number' => $doctor_surgery->phone ?? '',
];
```

### Controller Example

```php
use Mtc\GpAddresses\Models\NhsOrganisation;

class HealthProfileController extends Controller
{
    private function get_gp_practice()
    {
        $doctor_surgery = !empty($this->member->doctor_surgery)
            ? json_decode($this->member->doctor_surgery)
            : null;

        return (object) [
            'manual_gp_practice' => $doctor_surgery->manual_doctor_surgery ?? $doctor_surgery->manual_gp_practice ?? 0,
            'id' => $doctor_surgery->id ?? 0,
            'name' => $doctor_surgery->practice_name ?? $doctor_surgery->name ?? '',
            'org_id' => $doctor_surgery->ods_code ?? $doctor_surgery->org_id ?? '',
            'address_line_1' => $doctor_surgery->address_1 ?? $doctor_surgery->address_line_1 ?? '',
            'address_line_2' => $doctor_surgery->address_2 ?? $doctor_surgery->address_line_2 ?? '',
            'town' => $doctor_surgery->city ?? $doctor_surgery->town ?? '',
            'county' => $doctor_surgery->address_3 ?? $doctor_surgery->county ?? '',
            'postcode' => $doctor_surgery->postcode ?? '',
            'telephone_number' => $doctor_surgery->phone ?? $doctor_surgery->telephone_number ?? '',
        ];
    }
}
```

---

## Frontend Integration

### Vite Configuration

Add the `@mtcmedia` alias to your `vite.config.js`:

```js
resolve: {
    alias: {
        '@site': resolve(__dirname, 'sites/default'),
        '@components': resolve(__dirname, 'sites/default/js/components'),
        '@mtcmedia': resolve(__dirname, 'vendor/mtcmedia/'),
    },
    extensions: ['.mjs', '.js', '.ts', '.jsx', '.tsx', '.json', '.vue'],
},
```

### Registering Vue Components

Import and register the components in `main-app.js`:

```js
// GpAddresses package - GP practice search component
import GpPracticeSearch from '@mtcmedia/gp-addresses/resources/js/GpPracticeSearch.vue';
Vue.component("gp-practice-search", GpPracticeSearch);

// Optional: Import the base JS if needed
// import "@mtcmedia/gp-addresses/resources/js/mtc_gp_addresses.js";
```

### Available Vue Components

| Component | Description |
|-----------|-------------|
| `GpPracticeSearch` | Search component for finding GP practices by name/address |
| `GpPracticeSelection` | Full selection component with search and manual entry |
| `GpAddressesSearch` | Admin search component |
| `GpAddressesTable` | Admin table display component |

### Using in Twig Templates

#### Health Profile

```twig
<gp-practice-search
    search-url="/api/gp-addresses/search"
    search-placeholder="Search GP practice name or address"
    :user-practice="{{ gp_practice|json_encode|default('{}') }}"
></gp-practice-search>
```

#### Assessment Form

```twig
<assessment-form
    :set-sections="{{ question_sections | json_encode }}"
    :user-practice="{{ gp_practice | json_encode }}"
    :user-health-profile="{{ health_profile | json_encode }}"
></assessment-form>
```

### Vuex Store Integration

If using Vuex, add the practice state:

```js
// store.js
export default new Vuex.Store({
    state: {
        user: {
            practice: {
                name: '',
                address_line_1: '',
                address_line_2: '',
                town: '',
                county: '',
                postcode: '',
                telephone_number: '',
                id: '',
                org_id: '',
                manual_gp_practice: 0
            },
        },
    },
    mutations: {
        SET_USER_PRACTICE: (state, payload) => {
            state.user.practice = { ...state.user.practice, ...payload }
        },
    },
    actions: {
        setUserPractice: (context, payload) => {
            context.commit('SET_USER_PRACTICE', payload);
        },
    },
    getters: {
        getUserPractice: state => {
            return state.user.practice;
        },
    },
});
```

---

# API Endpoints

The package provides the following API endpoints:

| Method | Endpoint | Description |
|--------|----------|-------------|
| GET | `/api/gp-addresses/search` | Search GP practices by name or address |
| GET | `/api/gp-addresses/search-by-postcode` | Search GP practices by postcode |
| GET | `/api/gp-addresses/{id}` | Get a specific GP practice by ID |

### Example API Response

```json
{
    "data": [
        {
            "id": 1,
            "name": "Example Medical Practice",
            "org_id": "A12345",
            "status": "Active",
            "address_line_1": "123 High Street",
            "address_line_2": "",
            "town": "London",
            "county": "Greater London",
            "postcode": "SW1A 1AA",
            "telephone_number": "020 1234 5678"
        }
    ]
}
```

---

# Data Structure

## GpAddresses Format

| Field | Type | Description |
|-------|------|-------------|
| `id` | integer | Internal database ID |
| `org_id` | string | NHS ODS Organisation ID |
| `name` | string | Practice name |
| `address_line_1` | string | First line of address |
| `address_line_2` | string | Second line of address |
| `town` | string | Town/City |
| `county` | string | County |
| `postcode` | string | Postcode |
| `telephone_number` | string | Contact phone number |
| `manual_gp_practice` | boolean | Whether manually entered |

## Legacy DoctorSurgery Format (for reference)

| Legacy Field | New Field |
|--------------|-----------|
| `practice_name` | `name` |
| `address_1` | `address_line_1` |
| `address_2` | `address_line_2` |
| `city` | `town` |
| `address_3` | `county` |
| `phone` | `telephone_number` |
| `ods_code` | `org_id` |
| `manual_doctor_surgery` | `manual_gp_practice` |
