<?php

namespace Mtc\GpAddresses\Console\Commands;

use Carbon\Carbon;
use Illuminate\Support\Facades\Http;
use Mtc\GpAddresses\Models\NhsOrganisation;
use Mtc\GpAddresses\Services\GpAddressesService;
use Illuminate\Console\Command;

class PullAddressesCommand extends Command
{
    protected $signature = 'gp:pull';

    protected $description = 'Pull Organisations from NHS ODS API and dispatch jobs to fetch their addresses.';

    private $service;

    /**
     * Fetch the totla amount of organisations stored by the NHS ODS API.
     *
     * @return int $apiCount The total amount of records found in the 'x-total-count' header.
     * */
    private function nhsOrganisationsCount() {
        $url = config('gpaddresses.fetch') . '&Limit=1';

        $api = Http::get($url);

        $apiCount = $api->header('x-total-count');

        return $apiCount;
    }

    /**
     * Fetch the total count of all organisations stored in the database.
     *
     * @return int $totalModels The total amount of records found in the database.
     * */
    private function dbOrganisationsCount() {
        $totalModels = NhsOrganisation::all()->count();
        return $totalModels;
    }

    /**
     * Helps determine if we need to run a full fetch of all the data from the database.
     * Reasons to avoid doing this is because it takes a very long time to populate the database.
     *
     * @return bool Returns true if there is no date set or there is a mismatch in the data.
     * */
    private function doFullFetch() {
        $date = $this->service->commandLastRan();
        $apiCount = $this->nhsOrganisationsCount();
        $dbCount = $this->dbOrganisationsCount();

        return !$date || $dbCount == 0 || $dbCount != $apiCount ;
    }

    public function handle(): void
    {
        $this->service = app(GpAddressesService::class);

        // If there is no date, no database records or our database records don't match the NHS ODS API x-total-count
        if ($this->doFullFetch()) {
            $this->info('There was no previous run or there are missing records.');
            $this->service->fetchAndStore();
            $this->info('Finished setting up Jobs... awaiting dispatch.');
        } else {
            $date = $this->service->commandLastRan();
            $this->info("Syncing data from $date.");
            $this->service->sync($date);
        }
    }
}
