<?php

namespace Mtc\GpAddresses\Http\Controllers;

use Mtc\GpAddresses\Models\GpAddress;
use Mtc\GpAddresses\Models\NhsOrganisation;
use Illuminate\Http\Request;
use Mtc\GpAddresses\Http\Resources\NhsOrganisationResource;
use Mtc\GpAddresses\Http\Resources\GpPracticeResource;

class GpAddressesController extends Controller
{
    public function index()
    {
        return template('gpaddresses/index.twig');
    }

    public function fetch(Request $request)
    {
        $selectedCountry = $request->country;
        $selectedTown = $request->town;
        $selectedCounty = $request->county;
        $onlyActive = $request->onlyActive;

        $countries = GpAddress::query()
            ->distinct()
            ->orderBy('country')
            ->pluck('country');

        $counties = GpAddress::query()
            ->when($selectedCountry, function ($query) use ($selectedCountry) {
                $query->where('country', $selectedCountry);
            })
            ->whereNotNull('county')
            ->select('county')
            ->distinct()
            ->orderBy('county')
            ->pluck('county');

        $towns = GpAddress::query()
            ->when($selectedCounty, function ($query) use ($selectedCounty) {
                $query->where('county', $selectedCounty);
            })
            ->select('town')
            ->distinct()
            ->orderBy('town')
            ->pluck('town');

        // Get relationships
        $organisations = NhsOrganisation::with(['address', 'contact']);

        // Toggle active
        $organisations = $organisations->when($onlyActive, function ($query) {
            $query->where('status', 'Active');
        });

        // Filter by country and town
        $organisations = $organisations->whereHas('address', function ($query) use ($selectedCountry, $selectedTown) {
            $query->when($selectedCountry, function ($query) use ($selectedCountry) {
                $query->where('country', $selectedCountry);
            })
                ->when($selectedTown, function ($query) use ($selectedTown) {
                    $query->where('town', $selectedTown);
                });
        });

        // Handle search
        $organisations = $organisations->when($request->filled('search'), function ($query) use ($request) {
            $search = $request->search;

            // Search by org name, and address line
            $query->where(function ($query) use ($search) {
                $query->where('name', 'LIKE', "%{$search}%")
                    ->orWhereHas('address', function ($query) use ($search) {
                        $query->where('address_line_1', 'LIKE', "%{$search}%")
                            ->orWhere('address_line_2', 'LIKE', "%{$search}%")
                            ->orWhere('postcode', 'LIKE', "%{$search}");
                    });
            });
        });

        // Pagination
        $organisations = $organisations->paginate(20);

        return response()->json([
            'organisations' => NhsOrganisationResource::collection($organisations),
            'countries' => $countries,
            'towns' => $towns,
            'counties' => $counties,
            'current_page' => $organisations->currentPage(),
            'last_page' => $organisations->lastPage()
        ]);
    }

    /**
     * Customer-facing GP practice search by term (name, address, postcode)
     */
    public function search(Request $request)
    {
        $term = $request->input('term');

        if (empty($term)) {
            return response()->json([
                'status' => 'ok',
                'gp_practices' => [],
            ]);
        }

        $gp_practices = NhsOrganisation::query()
            ->with(['address', 'contact'])
            ->where('status', 'Active')
            ->where(function ($query) use ($term) {
                $query->where('name', 'like', "%{$term}%")
                    ->orWhereHas('address', function ($q) use ($term) {
                        $q->where('address_line_1', 'like', "%{$term}%")
                            ->orWhere('address_line_2', 'like', "%{$term}%")
                            ->orWhere('town', 'like', "%{$term}%")
                            ->orWhere('postcode', 'like', "%{$term}%");
                    });
            })
            ->take(10)
            ->get();

        return response()->json([
            'status' => 'ok',
            'gp_practices' => GpPracticeResource::collection($gp_practices),
        ]);
    }

    /**
     * Get a single GP practice by ID for display purposes (e.g., assessment answers)
     */
    public function show(int $id)
    {
        $practice = NhsOrganisation::with(['address', 'contact'])->find($id);

        if (!$practice) {
            return response()->json([
                'status' => 'error',
                'message' => 'GP practice not found',
            ], 404);
        }

        return response()->json([
            'status' => 'ok',
            'gp_practice' => new GpPracticeResource($practice),
        ]);
    }

    /**
     * Customer-facing GP practice search by postcode (for Health Profile / NHS plugin)
     */
    public function searchByPostcode(Request $request)
    {
        $postcode = $request->input('postcode');

        if (empty($postcode)) {
            return response()->json([
                'status' => 'ok',
                'gp_practices' => [],
            ]);
        }

        $resultsLimit = 100;

        // Extract postcode prefix for broader matching (e.g., "SW1A 1AA" -> "SW1A")
        $postcodePrefix = strtoupper(preg_replace('/\s+\d[A-Z]{2}$/i', '', trim($postcode)));

        $gp_practices = NhsOrganisation::query()
            ->with(['address', 'contact'])
            ->where('status', 'Active')
            ->whereHas('address', function ($query) use ($postcodePrefix, $postcode) {
                $query->where('postcode', 'LIKE', "{$postcodePrefix}%")
                    ->orWhere('town', 'LIKE', "%{$postcode}%");
            })
            ->limit($resultsLimit)
            ->get();

        if ($gp_practices->isEmpty()) {
            return response()->json([
                'status' => 'error',
                'errors' => "Couldn't find GP practices for the provided location.",
                'gp_practices' => [],
            ]);
        }

        // Format for map display (Health Profile compatibility)
        $formattedPractices = $gp_practices->map(function ($org) {
            return [
                'id' => $org->id,
                'title' => $org->name,
                'name' => $org->name,
                'org_id' => $org->org_id,
                'address_line_1' => $org->address?->address_line_1 ?? '',
                'address_line_2' => $org->address?->address_line_2 ?? '',
                'town' => $org->address?->town ?? '',
                'county' => $org->address?->county ?? '',
                'postcode' => $org->address?->postcode ?? '',
                'telephone_number' => $org->contact?->telephone_number ?? '',
                'description' => ucwords(strtolower($org->address?->address_line_1 ?? '')) . '<br />' .
                                ucwords(strtolower($org->address?->address_line_2 ?? '')) . '<br />' .
                                ($org->address?->postcode ?? ''),
                'coords' => '', // GpAddresses doesn't have coordinates
                'status' => $org->status,
            ];
        });

        return response()->json([
            'status' => 'ok',
            'gp_practices' => $formattedPractices,
        ]);
    }
}
