<?php

namespace Mtc\GpAddresses\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\GpAddresses\Models\GpContactInformation;
use Mtc\GpAddresses\Models\NhsOrganisation;
use Mtc\GpAddresses\Models\GpAddress;

class FetchGpInformation implements ShouldQueue
{
    private $links;

    use Queueable, Dispatchable;

    public function __construct($links)
    {
        $this->links = $links;
    }

    private function fetchAddresses($data)
    {
        $addresses = [];

        foreach ( $data as $item ) {
            $ext = $item['Organisation']['OrgId']['extension'] ?? null;
            $location = $item['Organisation']['GeoLoc']['Location'] ?? null;

            if (!$ext || !$location ) continue;

            $orgMap = NhsOrganisation::pluck('id', 'org_id')->toArray();

            $org = $orgMap[$ext] ?? null;
            if (!$org) continue;

            $addresses[] = [
                    'nhs_organisation_id' => $org,
                    'address_line_1'      => $location['AddrLn1'] ?? null,
                    'postcode'            => $location['PostCode'] ?? null,
                    'uprn'                => $location['UPRN'] ?? null,
                    'address_line_2'      => $location['AddrLn2'] ?? null,
                    'town'                => $location['Town'] ?? null,
                    'county'              => $location['County'] ?? null,
                    'country'             => $location['Country'] ?? null,
            ];
        }

        if (!empty($addresses)) {
            GpAddress::upsert(
                $addresses,
                ['nhs_organisation_id', 'address_line_1', 'postcode'],
                ['uprn', 'address_line_2', 'town', 'county', 'country']
            );
        }
    }

    private function fetchContacts(array $data)
    {
        $contacts = [];

        foreach ($data as $item) {
            $ext = $item['Organisation']['OrgId']['extension'] ?? null;
            $contact = $item['Organisation']['Contacts']['Contact'] ?? [];

            if (!$ext || empty($contact)) continue;

            // Find the first telephone contact
            $telephone = $contact[0]['value'];

            if (!$telephone) continue;

            $orgMap = NhsOrganisation::pluck('id', 'org_id')->toArray();

            $orgId = $orgMap[$ext] ?? null;
            if (!$orgId) continue;

            $contacts[] = [
                'nhs_organisation_id' => $orgId,
                'telephone_number'    => $telephone,
            ];
        }

        if ($contacts) {
            GpContactInformation::upsert(
                $contacts,
                ['nhs_organisation_id'], // unique key
                ['telephone_number']     // fields to update
            );
        }
    }

    public function handle(): void
    {
        // Fire all requests concurrently
        $responses = Http::pool(function ($pool) {
            foreach ($this->links as $link) {
                $pool->as($link)->get($link);
            }
        });

        $pages = [];

        foreach ($this->links as $link) {
            /** @var \Illuminate\Http\Client\Response|null $response */
            $response = $responses[$link] ?? null;

            if (!$response) {
                Log::warning("No response for $link");
                continue;
            }

            // Handle retry logic on a per-response basis
            $attempts = 1;
            while ($response->status() === 429 && $attempts < 5) {
                sleep(2);
                $response = Http::get($link);
                $attempts++;
            }

            if (!$response->ok()) {
                Log::warning("Failed fetching $link after $attempts attempts, status: {$response->status()}");
                continue;
            }

            // Flatten items properly
            $pages[] = $response->json();
        }

        // Process flattened items once
        $this->fetchAddresses($pages);
        $this->fetchContacts($pages);
    }
}
