<?php

namespace Mtc\Members\Http\Controllers\Auth;

use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Auth\Passwords\DatabaseTokenRepository;
use Illuminate\Auth\Passwords\TokenRepositoryInterface;
use Illuminate\Contracts\Auth\CanResetPassword as CanResetPasswordContract;
use Illuminate\Foundation\Application;
use Illuminate\Foundation\Auth\ResetsPasswords;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Str;
use Mtc\Members\Contracts\MemberModel;

/**
 * Class ResetPasswordController
 *
 * @package Mtc\Members\Http\Controllers\Auth
 */
class ResetPasswordController extends Controller
{
    use ResetsPasswords;

    /**
     * @var Application
     */
    protected $app;

    /**
     * @var TokenRepositoryInterface
     */
    protected $tokens;

    /**
     * Get the password reset validation rules.
     *
     * @return array
     */
    protected function rules()
    {
        $strength_rule = App::make(config('members.password_strength'));
        return [
            'token' => 'required',
            'email' => 'required|email',
            'password' => [
                'required',
                'confirmed',
                $strength_rule
            ]
        ];
    }

    /**
     * Render the password reset form
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show(Request $request)
    {
        $this->page_meta['title'] = 'Reset Password | ' . config('app.name');
        $this->page_meta['page_title'] = 'Reset password';
        if (!$request->hasValidSignature()) {
            return template('members/reset_password.twig', [
                'token_error' => 'The password token is not valid',
                'page_meta' => $this->page_meta,
            ]);
        }

        return template('members/reset_password.twig', [
            'page_meta' => $this->page_meta,
            'token' => $request->token,
            'email' => $request->email
        ]);
    }

    /**
     * Reset the given user's password.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    public function reset(Request $request, MemberModel $member)
    {
        $this->validate($request, $this->rules(), $this->validationErrorMessages());

        $member = $member->newQuery()->where('email', $request->input('email'))->first();
        if (!$member) {
            session()->flash('errors', [ 'password' => 'Admin ccount was not found']);
            return redirect()->back();
        }

        $member = $this->validateReset($request, $member);
        if (! $member instanceof CanResetPasswordContract) {
            return $member;
        }

        $password = $request->password;
        $this->resetPassword($member, $password);
        $this->tokens->delete($member);

        return redirect()->to(route('members.login'));
    }

    /**
     * Validate the token is correct
     * @param $credentials
     * @param $user
     * @return mixed
     */
    protected function validateReset($credentials, $user)
    {
        $this->app = app();
        $config = $this->app['config']["auth.passwords.members"];
        $this->tokens = $this->createTokenRepository($config);

        if (! $this->tokens->exists($user, $credentials['token'])) {
            return Password::INVALID_TOKEN;
        }

        return $user;
    }

    /**
     * Create token repository
     *
     * @param array $config
     * @return DatabaseTokenRepository
     */
    protected function createTokenRepository(array $config)
    {
        $key = $this->app['config']['app.key'];

        if (Str::startsWith($key, 'base64:')) {
            $key = base64_decode(substr($key, 7));
        }

        $connection = $config['connection'] ?? null;

        return new DatabaseTokenRepository(
            $this->app['db']->connection($connection),
            $this->app['hash'],
            $config['table'],
            $key,
            $config['expire']
        );
    }
    /**
     * Reset the given user's password.
     *
     * @param  \Illuminate\Contracts\Auth\CanResetPassword  $user
     * @param  string  $password
     * @return void
     */
    protected function resetPassword($member, $password)
    {
        $member->password = Hash::make($password);
        $member->save();

        event(new PasswordReset($member));
    }
}
