<?php

namespace Mtc\MercuryDataModels\Finance\Jobs;

use App\Facades\Settings;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceProvider;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceRequestData;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResult;
use Mtc\MercuryDataModels\Finance\FinanceServiceHelper;
use Mtc\MercuryDataModels\Vehicle;

class RunFinanceFetchForVehicle implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * Create a new job instance.
     */
    public function __construct(
        private readonly Vehicle $vehicle,
        private readonly int $term,
        private readonly int $mileage,
        private readonly float $depositPercentage,
        private readonly ?string $creditRating = null,
        private readonly ?array $extra = [],
    ) {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        if (FinanceServiceHelper::hasEnabledProvider() === false) {
            return;
        }

        $provider = FinanceServiceHelper::initializeForSite();

        if (empty($this->getDealerId($provider))) {
            return;
        }

        $request = new FinanceRequestData(
            uuid: $this->vehicle->uuid,
            registration_number: $this->vehicle->registration_number,
            cap_id: $this->vehicle->cap_id,
            dealer_id: $this->getDealerId($provider),
            engine_size: $this->vehicle->engine_size_cc,
            condition: 'new',
            registration_date: $this->vehicle->first_registration_date,
            term: $this->term,
            mileage: $this->vehicle->odometer_mi,
            annual_mileage: $this->mileage,
            price: $this->vehicle->price,
            deposit: round($this->vehicle->price * $this->depositPercentage * 0.01, 2),
            credit_rating: $this->creditRating ?? Settings::get('finance-credit_rating'),
            vehicle_type: $this->vehicle->type,
            clientKey: $this->getClientKey($provider),
            finance_option: $this->vehicle->getCustom('finance_option'),
            extra: $this->getExtraData(),
            is_vat_applicable: $this->vehicle->is_vat_applicable,
        );

        if ($provider->isEligibleForCreditRatingFinance($this->vehicle)) {
            $this->saveResults($provider->calculateRiskBased($request));
        } else {
            $this->saveResults($provider->calculate($request));
        }
    }

    protected function getExtraData(): array
    {
        return array_merge(
            $this->extra ?? [],
            [
                'dealership_data' => $this->vehicle->dealership?->data,
            ],
        );
    }

    private function getDealerId(FinanceProvider $provider): ?string
    {
        return $this->vehicle->dealership?->data[$provider->dealerIdDataField()] ?? null;
    }
    private function getClientKey(FinanceProvider $provider): ?string
    {
        if (Settings::get('finance-per-dealer-keys') !== true || empty($provider->dealerSpecificAccountKey())) {
            return null;
        }

        return $this->vehicle->dealership?->data[$provider->dealerSpecificAccountKey()] ?? null;
    }

    private function saveResults(Collection $results)
    {
        $this->vehicle->financeExamples()
            ->where('provider', '!=', 'manual')
            ->delete();
        $results->each(fn(FinanceResult $result) => $this->vehicle->financeExamples()->create($result->toArray()));
    }
}
