<?php

namespace Mtc\MercuryDataModels\Traits;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Mtc\ContentManager\Facades\Fields;
use Mtc\MercuryDataModels\Events\LoadVehicleDataFields;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleAttribute;
use Mtc\MercuryDataModels\VehicleAttributeValue;
use Mtc\MercuryDataModels\VehicleOffer;

trait HasVehicleCustomAttributes
{
    private Collection $allAttributes;

    public function attributeValues(): MorphMany
    {
        return $this->morphMany(VehicleAttributeValue::class, 'owner');
    }

    public function scopeWithAttributeCount(Builder $query): void
    {
        $query->addSelect([
            'attribute_count' => VehicleAttributeValue::query()
                ->select(DB::raw('COUNT(*)'))
                ->whereColumn('owner_id', $this->getTable() . '.id')
                ->where('owner_type', $this->getMorphClass())
        ]);
    }

    private function getExtraData($vehicle)
    {
        $extraData = collect();

        if ($vehicle instanceof Vehicle) {
            $extraData = collect(event(new LoadVehicleDataFields($vehicle)))
                ->filter()
                ->mapWithKeys(fn($field) => $field)
                ->map(function ($field) {
                    $field['readonly'] = true;
                    return $field;
                })->values();
        }

        // Merge extraData with custom attributes
        return $extraData->merge($this->getCustomAttributes($vehicle));
    }

    private function getCustomAttributes(Vehicle|VehicleOffer|NewCar $vehicle): Collection
    {
        return $this->allAttributes($vehicle)
            ->map(fn(VehicleAttribute $attribute) => $this->allocateAttributeValues($vehicle, $attribute));
    }

    private function allocateAttributeValues(Vehicle|VehicleOffer|NewCar $vehicle, VehicleAttribute $attribute): array
    {
        /** @var Collection $values */
        $values = $vehicle->attributeValues
            ->where('attribute_id', $attribute->id)
            ->map(fn(VehicleAttributeValue $value) => [
                'id' => $value->id,
                'value' => $value->getValue(),
            ])->take($attribute->count)
            ->toArray();
        $values = collect($values);
        if ($values->count() < $attribute->count) {
            $values = $values->merge(array_fill(0, $attribute->count - $values->count(), [
                'id' => Str::random(),
                'value' => null,
            ]));
        }
        return [
            'id' => $attribute->id,
            'name' => $attribute->name,
            'slug' => $attribute->slug,
            'type' => $attribute->type,
            'component' => 'EditableContent' . Fields::field($attribute->type)->getComponent(),
            'value' => $values,
            'meta' => $attribute->data,
        ];
    }

    private function allAttributes(Vehicle|VehicleOffer|NewCar $vehicle): Collection
    {
        if (!isset($this->allAttributes)) {
            $this->allAttributes = VehicleAttribute::query()->where('model', $vehicle->getMorphClass())->get();
        }

        return $this->allAttributes;
    }
}
