<?php

namespace Mtc\MercuryDataModels\Traits;

use Illuminate\Database\Eloquent\Builder;
use Mtc\MercuryDataModels\User;

trait HasDealershipAccessScope
{
    /**
     * Scope query to only include records from dealerships the user has access to.
     * If the model has a franchise_id column and dealership_id is null,
     * it will also allow access based on the user's accessible dealership franchises.
     *
     * @param Builder $query
     * @param User|null $user User to check access for, defaults to authenticated user
     */
    public function scopeForUser(Builder $query, ?User $user = null): void
    {
        $user = $user ?? auth()->user();

        if (!$user) {
            $query->whereRaw('1 = 0');
            return;
        }

        if ($user->hasRole(['mtc', 'Administrator'])) {
            return;
        }

        $dealershipIds = $user->getAccessibleDealershipIds();

        if (!$this->supportsFranchiseAccess()) {
            $query->whereIn('dealership_id', $dealershipIds);
            return;
        }

        $franchiseIds = $user->getAccessibleFranchiseIds();

        $query->where(function ($q) use ($dealershipIds, $franchiseIds) {
            $q->whereIn('dealership_id', $dealershipIds);

            if ($franchiseIds) {
                $q->orWhere(fn($sub) => $sub->whereNull('dealership_id')
                    ->whereIn('franchise_id', $franchiseIds));
            }
        });
    }

    /**
     * Determine if this model supports franchise-based access control.
     */
    protected function supportsFranchiseAccess(): bool
    {
        return in_array('franchise_id', $this->getFillable());
    }
}
