<?php

namespace Mtc\MercuryDataModels;

use App\Facades\Settings;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Mtc\ContentManager\Contracts\MediaUse;
use Mtc\ContentManager\Contracts\ModelWithContent;
use Mtc\ContentManager\Models\Template;
use Mtc\ContentManager\Traits\HasMedia;
use Mtc\Crm\Traits\ModelSortAndFilter;
use Mtc\MercuryDataModels\Contracts\ModelWithUrlPath;
use Mtc\MercuryDataModels\Contracts\SearchableModel;
use Mtc\MercuryDataModels\Factories\VehicleOfferFactory;
use Mtc\MercuryDataModels\Tools\UiUrlGenerator;
use Mtc\MercuryDataModels\Traits\EnsuresSlug;
use Mtc\MercuryDataModels\Traits\HasResourceViews;
use Mtc\MercuryDataModels\Traits\HasVehicleCustomAttributes;
use OwenIt\Auditing\Contracts\Auditable;

/**
 * @property ?Template $template
 * @property ?VehicleOfferContent $content
 */
class VehicleOffer extends Model implements SearchableModel, ModelWithUrlPath, Auditable, ModelWithContent
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;
    use HasMedia;
    use HasResourceViews;
    use ModelSortAndFilter;
    use EnsuresSlug;
    use HasVehicleCustomAttributes;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'vehicle_offers';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'slug',
        'uuid',
        'stock_provider',
        'vin',
        'type_id',
        'type',
        'vehicle_type',
        'published',
        'published_at',
        'unpublished_at',
        'featured',
        'name',
        'derivative',
        'dealership_id',
        'template_id',
        'make_id',
        'model_id',
        'new_car_id',
        'trim_id',
        'form_id',
        'test_drive_form_id',
        'price',
        'full_price',
        'deposit',
        'seo',
        'order',
        'data',
        'engine_size_cc',
        'colour',
        'battery_range',
        'battery_capacity_kwh',
        'battery_usable_capacity_kwh',
        'battery_charge_time',
        'battery_quick_charge_time',
        'battery_quick_charge_level',
        'franchise_id',
        'fuel_type_id',
        'drivetrain_id',
        'body_style_id',
        'transmission_id',
        'new_car_type',
        'trim',
        'mpg',
        'key_features',
        'description',
        'standard_spec',
        'technical_spec',
        'youtube_video',
        't_and_cs',
        'label',
        'excerpt',
        'cta_button1_label',
        'cta_button1_url',
        'cta_button2_label',
        'cta_button2_url',
        'footnote',
        'co2',
        'cap_id',
        'auto_trader_id',
        'motor_check_id',
    ];

    /**
     * Cast attributes to types
     *
     * @var string[]
     */
    protected $casts = [
        'published_at' => 'datetime',
        'unpublished_at' => 'datetime',
        'published' => 'boolean',
        'featured' => 'boolean',
        'seo' => 'array',
        'data' => 'array',
        'key_features' => 'array',
        'standard_spec' => 'array',
        'technical_spec' => 'array',
    ];

    protected $appends = [
        'status',
    ];

    /**
     * Model Factory
     *
     * @return VehicleOfferFactory
     */
    protected static function newFactory()
    {
        return VehicleOfferFactory::new();
    }

    protected static function boot()
    {
        parent::boot();
        self::creating(function (self $offer) {
            match (Settings::get('automotive-vehicle-offers-slug_type')) {
                'offer_id' => $offer->ensureNumericSlug(),
                default => $offer->ensureSlug(),
            };

            $offer->setDefaultValues();
        });
    }

    public function getOwnerColumn(): string
    {
        return 'offer_id';
    }

    private function ensureNumericSlug()
    {
        if (empty($this->attributes['slug'])) {
            $this->attributes['slug'] = (int) VehicleOffer::query()->max('id') + 1;
        }
    }

    /**
     * Relationship with model
     *
     * @return BelongsTo
     */
    public function make(): BelongsTo
    {
        return $this->belongsTo(VehicleMake::class, 'make_id');
    }

    /**
     * Relationship with model
     *
     * @return BelongsTo
     */
    public function model(): BelongsTo
    {
        return $this->belongsTo(VehicleModel::class, 'model_id');
    }

    /**
     * Relationship with content of the offer
     *
     * @return HasMany
     */
    public function content(): HasMany
    {
        return $this->hasMany(VehicleOfferContent::class, 'offer_id')
            ->whereNull('parent_id')
            ->orderBy('order');
    }

    /**
     * Relationship with content of the offer
     *
     * @return HasMany
     */
    public function allContent(): HasMany
    {
        return $this->hasMany(VehicleOfferContent::class, 'offer_id')
            ->orderBy('order');
    }

    /**
     * Relationship with drivetrain
     *
     * @return BelongsTo
     */
    public function drivetrain(): BelongsTo
    {
        return $this->belongsTo(DrivetrainType::class, 'drivetrain_id');
    }

    /**
     * Relationship with fuel type
     * There can be one - with direct relation or multiple using a belongsToMany table
     * This is defined by a setting flag
     *
     * @return BelongsTo
     */
    public function fuelType(): BelongsTo
    {
        return $this->belongsTo(FuelType::class, 'fuel_type_id');
    }

    public function fuelTypes(): BelongsToMany
    {
        return $this->belongsToMany(FuelType::class, 'vehicle_offer_fuel_types', 'offer_id', 'fuel_type_id');
    }

    /**
     * Relationship with transmission
     *
     * @return BelongsTo
     */
    public function transmission(): BelongsTo
    {
        return $this->belongsTo(TransmissionType::class, 'transmission_id');
    }

    /**
     * Relationship with bodyStyle
     *
     * @return BelongsTo
     */
    public function bodyStyle(): BelongsTo
    {
        return $this->belongsTo(BodyStyleType::class, 'body_style_id');
    }

    /**
     * Offer content history versions
     *
     * @return HasMany
     */
    public function versions(): HasMany
    {
        return $this->hasMany(VehicleOfferContentHistory::class, 'offer_id')->latest();
    }

    /**
     * Relationship with dealership
     *
     * @return BelongsTo
     */
    public function dealership(): BelongsTo
    {
        return $this->belongsTo(Dealership::class);
    }

    /**
     * Relationship with franchise
     *
     * @return BelongsTo
     */
    public function franchise(): BelongsTo
    {
        return $this->belongsTo(Franchise::class);
    }

    /**
     * Relationship with finance data for offer
     *
     * @return HasMany
     */
    public function finance(): HasMany
    {
        return $this->hasMany(VehicleOfferFinance::class, 'offer_id');
    }

    /**
     * Relationship with template that defines content
     *
     * @return BelongsTo
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.template_model'));
    }

    /**
     * Relationship with offer type
     *
     * @return BelongsTo
     */
    public function offerType(): BelongsTo
    {
        return $this->belongsTo(OfferType::class, 'type_id');
    }

    /**
     * Relationship with specs
     *
     * @return MorphMany
     */
    public function specs(): MorphMany
    {
        return $this->morphMany(VehicleTechnicalData::class, 'vehicle');
    }

    public function features(): MorphMany
    {
        return $this->morphMany(VehicleFeature::class, 'vehicle');
    }

    public function colourMap(): BelongsTo
    {
        return $this->belongsTo(ColourMap::class, 'colour', 'sub_colour');
    }


    public function equipment(): MorphMany
    {
        return $this->morphMany(VehicleStandardEquipment::class, 'vehicle');
    }

    public function form(): BelongsTo
    {
        return $this->belongsTo(Form::class);
    }

    public function testDriveForm(): BelongsTo
    {
        return $this->belongsTo(Form::class, 'test_drive_form_id');
    }

    public function labels(): BelongsToMany
    {
        return $this->belongsToMany(Label::class, 'vehicle_offer_labels');
    }

    /**
     * Relationship with resource views
     *
     * @return MorphMany
     */
    public function views(): MorphMany
    {
        return $this->morphMany(ResourceView::class, 'viewable');
    }

    public function conversions(): MorphMany
    {
        return $this->morphMany(Conversion::class, 'owner');
    }

    /**
     * All enquiries submitted for this offer
     *
     * @return MorphMany
     */
    public function allEnquiries(): MorphMany
    {
        return $this->morphMany(config('crm.enquiry_model'), 'reason');
    }

    /**
     * Enquiries submitted for this offer in current month
     *
     * @return MorphMany
     */
    public function currentMonthEnquiries(): MorphMany
    {
        return $this->morphMany(config('crm.enquiry_model'), 'reason')
            ->where('created_at', '>=', Carbon::now()->startOfMonth());
    }

    /**
     * active() scope
     *
     * @param Builder $query
     * @return void
     */
    public function scopeActive(Builder $query)
    {
        $query->where('published', 1)
            ->where(fn ($query) => $query
                ->whereNull('published_at')
                ->orWhere('published_at', '<=', Carbon::now()))
            ->where(fn ($query) => $query
                ->whereNull('unpublished_at')
                ->orWhere('unpublished_at', '>', Carbon::now()));
    }

    /**
     * add deleted() scope to query
     *
     * @param Builder $builder
     * @return Builder
     */
    public function scopeDeleted(Builder $builder): Builder
    {
        return $builder->onlyTrashed();
    }

    public function scopeWithListingRelationships(Builder $query)
    {
        $query
            ->with([
                'primaryMediaUse.media',
                'make.filterIndex',
                'model.filterIndex',
                'fuelType',
                'offerType',
                'franchise',
                'transmission',
                'drivetrain',
                'bodyStyle',
                'dealership',
                'cheapestFinance',
            ])
            ->when(Settings::get('offer-list-load-features'), fn($query) => $query->with('features'));
    }


    public function scopeWhereModelSlug(Builder $query, ?string $slug): void
    {
        if ($slug) {
            $query->whereHas('model.filterIndex', fn ($indexQuery) => $indexQuery->where('slug', $slug));
        }
    }

    /**
     * Find vehicles similar to provided
     *
     * @param Builder $query
     * @param string $matchType
     * @param Vehicle $vehicle
     * @return void
     */
    public function scopeSimilar(Builder $query, string $matchType, self $vehicle)
    {
        match ($matchType) {
            'price' => $query->orderBy(DB::raw('ABS(price - ' . (int) $vehicle->price . ')')),
            'make' => $query->orderBy(DB::raw('make_id = - ' . (int) $vehicle->make_id)),
            default => $query,
        };
    }

    /**
     * New vehicle type filter
     *
     * @param Builder $query
     * @return void
     */
    public function scopeNew(Builder $query): void
    {
        $query->where('type', 'new');
    }

    /**
     * Motability type filter
     *
     * @param Builder $query
     * @return void
     */
    public function scopeMotability(Builder $query): void
    {
        $query->where('type', 'motability');
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithViewCount(Builder $query, int $dayRange = 7): void
    {
        $query->addSelect([
            'view_count' => ResourceView::query()
                ->selectRaw('COALESCE(SUM(hits), 0)')
                ->where('viewable_type', 'offer')
                ->whereColumn('viewable_id', $this->getTable() . '.id')
                ->where('date', '>=', Carbon::now()->subDays($dayRange))
        ]);
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithSearchViewCount(Builder $query, int $dayRange = 7): void
    {
        $query->addSelect([
            'search_views' => ResourceView::query()
                ->selectRaw('COALESCE(SUM(filter_views), 0)')
                ->where('viewable_type', 'offer')
                ->whereColumn('viewable_id', $this->getTable() . '.id')
                ->where('date', '>=', Carbon::now()->subDays($dayRange))
        ]);
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithImageCount(Builder $query, int $dayRange = 7): void
    {
        $query->addSelect([
            'image_count' => MediaUse::query()
                ->selectRaw('COUNT(*)')
                ->where('owner_type', 'offer')
                ->whereColumn('owner_id', $this->getTable() . '.id')
        ]);
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithEnquiryCount(Builder $query, int $dayRange = 7): void
    {
        $query->addSelect([
            'enquiry_count' => Enquiry::query()
                ->selectRaw('COUNT(*)')
                ->where('reason_type', 'offer')
                ->whereColumn('reason_id', $this->getTable() . '.id')
        ]);
    }


    /**
     * Sorting - customized from default due to views
     *
     * @param Builder $query
     * @param string|null $sortOption
     * @return Builder
     */
    public function scopeSetSortBy(Builder $query, ?string $sortOption): Builder
    {
        if (empty($sortOption)) {
            return $query->orderBy('order');
        }

        if ($sortOption === 'views') {
            return $query->withViewCount()->orderByDesc('view_count');
        }

        $direction = str_ends_with($sortOption, '_desc') ? 'desc' : 'asc';
        return $query->orderBy(str_replace('_desc', '', $sortOption), $direction);
    }

    /**
     * Search name text
     *
     * @return string
     */
    public function getSearchNameAttribute(): string
    {
        return $this->name;
    }

    /**
     * Search excerpt text
     *
     * @return string
     */
    public function getSearchExcerptAttribute(): string
    {
        return collect([
            ucwords($this->status),
            $this->urlPath(),
            $this->offerType()->pluck('name')->implode(','),
        ])->filter()->implode(' | ');
    }

    public function getDaysInStockAttribute(): int
    {
        return Carbon::now()->diffInDays($this->created_at);
    }

    public function getDaysSinceUpdateAttribute(): int
    {
        return Carbon::now()->diffInDays($this->updated_at);
    }

    /**
     * Search result icon
     *
     * @return string
     */
    public function getSearchIcon(): string
    {
        return 'badge-dollar';
    }

    /**
     * Route to viewing a vehicle as a part of search response
     *
     * @return string
     */
    public function getSearchResultRoute(): string
    {
        return UiUrlGenerator::make('manage-content/offers/edit/' . $this->id, [], false);
    }

    /**
     * Status attribute
     *
     * @return string
     */
    public function getStatusAttribute(): string
    {
        return $this->published ? 'Published' : 'Draft';
    }

    /**
     * Path to the front-end url
     *
     * @return string
     */
    public function urlPath(): string
    {
        $path = Settings::get('offers-url-path');

        $replacements = [
            '{{MAKE}}' => !empty($this->make?->slug) ? $this->make?->slug : 'make',
            '{{MODEL}}' => !empty($this->model?->slug) ? $this->model?->slug : 'model',
            '{{FRANCHISE}}' => !empty($this->franchise?->slug) ? $this->franchise?->slug : 'franchise',
            '{{SLUG}}' => $this->slug,
        ];

        return str_replace(array_keys($replacements), $replacements, $path);
    }

    /**
     * Define thumbnail sizes to auto-generate for this model
     *
     * @return mixed
     */
    public function getDefaultAllowedMediaSizesAttribute()
    {
        return Config::get('automotive.offer_image_sizes', []);
    }

    protected function setDefaultValues()
    {
        if (empty($this->attributes['template_id'])) {
            $this->template_id = Template::query()->where('slug', 'offer-show')->first()?->id;
        }

        if (empty($this->attributes['form_id'])) {
            $this->form_id = Form::query()
                ->whereHas('type', fn($query) => $query->where('name', 'Offer Enquiry'))
                ->first()?->id;
        }
    }

    /**
     * Set filters based on terms passed
     *
     * @param Builder $query
     * @param array|string $filters
     * @return void
     */
    public function scopeSetFilters(Builder $query, array|string $filters): void
    {
        collect($filters)
            ->filter()
            ->each(function ($filter) use ($query) {
                if (str_starts_with($filter, 'franchise-')) {
                    $query->whereHas(
                        'franchise',
                        fn($query) => $query->where('slug', str_replace('franchise-', '', $filter))
                    );
                } elseif (method_exists($this, 'scope' . ucfirst(Str::camel($filter)))) {
                    $scopeMethod = 'scope' . ucfirst(Str::camel($filter));
                    $this->{$scopeMethod}($query);
                } else {
                    $query->whereHas('offerType', fn ($typeQuery) => $typeQuery->where('slug', $filter));
                }
            });
    }

    /**
     * @return array
     */
    public function replacementsAttributes(): array
    {
        return [
            'name',
        ];
    }

    public function scopeWithRelationshipsForCardView(Builder $query): void
    {
        $query
            ->with([
                'make',
                'model',
                'offerType',
                'fuelType',
                'transmission',
                'attributeValues',
                'bodyStyle',
                'dealership.franchise',
                'cheapestFinance',
                'franchise',
            ]);

        if (in_array('form', request('with', []))) {
            $query->with('form');
        }
    }

    public function cheapestFinance()
    {
        return $this->hasOne(VehicleOfferFinance::class, 'offer_id')->orderBy('monthly_price');
    }

    public function getAllCustom()
    {
        $this->loadMissing('attributeValues');

        $custom_values = [];

        $this->attributeValues->each(function ($attribute_value) use (&$custom_values) {
            $custom_values[$attribute_value->slug] = $attribute_value->getValue();
        });

        return $custom_values;
    }

    public function getCustom(string $attribute): mixed
    {
        $value = $this->attributeValues()->where('slug', $attribute)->get()
            ->map(fn(VehicleAttributeValue $value) => $value->getValue());

        if (empty($value) || empty($value->count())) {
            return null;
        }

        return $this->getCustomAttribute($attribute)->count === 1
            ? $value->first()
            : $value;
    }

    public function scopeSetSelections(Builder $query, array $selections = [])
    {
        collect($selections)
            ->each(function ($selection) use ($query) {
                match ($selection['type']) {
                    'status' => $this->oneOfStatuses($query, $selection['values']),
                    'type_id',
                    'make_id',
                    'model_id',
                    'fuel_type_id',
                    'body_style_id',
                    'transmission_id',
                    'franchise_id',
                    'dealership_id' => $query->when(
                        in_array(null, $selection['values']),
                        fn($subQuery) => $subQuery->where(fn($nullableQuery) => $nullableQuery
                            ->whereNull($selection['type'])
                            ->orWhereIn($selection['type'], $selection['values'])),
                        fn($subQuery) => $subQuery->whereIn($selection['type'], $selection['values'])
                    ),
                    'price_min' => $query->where('price', '>=', $selection['values'][0]),
                    'price_max' => $query->where('price', '<=', $selection['values'][0]),
                    'image_count_min' => $query->having('image_count', '>=', $selection['values'][0]),
                    'image_count_max' => $query->having('image_count', '<=', $selection['values'][0]),
                    default => $query
                };
            });
    }

    public function oneOfStatuses(Builder $query, $selections): void
    {
        $query->where(function ($statusQuery) use ($selections) {
            foreach ($selections as $selection) {
                match ($selection) {
                    'published',
                    'featured' => $statusQuery->orWhere($selection, 1),
                    'draft' => $statusQuery->orWhere('published', 0),
                    'deleted_at' => count($selections) > 1
                        ? $statusQuery->whereNotNull('deleted_at')
                        : $statusQuery->onlyTrashed(),
                    default => $statusQuery,
                };
            }
        });
    }
}
