<?php

namespace Mtc\MercuryDataModels\Finance\Services;

use Mtc\MercuryDataModels\Finance\CodeweaversCreditRating;
use Mtc\MercuryDataModels\Finance\Config\CodeWeaversConfig;
use Mtc\MercuryDataModels\Finance\Contracts\BatchRequestsQuotes;
use Mtc\MercuryDataModels\Finance\Contracts\ExportsStockToFinance;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceProvider;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceRequestData;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResult;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResultCollection;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceType;
use App\Facades\Settings;
use Mtc\MercuryDataModels\Finance\FinanceEligibilityChecker;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Mtc\MercuryDataModels\ApiNotification;
use Mtc\MercuryDataModels\Vehicle;

class CodeWeavers implements FinanceProvider, ExportsStockToFinance, BatchRequestsQuotes
{
    protected array $dealership_product_keys = [];

    public function __construct(
        protected readonly CodeWeaversConfig $config,
        protected readonly FinanceEligibilityChecker $eligibilityChecker,
    ) {
        //
    }

    public function enabled(): bool
    {
        return $this->config->enabled();
    }

    /**
     * Perform calculation on CodeWeavers API
     *
     * @param FinanceRequestData $data
     * @return FinanceResultCollection
     */
    public function calculate(FinanceRequestData $data): FinanceResultCollection
    {
        return $this->process($this->runCalculate($this->mapRequest(collect([$data]))));
    }

    public function export(): bool
    {
        if (Settings::get('sales-codeweavers-export') !== true) {
            return true;
        }

        $query =
            Vehicle::query()
                ->with([
                    'dealership',
                    'primaryMediaUse.media',
                    'fuelType',
                    'make',
                    'model',
                    'transmission',
                    'bodyStyle',
                    'drivetrain',
                ])
                ->where(fn($query) => $query->whereNull('finance_exported_at')
                    ->orWhere('finance_exported_at', '<=', Carbon::now()->subWeek()))
                ->withTrashed()
                ->where(fn($query) => $query->whereNull('deleted_at')
                    ->orWhere('deleted_at', '>=', Carbon::now()->subWeek()));

        if ($query->count() == 0 || $this->tooEarlyForExport()) {
            return false;
        }
        $reference = $this->createExportReference();
        if (!empty($reference)) {
            $query->chunk(100, function (Collection $chunk) use ($reference) {
                $data = $this->buildExport($chunk);
                if ($data['Vehicles']->isNotEmpty()) {
                    $this->runExport($reference, $data);
                }
            });
            $this->startExport($reference);
            return true;
        }
        return false;
    }

    public function batchRequest(Collection $batch): Collection
    {
        return collect($this->runCalculate($this->mapRequest($batch), 'VehicleResults'))
            ->keyBy(fn($result, $index) => collect($result['FinanceProductResults'])
                ->pluck('Vehicle.RegistrationNumber')
                ->filter()
                ->first() ?? $index)
            ->map(fn($vehicleQuotes) => $this->process($vehicleQuotes['FinanceProductResults']));
    }

    public function dealerIdDataField(): string
    {
        return 'codeweavers';
    }

    public function dealerSpecificAccountKey(): string
    {
        return '';
    }

    public function isEligibleForCreditRatingFinance(Vehicle $vehicle): bool
    {
        return $this->eligibilityChecker->allowCreditRatingFinance($vehicle);
    }

    public function calculateRiskBased(FinanceRequestData $data): FinanceResultCollection
    {
        return $this->process($this->runCalculate($this->mapRequest(collect([$data]), true)));
    }

    /**
     * Map vehicle data to request data
     *
     * @param Collection $data
     * @return array
     */
    protected function mapRequest(Collection $entries, $risk_based = false): array
    {
        $parameters = $entries->first();
        $is_percentage_deposit = $entries->count() > 1;
        if (!empty($parameters->extra['flat_deposit_rate'])) {
            $is_percentage_deposit = false;
        }
        $request = [
            'Credentials' => [
                'Username' => $this->config->username(),
                'Password' => $this->config->password(),
            ],
            'Customer' => [
                'Reference' => uniqid('', true),
            ],
            'VehicleRequests' => $entries->map(function ($data) use ($risk_based, $parameters) {
                if (!empty($data->cap_id)) {
                    $identifier_type = 'CapShortCode';
                    $identifier = $data->cap_id;
                } elseif (!empty($data->registration_number)) {
                    $identifier_type = 'VRM';
                    $identifier = $data->registration_number;
                } else {
                    $identifier_type = 'VIN';
                    $identifier = $data->vin;
                }
                return [
                    'Id' => uniqid('', true),
                    'Dealer' => $data->clientKey ?? $this->config->dealerName(),
                    'Products' => $risk_based
                        ? $this->getProductsByCreditRating($parameters->credit_rating)
                        : $this->getProducts($data),
                    'Vehicle' => [
                        'CashPrice' => $data->price,
                        'CurrentMileage' => $data->mileage,
                        'RegistrationDate' => $data->registration_date?->format('Y-m-d'),
                        'IsNew' => $data->condition === 'new',
                        'Identifier' => $identifier,
                        'IdentifierType' => $identifier_type,
                        'Type' => $this->getVehicleType($data->vehicle_type ?? ''),
                        'StockId' => $data->uuid,
                        'RegistrationNumber' => $data->registration_number ?: null,
                        'Vin' => $data->vin,
                    ]
                ];
            })->values()->toArray(),
            'Parameters' => $parameters ? [
                'AnnualMileage' => $parameters->annual_mileage,
                'Term' => $parameters->term,
                'Deposit' => $parameters->extra['flat_deposit_rate'] ?? $parameters->deposit,
                // Controller::getDepositValue() sets all user requested deposits to amount
                // Therefore only ones that can be percentage are bulk ones, so we check for count > 1
                'DepositType' => $is_percentage_deposit ? 'Percentage' : 'Amount',
                'SpecificProductType' => $parameters->extra['product_type'] ?? null,
            ] : null,
        ];

        if ($risk_based && $parameters) {
            $request['Parameters']['CustomerSelectedCreditTier'] = $this->getProviderSpecificCreditRating(
                $parameters->credit_rating
            );
        }

        if (!empty($parameters->make_name)) {
            $request['VehicleRequests'][0]['Vehicle']['Manufacturer'] = $parameters->make_name;
        }

        if (Settings::get('finance-codeweavers-send-vat')) {
            // Check if we have any vehicles that are VAT applicable.
            // Cars should be VAT applicable. LCVs should not be VAT applicable.
            $vat_applicable_vehicles = $entries->filter(fn ($vehicle) => $vehicle->is_vat_applicable);

            if ($vat_applicable_vehicles->isEmpty()) {
                // all of the vehicles in the request should be VAT exempt, so add the required parameter
                $request['Parameters']['CashValuesAreVatExclusive'] = true;
            }
        }

        return $request;
    }

    protected function getProducts(FinanceRequestData $data): array
    {
        if (!array_key_exists($data->dealer_id, $this->dealership_product_keys)) {
            $product_key_string = $data->extra['dealership_data']['codeweavers-product-keys'] ?? '';
            $this->dealership_product_keys[$data->dealer_id] = collect(explode(',', $product_key_string))
                ->filter()
                ->map(fn($product) => ['Key' => $product])
                ->toArray();
        }

        return $this->dealership_product_keys[$data->dealer_id] ?? [];
    }

    /**
     * Call the api with data
     *
     * @param array $request_data
     * @param string $return
     * @return array
     */
    protected function runCalculate(
        array $request_data,
        ?string $return = 'VehicleResults.0.FinanceProductResults'
    ): array {
        $response = Http::withHeaders($this->authHeaders())
            ->post($this->endpoint('public/v3/jsonfinance/calculate'), $request_data);

        if ($response->failed()) {
            Log::warning('Failed Codeweavers finance retrieval', [
                'request' => $request_data,
                'api_key' => $this->config->apiKey(),
                'status_code' => $response->status(),
                'result' => $response->body(),
            ]);

            return [];
        }

        $reference = null;
        if (count($request_data['VehicleRequests']) == 1) {
            $reference = str_replace(' ', '', $request_data['VehicleRequests'][0]['Vehicle']['RegistrationNumber']);
        }

        // Store response data in logs only if below 300_000 characters to avoid bloating logs with large batch requests
        $response_data = strlen($response->body()) < 300_000
            ? $response->json()
            : [];

        ApiNotification::query()
            ->create([
                'provider' => 'codeweavers',
                'processed' => $response->successful(),
                'reference' => $reference,
                'data' => [
                    'request' => $request_data,
                    'result' => $response_data,
                ],
            ]);

        return $response->json($return) ?? [];
    }

    protected function getTermsAndConditions(string $quote_reference = ''): string
    {
        $response = Http::withHeaders($this->authHeaders())
            ->get($this->endpoint("/api/finance/quote/$quote_reference/termsandconditions"));

        if ($response->failed()) {
            if ($response->status() != 404) {
                Log::warning('Failed Codeweavers terms and conditions retrieval', [
                    'request' => $quote_reference,
                    'api_key' => $this->config->apiKey(),
                    'status_code' => $response->status(),
                    'result' => $response->body(),
                ]);
            }

            return '';
        }

        ApiNotification::query()
            ->create([
                'provider' => 'codeweavers',
                'processed' => $response->successful(),
                'data' => [
                    'request' => $quote_reference,
                    'result' => $response->json(),
                ],
            ]);

        return $response->json('TermsAndConditions') ?? '';
    }

    protected function runExport(string $reference, array $request_data): bool
    {
        $response = Http::withHeaders($this->apiAuthHeaders())
            ->put($this->endpoint("/api/vehicle-imports/{$reference}"), $request_data);


        if ($response->failed()) {
            Log::warning('Failed Codeweavers stock upload', [
                'request' => $request_data,
                'api_key' => $this->config->apiKey(),
                'status_code' => $response->status(),
                'result' => $response->body(),
            ]);

            return false;
        }

        return true;
    }


    protected function createExportReference(): string
    {
        $response = Http::withHeaders($this->apiAuthHeaders())
            ->post($this->endpoint('/api/vehicle-imports'), ["Type" => "Delta"]);

        if ($response->failed()) {
            Log::warning('Failed creating CodeWeavers bulk-export', [
                'tenant' => tenant('id'),
                'status_code' => $response->status(),
                'result' => $response->body(),
            ]);

            return false;
        }

        return $response->json("Reference");
    }


    protected function startExport(string $reference): bool
    {
        $response = Http::withHeaders($this->apiAuthHeaders())
            ->get($this->endpoint("/api/vehicle-imports/{$reference}/status"));
        dump($reference, $response->json());

        $response = Http::withHeaders($this->apiAuthHeaders())
            ->post($this->endpoint("/api/vehicle-imports/{$reference}/start"));

        dump($response->json());
        if ($response->failed()) {
            Log::warning('Failed Triggering stock upload for CW', [
                'tenant' => tenant('id'),
                'reference' => $reference,
                'status_code' => $response->status(),
                'result' => $response->body(),
            ]);

            return false;
        }

        $response = Http::withHeaders($this->apiAuthHeaders())
            ->get($this->endpoint("/api/vehicle-imports/{$reference}/status"));

        $stock_updated = collect($response->json('Vehicles'))
            ->filter(fn($vehicle) => $vehicle['Action'] === 'AddOrUpdate')
            ->pluck('ExternalVehicleId');

        Vehicle::query()
            ->whereIn('uuid', $stock_updated)
            ->update([
                'finance_exported_at' => Carbon::now()
            ]);

        Log::info('CodeWeavers export', [
            'reference' => $reference,
            'status' => $response->json()
        ]);

        return true;
    }

    /**
     * Process response
     *
     * @param array $response
     * @return FinanceResultCollection
     */
    protected function process(array $response): FinanceResultCollection
    {
        if (empty($response)) {
            return new FinanceResultCollection();
        }

        $best_of_quotes = array_filter(
            $response,
            fn($quote) => in_array($quote['Key'] ?? '', ['BEST_OF_PCP', 'BEST_OF_HP'])
        );

        $quotes = $best_of_quotes ?: $response;

        $product_quotes = [];

        foreach ($quotes as $quote) {
            $finance_type = $this->productType($quote['Product']['Type']);

            if (empty(empty($quote['HasError'])) || empty($finance_type)) {
                continue;
            }

            if (!array_key_exists($finance_type->value, $product_quotes)) {
                $product_quotes[$finance_type->value] = [];
            }

            // get the cheapest quote of each type, ignoring finance option.
            if (
                empty($product_quotes[$finance_type->value])
                || $quote['Quote']['AllInclusiveRegularPayment'] < $product_quotes[$finance_type->value]->monthly_price
            ) {
                $payments = collect($quote['Quote']['Payments']);

                // if number of payments > 1, this is monthly payment
                $number_of_payments = $payments->max('NumberOfPayments');

                $one_off_payments = $payments->where('NumberOfPayments', '=', 1);

                // assume that if we have more than one one-off payment,
                // the more expensive payment is the final payment
                // and the least expensive payment is the first payment
                $first_payment = $one_off_payments
                    ->where('Amount', '!=', $quote['Quote']['FinalPayment'])
                    ->min('Amount');

                // final payment is always displayed.
                // if the final payment is not separate from the regular payments,
                // decrement the number of payments so that the months add up to the expected term
                if (
                    $quote['Quote']['FinalPayment']
                    && $number_of_payments == $quote['Quote']['Term']
                ) {
                    $number_of_payments--;
                }

                $apply_url = '';
                if (!empty($quote['Quote']['QuoteActions']['Apply'])) {
                    $apply_url = $quote['Quote']['QuoteActions']['Apply'];
                } elseif (!empty($quote['Quote']['QuoteLink'])) {
                    $apply_url = $quote['Quote']['QuoteLink'] . '/apply';
                }

                $product_quotes[$finance_type->value] = new FinanceResult(
                    provider: 'codeweavers',
                    finance_type: $this->productType($quote['Product']['Type']),
                    monthly_price: $quote['Quote']['AllInclusiveRegularPayment'],
                    term: $quote['Quote']['Term'],
                    number_of_payments: $number_of_payments ?? $quote['Quote']['Term'],
                    total_deposit: $quote['Quote']['TotalDeposit'],
                    apr: $quote['Quote']['Apr'],
                    first_payment: $first_payment ?? 0,
                    final_payment: $quote['Quote']['FinalPayment'] ?? null,
                    interest_rate: $quote['Quote']['RateOfInterest'] ?? null,
                    apply_url: $apply_url,
                    product_name: $quote['Product']['Name'],
                    annual_mileage: $quote['Quote']['AnnualMileage'] ?? 0,
                    cash_price: $quote['Quote']['CashPrice'] ?? 0,
                    dealer_deposit: $quote['Quote']['DepositContribution'] ?? 0,
                    customer_deposit: $quote['Quote']['CashDeposit'] ?? 0,
                    payable_amount: $quote['Quote']['TotalAmountPayable'] ?? 0,
                    option_to_purchase_fee: $this->getFee($quote['Quote']['Fees'], 'Option'),
                    documentation_fee: $this->getFee($quote['Quote']['Fees'], 'Admin'),
                    messages: collect($quote['Notifications']['Public'] ?? [])->pluck('Message')->toArray(),
                    terms_and_conditions: $quote['terms_and_conditions'] ?? '',
                    excess_mileage_charge: $quote['Quote']['ExcessMileageRate'] ?? 0,
                );
            }
        }

        return new FinanceResultCollection($product_quotes);
    }

    /**
     * Endpoint for API
     *
     * @param $path
     * @return string
     */
    protected function endpoint($path): string
    {
        return 'https://services.codeweavers.net/' . ltrim($path, '/');
    }

    /**
     * Headers for API
     *
     * @return array
     */
    protected function authHeaders(): array
    {
        $headers = [
            'Content-Type: application/json',
        ];
        if (!empty($this->config->apiKey())) {
            $headers['X-CW-ApiKey'] = 'Codeweavers-' . $this->config->apiKey();
        }
        return $headers;
    }

    protected function apiAuthHeaders(): array
    {
        return [
            'Content-Type' => 'application/json',
            'Authorization' => 'Basic ' . base64_encode(
                config('services.codeweavers.username') . ':' . config('services.codeweavers.password')
            )
        ];
    }

    private function productType(mixed $type): ?FinanceType
    {
        return match ($type) {
            'Lease Purchase' => FinanceType::LP,
            'Hire Purchase' => FinanceType::HP,
            'HP' => FinanceType::HP,
            'PCP' => FinanceType::PCP,
            'CS' => FinanceType::CS,
            'Conditional Sale' => FinanceType::CS,
            default => null,
        };
    }

    /**
     * @param $fees
     * @param $type
     * @return float
     */
    private function getFee($fees, $type): float
    {
        if (is_array($fees)) {
            foreach ($fees as $fee) {
                if ($fee['FeeType'] == $type) {
                    return $fee['Amount'];
                }
            }
        }

        return 0;
    }

    protected function buildExport(Collection $batch): array
    {
        $is_km = Settings::get('automotive-distance_measurement') == 'km';
        return [
            'Vehicles' => $batch
                ->filter(fn(Vehicle $vehicle) => !empty($vehicle->cap_id) || !empty($vehicle->registration_number))
                ->filter(fn(Vehicle $vehicle) => $vehicle->price > 1)
                ->map(fn(Vehicle $vehicle) => [
                    "OrganisationReference" => [
                        'Type' => 'MTCRetailerReference',
                        'Value' => $vehicle->dealership->data['codeweavers']
                            ?? Settings::get('finance-codeweavers-dealer_name'),
                    ],
                    "Physical" => [
                        "Type" => $vehicle->type === 'car' ? "Car" : "LCV",
                        "ExternalVehicleId" => trim($vehicle->uuid ?? $vehicle->id),
                        "Vin" => Str::limit($vehicle->vin, 17, ''),
                        "Status" => $vehicle->is_new ? 'New' : "PreOwned",
                        "OnTheRoadPrice" => $vehicle->price,
                        "IsReserved" => $vehicle->is_reserved,
                        "NoLongerAvailable" => !empty($vehicle->deleted_at) || $vehicle->is_sold,
                        "Mileage" => $is_km ? $vehicle->odometer_km : $vehicle->odometer_mi,
                        "MileageUnit" => $is_km ? "Kilometres" : "Miles",
                        "IsFranchiseApproved" => false,
                        "NumberOfRegisteredOwners" => $vehicle->previous_owner_count,
                        "Registration" => [
                            "RegistrationNumber" => $vehicle->registration_number,
                            "DateRegisteredWithDvla" => $vehicle->first_registration_date?->format('Y-m-d'),
                            "CountryCode" => Settings::get('app-details-country')
                        ],
                        "Images" => [
                            [
                                "Url" => $vehicle->primaryMediaUse?->getUrl('square-tile')
                                    ?? $vehicle->primaryMediaUse?->media?->getOriginalUrlAttribute(),
                                "Description" => $vehicle->title,
                                "AltText" => $vehicle->title
                            ]
                        ],
                        "Codes" => [
                            $this->getIdentifierCode($vehicle)
                        ],
                    ],
                    "Specification" => collect([
                        "Manufacturer" => $vehicle->make?->name,
                        "Model" => $vehicle->model?->name,
                        "FuelType" => $vehicle->fuelType?->name,
                        "BodyStyle" => $vehicle->bodyStyle?->name,
                        "Transmission" => $vehicle->transmission?->name,
                        "DriveTrain" => $vehicle->drivetrain?->name,
                        "Derivative" => $vehicle->derivative,
                        "Seats" => $vehicle->seats,
                        "Doors" => $vehicle->door_count,
                        "EngineSize" => $vehicle->engine_size_cc,
                    ])->filter()->toArray(),
                ])->values(),
        ];
    }

    private function getIdentifierCode(Vehicle $vehicle): array
    {
        if (!empty($vehicle->cap_id)) {
            return [
                "Type" => "CapCarShortCode",
                "Value" => $vehicle->cap_id
            ];
        }

        return [
            "Type" => "VRM",
            "Value" => $vehicle->registration_number
        ];
    }

    /**
     * @param $credit_rating
     * @return string
     */
    private function getProviderSpecificCreditRating($credit_rating = ''): string
    {
        return match (strtoupper($credit_rating)) {
            'EXCELLENT' => CodeweaversCreditRating::EXCELLENT,
            'VERY_GOOD' => CodeweaversCreditRating::VERY_GOOD,
            'GOOD' => CodeweaversCreditRating::GOOD,
            'FAIR' => CodeweaversCreditRating::FAIR,
            'BELOW_AVERAGE', 'POOR' => CodeweaversCreditRating::BELOW_AVERAGE,
            default => '',
        };
    }

    /**
     * @param $credit_rating
     * @return array
     */
    private function getProductsByCreditRating($credit_rating = ''): array
    {
        if (!Settings::get('finance-use-products-risk-based-pricing')) {
            return [];
        }

        return match (strtoupper($credit_rating)) {
            'EXCELLENT' => [
                [
                    'Key' => 'PCPEXCELLENT', // PCP
                ],
                [
                    'Key' => 'EXCELLENTCREDIT', // HP
                ]
            ],
            'GOOD' => [
                [
                    'Key' => 'PCPGOOD',
                ],
                [
                    'Key' => 'GOODCREDIT',
                ],
            ],
            'FAIR' => [
                [
                    'Key' => 'PCPAVERAGE',
                ],
                [
                    'Key' => 'AVERAGECREDIT',
                ],
            ],
            'BELOW_AVERAGE', 'POOR' => [
                [
                    'Key' => 'PCPPOOR',
                ],
                [
                    'Key' => 'POORCREDIT',
                ],
            ],
            default => [],
        };
    }

    private function tooEarlyForExport(): bool
    {
        $has_run_recently = ApiNotification::query()
            ->where('provider', 'codeweavers')
            ->where('reference', 'bulk-export')
            ->where('created_at', '>=', Carbon::now()->subHour())
            ->exists();

        if ($has_run_recently) {
            return true;
        }

        ApiNotification::query()->create([
            'provider' => 'codeweavers',
            'reference' => 'bulk-export',
        ]);
        return false;
    }

    private function getVehicleType(string $vehicle_type): string
    {
        return match (strtolower($vehicle_type)) {
            'car' => 'Car',
            'lcv' => 'Lcv',
            'motorcycle' => 'Bike',
            default => 'Other',
        };
    }
}
