<?php

namespace Mtc\MercuryDataModels;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Mail;
use Mtc\MercuryDataModels\Mail\RepeatedProcessFailureMail;

class QueueProcessLog extends Model
{
    public const STATUS_RUNNING = 'running';
    public const STATUS_COMPLETED = 'completed';
    public const STATUS_FAILED = 'failed';

    public const TYPE_STOCK_SYNC = 'stock_sync';
    public const TYPE_FINANCE_SYNC = 'finance_sync';
    public const TYPE_IMAGE_SYNC = 'image_sync';
    public const TYPE_EXPORT = 'export';

    protected $table = 'queue_process_logs';

    protected $casts = [
        'started_at' => 'datetime',
        'finished_at' => 'datetime',
        'error_details' => 'array',
        'metadata' => 'array',
    ];

    protected $fillable = [
        'process_type',
        'provider',
        'status',
        'started_at',
        'finished_at',
        'duration_seconds',
        'items_processed',
        'items_created',
        'items_updated',
        'items_failed',
        'error_message',
        'error_details',
        'metadata',
    ];

    protected static function boot()
    {
        parent::boot();

        self::created(fn(self $record) => $record->cleanOldLogs());
    }

    /**
     * Keep only the last 100 logs per process type/provider combination
     */
    private function cleanOldLogs(): void
    {
        $query = self::query()->where('process_type', $this->process_type);

        if ($this->provider) {
            $query->where('provider', $this->provider);
        }

        $oldRecord = $query->latest()->skip(100)->first();

        if ($oldRecord) {
            $deleteQuery = self::query()->where('process_type', $this->process_type);

            if ($this->provider) {
                $deleteQuery->where('provider', $this->provider);
            }

            $deleteQuery->where('id', '<', $oldRecord->id)->delete();
        }
    }

    /**
     * Start a new process log
     */
    public static function start(string $processType, ?string $provider = null, array $metadata = []): self
    {
        return self::create([
            'process_type' => $processType,
            'provider' => $provider,
            'status' => self::STATUS_RUNNING,
            'started_at' => now(),
            'metadata' => $metadata,
        ]);
    }

    /**
     * Mark the process as completed
     */
    public function markCompleted(array $stats = []): self
    {
        $this->update([
            'status' => self::STATUS_COMPLETED,
            'finished_at' => now(),
            'duration_seconds' => $this->started_at ? now()->diffInSeconds($this->started_at) : null,
            'items_processed' => $stats['processed'] ?? null,
            'items_created' => $stats['created'] ?? null,
            'items_updated' => $stats['updated'] ?? null,
            'items_failed' => $stats['failed'] ?? null,
        ]);

        return $this;
    }

    /**
     * Mark the process as failed
     */
    public function markFailed(string $errorMessage, array $errorDetails = []): self
    {
        $this->update([
            'status' => self::STATUS_FAILED,
            'finished_at' => now(),
            'duration_seconds' => $this->started_at ? now()->diffInSeconds($this->started_at) : null,
            'error_message' => $errorMessage,
            'error_details' => $errorDetails,
        ]);

        $this->checkForRepeatedFailures();

        return $this;
    }

    /**
     * Check if this process type/provider has failed multiple times consecutively
     * and send an alert to developers if so
     */
    private function checkForRepeatedFailures(): void
    {
        $recentLogs = self::query()
            ->where('process_type', $this->process_type)
            ->when($this->provider, fn($q) => $q->where('provider', $this->provider))
            ->latest('id')
            ->take(2)
            ->get();

        // If we have at least 2 logs and both are failures, send alert
        if ($recentLogs->count() >= 2 && $recentLogs->every(fn($log) => $log->status === self::STATUS_FAILED)) {
            $this->sendRepeatedFailureAlert();
        }
    }

    /**
     * Send alert to developers about repeated failures
     */
    private function sendRepeatedFailureAlert(): void
    {
        $developerEmail = config('mail.developer_alerts');

        if (empty($developerEmail)) {
            return;
        }

        Mail::to($developerEmail)->send(new RepeatedProcessFailureMail($this));
    }

    /**
     * Update metadata during process execution
     */
    public function updateMetadata(array $metadata): self
    {
        $this->update([
            'metadata' => array_merge($this->metadata ?? [], $metadata),
        ]);

        return $this;
    }

    /**
     * Scope to filter by process type
     */
    public function scopeOfType($query, string $type)
    {
        return $query->where('process_type', $type);
    }

    /**
     * Scope to filter by provider
     */
    public function scopeForProvider($query, string $provider)
    {
        return $query->where('provider', $provider);
    }

    /**
     * Scope to filter by status
     */
    public function scopeWithStatus($query, string $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope to get running processes
     */
    public function scopeRunning($query)
    {
        return $query->where('status', self::STATUS_RUNNING);
    }

    /**
     * Scope to get failed processes
     */
    public function scopeFailed($query)
    {
        return $query->where('status', self::STATUS_FAILED);
    }

    /**
     * Check if process is still running
     */
    public function isRunning(): bool
    {
        return $this->status === self::STATUS_RUNNING;
    }

    /**
     * Check if process failed
     */
    public function hasFailed(): bool
    {
        return $this->status === self::STATUS_FAILED;
    }
}
