<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Label;

class LabelFilter extends IsFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->whereHas('labels', fn ($query) => $query->whereIn('label_id', $selection));
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return Label::query()
            ->whereHas(
                'vehicles',
                fn($query) => $query->when(
                    Settings::get('filter-apply-selections-to-results'),
                    fn($query) => $query->where($product_filtering)
                )
            )
            ->get()
            ->each(fn($label) => $label->slug = (string)$label->id);
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return Label::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'Labels';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->id;
    }

    public function filterType(): string
    {
        return 'labels';
    }
    public function minCountToShow()
    {
        return 2;
    }

    protected function hideInUi(): bool
    {
        return !empty(Settings::get('automotive-vehicle-filters-labels_hide_in_ui') ?? false);
    }

    public function getSelectionName($selection): string
    {
        return Label::query()->find($selection)?->name ?? '';
    }
}
