<?php

namespace Mtc\MercuryDataModels;

use App\ImportConditionRepository;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use OwenIt\Auditing\Contracts\Auditable;

class CatalogOfferRule extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;

    protected $fillable = [
        'catalog_offer_id',
        'field',
        'condition',
        'value',
        'data',
    ];

    protected $casts = [
        'data' => 'array',
    ];

    protected $appends = [
        'field_name',
        'condition_name',
        'value_name',
        'parsed_value',
    ];

    public function catalogOffer(): BelongsTo
    {
        return $this->belongsTo(CatalogOffer::class);
    }

    public function getFieldNameAttribute(): string
    {
        return __('vehicles.' . $this->field);
    }

    public function getConditionNameAttribute(): string
    {
        return __('labels.conditions.' . $this->condition);
    }

    public function getValueNameAttribute()
    {
        $value = $this->value;

        // Check if value is JSON-encoded array (for in/not_in conditions)
        if (is_string($value) && str_starts_with($value, '[')) {
            $decoded = json_decode($value, true);
            if (is_array($decoded)) {
                $repository = new ImportConditionRepository();
                return collect($decoded)
                    ->map(fn($id) => $repository->getValue($this->field, $id))
                    ->filter()
                    ->implode(', ');
            }
        }

        return (new ImportConditionRepository())->getValue($this->field, $value);
    }

    /**
     * Get the parsed value - returns array if JSON-encoded, otherwise the raw value.
     * Used by frontend for editing multi-select values.
     */
    public function getParsedValueAttribute()
    {
        $value = $this->value;

        if (is_string($value) && str_starts_with($value, '[')) {
            $decoded = json_decode($value, true);
            if (is_array($decoded)) {
                return $decoded;
            }
        }

        return $value;
    }
}
