<?php

namespace Mtc\MercuryDataModels;

use App\Facades\Feature;
use App\Facades\Settings;
use App\Facades\Site;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Mtc\ContentManager\Contracts\MediaUse;
use Mtc\ContentManager\Traits\HasMedia;
use Mtc\ContentManager\Traits\ModelSortAndFilter;
use Mtc\MercuryDataModels\Contracts\HasMediaChecksums;
use Mtc\MercuryDataModels\Contracts\ModelWithUrlPath;
use Mtc\MercuryDataModels\Contracts\SearchableModel;
use Mtc\MercuryDataModels\Events\VehicleTrashed;
use Mtc\MercuryDataModels\Factories\VehicleFactory;
use Mtc\MercuryDataModels\Repositories\FeatureRepository;
use Mtc\MercuryDataModels\Tools\UiUrlGenerator;
use Mtc\MercuryDataModels\Traits\FormatAsCurrency;
use Mtc\MercuryDataModels\Traits\HasResourceViews;
use Mtc\MercuryDataModels\Traits\HasVehicleCustomAttributes;
use Mtc\MercuryDataModels\Traits\VehicleAgeIdentifierFinder;
use Mtc\VehicleReservations\Reservation;
use OwenIt\Auditing\Contracts\Auditable;

/**
 * @property int $id
 * @property string $slug
 * @property string $uuid
 * @property string $vin
 * @property string $type
 * @property boolean $featured
 * @property boolean $was_recently_synced
 * @property string $auto_trader_id
 * @property string $cap_id
 * @property string $cap_code
 * @property string $motor_check_id
 * @property string $title
 * @property string $derivative
 * @property string $registration_number
 * @property bool $is_new
 * @property int $make_id
 * @property int $model_id
 * @property int $dealership_id
 * @property int $transmission_id
 * @property int $fuel_type_id
 * @property int $drivetrain_id
 * @property string $colour
 * @property float $price
 * @property float|null $original_price
 * @property float|null $previous_price
 * @property float $monthly_price
 * @property float|null $rrp_price
 * @property float|null $deposit
 * @property int $co2
 * @property double $mpg
 * @property int $door_count
 * @property int $manufacture_year
 * @property int $odometer_km
 * @property int $odometer_mi
 * @property int $engine_size_cc
 * @property int $previous_owner_count
 * @property ?Carbon $first_registration_date
 * @property string $description
 * @property array $data
 * @property string $exterior_video_url
 * @property string $interior_video_url
 * @property Carbon $date_in_stock
 */
class Vehicle extends Model implements SearchableModel, ModelWithUrlPath, Auditable, HasMediaChecksums
{
    use \OwenIt\Auditing\Auditable;
    use HasFactory;
    use HasMedia;
    use HasResourceViews;
    use SoftDeletes;
    use ModelSortAndFilter;
    use VehicleAgeIdentifierFinder;
    use FormatAsCurrency;
    use HasVehicleCustomAttributes;

    public const KM_IN_MILES = 1.609344;

    protected static Collection $reservation_price_rules;

    protected $dispatchesEvents = [
        'trashed' => VehicleTrashed::class
    ];

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'vehicles';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'stock_provider',
        'uuid',
        'vin',
        'auto_trader_id',
        'cap_id',
        'cap_code',
        'motor_check_id',
        'type',
        'featured',
        'published_at',
        'sold_at',
        'reserved_at',
        'available_date',
        'is_published',
        'archived_at',
        'finance_exported_at',
        'stock_arrival_date',
        'is_vat_applicable',
        'is_sold',
        'is_demo',
        'is_reserved',
        'title',
        'derivative',
        'registration_number',
        'vrm_condensed',
        'personalized_number_plate',
        'is_new',
        'make_id',
        'model_id',
        'body_style_id',
        'dealership_id',
        'transmission_id',
        'fuel_type_id',
        'drivetrain_id',
        'colour',
        'price',
        'admin_fee',
        'original_price',
        'previous_price',
        'monthly_price',
        'monthly_cost_type',
        'rrp_price',
        'deposit',
        'door_count',
        'seats',
        'manufacture_year',
        'odometer_km',
        'odometer_mi',
        'engine_size_cc',
        'co2',
        'mpg',
        'previous_owner_count',
        'first_registration_date',
        'description',
        'attention_grabber',
        'trim',
        'data',
        'vehicle_length',
        'main_video_url',
        'exterior_video_url',
        'interior_video_url',
        'battery_range',
        'battery_capacity_kwh',
        'battery_usable_capacity_kwh',
        'battery_charge_time',
        'battery_quick_charge_time',
        'battery_quick_charge_level',
        'battery_quick_charge_start_level',
        'battery_slow_charge_description',
        'battery_quick_charge_description',
        'plug_type',
        'rapid_charge_plug_type',
        'wheelbase_type',
        'wheelbase_mm',
        'payload_kg',
        'gross_vehicle_weight_kg',
        'bhp',
        'was_recently_synced',
        'pending_stock_sync',
        'lat',
        'lng',
    ];

    /**
     * Cast attributes to types
     *
     * @var string[]
     */
    protected $casts = [
//        'type' => VehicleType::class,  // need L9
        'is_published' => 'boolean',
        'is_new' => 'boolean',
        'is_sold' => 'boolean',
        'is_reserved' => 'boolean',
        'is_demo' => 'boolean',
        'is_vat_applicable' => 'boolean',
        'archived_at' => 'datetime',
        'featured' => 'boolean',
        'was_recently_synced' => 'boolean',
        'data' => 'array',
        'created_at' => 'datetime',
        'first_registration_date' => 'datetime:Y-m-d H:i:s',
        'published_at' => 'datetime:Y-m-d H:i:s',
        'available_date' => 'datetime:Y-m-d H:i:s',
        'date_in_stock' => 'datetime:Y-m-d H:i:s',
        'stock_arrival_date' => 'datetime:Y-m-d',
    ];

    protected $appends = [
        'status',
    ];

    protected static function newFactory()
    {
        return VehicleFactory::new();
    }

    /**
     * Relationship with finance examples
     *
     * @return HasMany
     */
    public function financeExamples(): HasMany
    {
        return $this->hasMany(VehicleFinance::class);
    }

    public function stockSyncLogs(): HasMany
    {
        return $this->hasMany(VehicleStockSyncLog::class);
    }

    /**
     * @return VehicleFinance
     */
    public function defaultFinanceExample(): HasOne
    {
        return $this->hasOne(VehicleFinance::class)->orderBy('monthly_price');
    }

    /**
     * Relationship with dealership
     *
     * @return BelongsTo
     */
    public function dealership(): BelongsTo
    {
        return $this->belongsTo(Dealership::class);
    }

    /**
     * Relationship with specs
     *
     * @return MorphMany
     */
    public function specs(): MorphMany
    {
        return $this->morphMany(VehicleTechnicalData::class, 'vehicle');
    }

    public function features(): MorphMany
    {
        return $this->morphMany(VehicleFeature::class, 'vehicle')->orderBy('name');
    }

    public function equipment(): MorphMany
    {
        return $this->morphMany(VehicleStandardEquipment::class, 'vehicle');
    }

    public function standardEquipment(): MorphMany
    {
        return $this->equipment()->where('type', VehicleSpecType::STANDARD_EQUIPMENT->value);
    }

    public function optionalEquipment(): MorphMany
    {
        return $this->equipment()->where('type', VehicleSpecType::OPTIONAL_EQUIPMENT->value);
    }

    public function labels(): BelongsToMany
    {
        return $this->belongsToMany(Label::class, 'vehicle_labels');
    }

    public function priceHistory(): HasMany
    {
        return $this->hasMany(VehiclePriceHistory::class, 'vehicle_id');
    }

    public function subscriptions(): HasMany
    {
        return $this->hasMany(VehicleSubscription::class, 'vehicle_id');
    }

    /**
     * Relationship with model
     *
     * @return BelongsTo
     */
    public function make(): BelongsTo
    {
        return $this->belongsTo(VehicleMake::class, 'make_id');
    }

    /**
     * Relationship with model
     *
     * @return BelongsTo
     */
    public function model(): BelongsTo
    {
        return $this->belongsTo(VehicleModel::class, 'model_id');
    }

    /**
     * Relationship with drivetrain
     *
     * @return BelongsTo
     */
    public function drivetrain(): BelongsTo
    {
        return $this->belongsTo(DrivetrainType::class, 'drivetrain_id');
    }

    /**
     * Relationship with fuel type
     *
     * @return BelongsTo
     */
    public function fuelType(): BelongsTo
    {
        return $this->belongsTo(FuelType::class, 'fuel_type_id');
    }

    /**
     * Relationship with transmission
     *
     * @return BelongsTo
     */
    public function transmission(): BelongsTo
    {
        return $this->belongsTo(TransmissionType::class, 'transmission_id');
    }

    /**
     * Relationship with bodyStyle
     *
     * @return BelongsTo
     */
    public function bodyStyle(): BelongsTo
    {
        return $this->belongsTo(BodyStyleType::class, 'body_style_id');
    }

    public function featureEquipment(): MorphMany
    {
        return $this->morphMany(VehicleStandardEquipment::class, 'vehicle')->whereNotNull('feature_id');
    }

    /**
     * All enquiries submitted for this vehicle
     *
     * @return MorphMany
     */
    public function allEnquiries(): MorphMany
    {
        return $this->morphMany(config('crm.enquiry_model'), 'reason');
    }

    /**
     * Enquiries submitted for this vehicle in the last 7 days
     *
     * @return MorphMany
     */
    public function last7DaysEnquiries(): MorphMany
    {
        return $this->morphMany(config('crm.enquiry_model'), 'reason')
            ->where('created_at', '>=', Carbon::now()->subDays(7));
    }

    /**
     * Enquiries submitted for this vehicle in current month
     *
     * @return MorphMany
     */
    public function currentMonthEnquiries(): MorphMany
    {
        return $this->morphMany(config('crm.enquiry_model'), 'reason')
            ->where('created_at', '>=', Carbon::now()->startOfMonth());
    }

    public function last24HoursEnquiries(): MorphMany
    {
        return $this->morphMany(config('crm.enquiry_model'), 'reason')
            ->where('created_at', '>=', Carbon::now()->subHours(24));
    }

    public function getOdometerUnitsAttribute(): ?string
    {
        return Settings::get('automotive-distance_measurement');
    }

    public function getRegistrationYearAttribute(): ?string
    {
        return $this->first_registration_date?->format('Y');
    }

    public function getDateCreatedAttribute(): string
    {
        return $this->created_at->format('Y-m-d');
    }

    /**
     * Get engine size in litres, rounded up to nearest 100 cc.
     * 1234 cc should give 1.3 litres
     *
     * @return float|null
     */
    public function getEngineSizeLitresAttribute(): ?float
    {
        return is_null($this->engine_size_cc)
            ? null
            : ceil($this->engine_size_cc / 100) / 10;
    }

    public function getHasVideoAttribute(): bool
    {
        return $this->videoCount > 0;
    }

    public function getVideoCountAttribute(): int
    {
        return collect([
            $this->main_video_url,
            $this->exterior_video_url,
            $this->interior_video_url,
        ])->filter()
            ->count();
    }

    public function colourMap(): BelongsTo
    {
        return $this->belongsTo(ColourMap::class, 'colour', 'sub_colour');
    }

    public function getImpressionsAttribute(): ?int
    {
        return $this->resourceViews()->sum('hits');
    }

    /**
     * Relationship with page views on a particular date
     * @deprecated Use resourceViews() from HasResourceViews trait instead
     *
     * @return MorphMany
     */
    public function views(): MorphMany
    {
        return $this->resourceViews();
    }

    public function conversions(): MorphMany
    {
        return $this->morphMany(Conversion::class, 'owner');
    }

    /**
     * Relationship with AutoTrader data for vehicle
     *
     * @return HasOne
     */
    public function autoTraderData(): HasOne
    {
        return $this->hasOne(VehicleAutoTraderData::class);
    }

    public function reservations(): HasMany
    {
        return $this->hasMany(Reservation::class, 'vehicle_id');
    }

    /**
     * active() scope
     *
     * @param Builder $query
     * @return void
     */
    public function scopeActive(Builder $query): void
    {
        $query->where(fn($query) => $query->where('is_published', 1));
    }

    public function scopeSetSelections(Builder $query, array $selections = [])
    {
        collect($selections)
            ->each(function ($selection) use ($query) {
                match ($selection['type']) {
                    'status' => $this->oneOfStatuses($query, $selection['values']),
                    'condition' => $this->oneOfConditionStatuses($query, $selection['values']),
                    'type',
                    'make_id',
                    'model_id',
                    'fuel_type_id',
                    'body_style_id',
                    'transmission_id',
                    'dealership_id' => $query->when(
                        in_array(null, $selection['values']),
                        fn($subQuery) => $subQuery->where(fn($nullableQuery) => $nullableQuery
                            ->whereNull($selection['type'])
                            ->orWhereIn($selection['type'], $selection['values'])),
                        fn($subQuery) => $subQuery->whereIn($selection['type'], $selection['values'])
                    ),
                    'stock_provider' => $query->whereIn('stock_provider', $selection['values']),
                    'price_min' => $query->where('price', '>=', $selection['values'][0]),
                    'price_max' => $query->where('price', '<=', $selection['values'][0]),
                    'image_count_min' => $query->having('image_count', '>=', $selection['values'][0]),
                    'image_count_max' => $query->having('image_count', '<=', $selection['values'][0]),
                    'auto_trader_publish_advert' => $this->oneOfAutoTraderPublishAdvertStatuses(
                        $query,
                        $selection['values']
                    ),
                    'has_finance' => $this->filterByFinance($query, $selection['values']),
                    default => $query
                };
            });
    }

    public function filterByFinance(Builder $query, array $values): void
    {
        $query->where(function ($subQuery) use ($values) {
            foreach ($values as $value) {
                match ($value) {
                    'yes' => $subQuery->orWhereHas('financeExamples'),
                    'no' => $subQuery->orWhereDoesntHave('financeExamples'),
                    default => $subQuery,
                };
            }
        });
    }

    public function oneOfStatuses(Builder $query, $selections): void
    {
        $query->where(function ($statusQuery) use ($selections) {
            foreach ($selections as $selection) {
                match ($selection) {
                    'is_published',
                    'is_reserved',
                    'featured',
                    'is_sold' => $statusQuery->orWhere($selection, 1),
                    'draft' => $statusQuery->orWhere('is_published', 0),
                    'deleted_at' => count($selections) > 1
                        ? $statusQuery->whereNotNull('deleted_at')
                        : $statusQuery->onlyTrashed(),
                    default => $statusQuery,
                };
            }
        });
    }

    public function oneOfConditionStatuses(Builder $query, $selections): void
    {
        $query->where(function ($statusQuery) use ($selections) {
            foreach ($selections as $selection) {
                match ($selection) {
                    'used' => $statusQuery->orWhere(fn($subQuery) => $subQuery->used()),
                    'is_new',
                    'is_demo' => $statusQuery->orWhere($selection, 1),
                    'is_pre_reg' => $statusQuery->orWhere(fn($subQuery) => $subQuery->isPreReg()),
                    default => $statusQuery,
                };
            }
        });
    }

    public function oneOfAutoTraderPublishAdvertStatuses(Builder $query, $selections): void
    {
        $query->where(function ($statusQuery) use ($selections) {
            foreach ($selections as $selection) {
                match ($selection) {
                    'yes' => $statusQuery->orWhere(function ($yesQuery) {
                        $yesQuery->whereHas(
                            'autoTraderData',
                            fn($subQuery) => $subQuery->where('publish_advert', 1)
                        )->orWhere(function ($fallbackQuery) {
                            $fallbackQuery->where('is_published', 1)
                                ->whereHas(
                                    'autoTraderData',
                                    fn($subQuery) => $subQuery->whereNull('publish_advert')
                                );
                        });
                    }),
                    'no' => $statusQuery->orWhere(function ($noQuery) {
                        $noQuery->whereHas(
                            'autoTraderData',
                            fn($subQuery) => $subQuery->where('publish_advert', 0)
                        )->orWhere(function ($fallbackQuery) {
                            $fallbackQuery->where('is_published', 0)
                                ->whereHas(
                                    'autoTraderData',
                                    fn($subQuery) => $subQuery->whereNull('publish_advert')
                                );
                        });
                    }),
                    default => $statusQuery,
                };
            }
        });
    }

    public function scopeNoMakeActive(Builder $query)
    {
        $query->active()->whereDoesntHave('make');
    }

    public function scopeNoMake(Builder $query)
    {
        $query->whereDoesntHave('make');
    }

    public function scopeNoModelActive(Builder $query)
    {
        $query->active()->whereDoesntHave('model');
    }

    public function scopeNoModel(Builder $query)
    {
        $query->whereDoesntHave('model');
    }

    public function scopeNoPriceActive(Builder $query)
    {
        $query->active()->where('price', '<=', 1);
    }

    public function scopeNoPrice(Builder $query)
    {
        $query->where('price', '<=', 1);
    }

    public function scopeNoPrimaryImageActive(Builder $query)
    {
        $query->active()->whereDoesntHave('primaryMediaUse');
    }

    public function scopeNoPrimaryImage(Builder $query)
    {
        $query->whereDoesntHave('primaryMediaUse');
    }

    public function scopeNoFuelTypeActive(Builder $query)
    {
        $query->active()->whereDoesntHave('fuelType');
    }

    public function scopeNoFuelType(Builder $query)
    {
        $query->whereDoesntHave('fuelType');
    }

    public function scopeNoBodyStyleActive(Builder $query)
    {
        $query->active()->whereDoesntHave('bodyStyle');
    }

    public function scopeNoBodyStyle(Builder $query)
    {
        $query->whereDoesntHave('bodyStyle');
    }

    public function scopeNoFinanceActive(Builder $query)
    {
        $query->active()->whereDoesntHave('financeExamples');
    }

    public function scopeNoFinance(Builder $query)
    {
        $query->whereDoesntHave('financeExamples');
    }

    public function scopeNoDescriptionActive(Builder $query)
    {
        $query->active()->whereNull('description');
    }

    public function scopeNoDescription(Builder $query)
    {
        $query->whereNull('description');
    }

    public function scopeNoImagesActive(Builder $query)
    {
        $query->active()->whereDoesntHave('mediaUses');
    }

    public function scopeNoImages(Builder $query)
    {
        $query->whereDoesntHave('mediaUses');
    }

    public function scopeNoTransmissionActive(Builder $query)
    {
        $query->active()->whereDoesntHave('transmission');
    }

    public function scopeNoTransmission(Builder $query)
    {
        $query->whereDoesntHave('transmission');
    }

    public function scopeWithRelationshipsForCardView(Builder $query): void
    {
        $query
            ->with([
                'make',
                'model',
                'fuelType',
                'transmission',
                'bodyStyle',
                'primaryMediaUse.media',
                'attributeValues',
                'features',
                'dealership.franchise.primaryMediaUse.media',
                'dealership.primaryDepartment',
            ])
            ->when(
                Settings::get('vehicle-list-load-finance-example') &&
                Settings::get('vehicle-list-load-finance-example-type', 'cheapest') !== 'cheapest',
                fn($query) => $query->with('financeExamples')
            )
            ->when(
                Settings::get('automotive-vehicle-brand-on-filter-card'),
                fn($query) => $query->with('featureEquipment.filterFeature')
            )
            ->when(
                Settings::get('autotrader-advert-performance'),
                fn($query) => $query->with('autoTraderData')
            )
            ->when(
                Settings::get('vehicle-card-image-count') > 1,
                fn($query) => $query->with('mediaUses.media')
            )
            ->when(
                Settings::get('vehicle-list-load-finance-example'),
                fn($query) => $query->with('defaultFinanceExample')
            )
            ->when(
                Settings::get('vehicle-labels-enabled'),
                fn($query) => $query->with('labels')
            );
    }

    /**
     * active() scope
     *
     * @param Builder $query
     * @return void
     */
    public function scopeFeatured(Builder $query): void
    {
        $query->where(fn($query) => $query->where('featured', 1));
    }

    public function scopeSold(Builder $query): void
    {
        $query->where(fn($query) => $query->where('is_sold', 1));
    }

    public function scopeNew(Builder $query): void
    {
        $query->where(fn($query) => $query->where('is_new', 1));
    }

    public function scopeUsed(Builder $query): void
    {
        $query->where(fn($subQuery) => $subQuery->whereNull('is_new')->orWhere('is_new', 0))
            ->where(fn($subQuery) => $subQuery->whereNull('is_demo')->orWhere('is_demo', 0));
    }

    public function scopeIsPreReg(Builder $query): void
    {
        $query->whereHas('attributeValues', function (Builder $subQuery) {
            $subQuery->where('value_integer', 1)
                ->where('slug', 'is-pre-reg');
        });
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithViewCount(Builder $query, int|null $dayRange = 7): void
    {
        $query->addSelect([
            'view_count' => ResourceView::query()
                ->select(DB::raw('COALESCE(SUM(hits), 0)'))
                ->where('viewable_type', 'vehicle')
                ->when($dayRange, fn($subQuery) => $subQuery
                    ->where('date', '>=', Carbon::now()->subDays($dayRange)))
                ->whereColumn('viewable_id', $this->getTable() . '.id')
        ]);
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithSearchViewCount(Builder $query, int|null $dayRange = 7): void
    {
        $query->addSelect([
            'search_views' => ResourceView::query()
                ->select(DB::raw('COALESCE(SUM(filter_views), 0)'))
                ->where('viewable_type', 'vehicle')
                ->when($dayRange, fn($subQuery) => $subQuery
                    ->where('date', '>=', Carbon::now()->subDays($dayRange)))
                ->whereColumn('viewable_id', $this->getTable() . '.id')
        ]);
    }

    /**
     * @param Builder $query
     * @return void
     */
    public function scopeWithImageCount(Builder $query): void
    {
        $query->addSelect([
            'image_count' => MediaUse::query()
                ->select(DB::raw('COUNT(*)'))
                ->where('owner_type', 'vehicle')
                ->whereColumn('owner_id', $this->getTable() . '.id')
        ]);
    }

    /**
     * @param Builder $query
     * @param int $dayRange
     * @return void
     */
    public function scopeWithEnquiryCount(Builder $query, int|null $dayRange = 7): void
    {
        $query->addSelect([
            'enquiry_count' => Enquiry::query()
                ->select(DB::raw('COUNT(*)'))
                ->whereHas('objects', fn($object) => $object
                    ->where('object_type', 'vehicle')
                    ->whereColumn('object_id', $this->getTable() . '.id'))
                ->when($dayRange, fn($subQuery) => $subQuery
                    ->where('ingested_at', '>=', Carbon::now()->subDays($dayRange)))
        ]);
    }

    public function scopeWithReservationCount(Builder $query, int|null $dayRange = 7): void
    {
        $query->addSelect([
            'reservation_count' => Reservation::query()
                ->select(DB::raw('COUNT(*)'))
                ->when($dayRange, fn($subQuery) => $subQuery
                    ->where('ingested_at', '>=', Carbon::now()->subDays($dayRange)))
                ->whereColumn('vehicle_id', $this->getTable() . '.id')
        ]);
    }

    public function scopeIsWithinDistance($query, float $lat, float $lng, int $distance = 5)
    {
        $distance_multiplier = Settings::get('automotive-distance_measurement') === 'mi' ? 0.000621371192 : 0.001;
        return $query->whereRaw("ST_Distance_Sphere(point(lng, lat), point(?, ?)) * ? < ?", [
            $lng,
            $lat,
            $distance_multiplier,
            $distance,
        ]);
    }

    /**
     * Sorting - customized from default due to views
     *
     * @param Builder $query
     * @param string|null $sortOption
     * @return Builder
     */
    public function scopeSetSortBy(Builder $query, ?string $sortOption): Builder
    {
        if (empty($sortOption)) {
            return $query;
        }

        if ($sortOption === 'views') {
            return $query->withViewCount()->orderByDesc('view_count');
        }

        $direction = str_ends_with($sortOption, '_desc') ? 'desc' : 'asc';
        return $query->orderBy(str_replace('_desc', '', $sortOption), $direction);
    }

    /**
     * Find vehicles similar to provided
     *
     * @param Builder $query
     * @param string $matchType
     * @param Vehicle $vehicle
     * @return void
     */
    public function scopeSimilar(Builder $query, string $matchType, self $vehicle): void
    {
        match ($matchType) {
            'price' => $query->orderBy(DB::raw('ABS(price - ' . (int)$vehicle->price . ')')),
            'make' => $query->where('make_id', '=', $vehicle->make_id),
            'price-20' => $query->whereBetween('price', [$vehicle->price * 0.8, $vehicle->price * 1.2]),
            'make-model' => $query->where('make_id', $vehicle->make_id)
                ->where('model_id', $vehicle->model_id),
            'body-type' => $query->where('body_style_id', $vehicle->body_style_id),
            default => $query,
        };
    }

    /**
     * add deleted() scope to query
     *
     * @param Builder $builder
     * @return Builder
     */
    public function scopeDeleted(Builder $builder): Builder
    {
        return $builder->onlyTrashed();
    }

    /**
     * exportable()
     * Stub of active selection for now.
     * Should add settings for making customizing this over time
     *
     * @param Builder $query
     * @return void
     */
    public function scopeExportable(Builder $query): void
    {
        $query->active();
    }

    /**
     * Convert miles to km
     *
     * @param int $miles
     * @return int
     */
    public function milesToKm(int $miles): int
    {
        return (int)($miles * self::KM_IN_MILES);
    }

    /**
     * Convert kilometers to miles
     *
     * @param int $km
     * @return int
     */
    public function kmToMiles(int $km): int
    {
        return (int)($km / self::KM_IN_MILES);
    }

    /**
     * date_in_stock attribute
     * When vehicle was added to stock
     *
     * @return string
     */
    public function getDateInStockAttribute()
    {
        if (!empty($this->attributes['stock_arrival_date'])) {
            return $this->attributes['stock_arrival_date'];
        }

        return $this->created_at;
    }

    /**
     * Condition attribute
     *
     * @return string
     */
    public function getConditionAttribute(): string
    {
        return $this->is_new ? 'new' : 'used';
    }

    public function getIsPreRegAttribute(): bool
    {
        return (bool)$this->attributeValues
            ->where('slug', 'is-pre-reg')
            ->where('value_integer', 1)
            ->first();
    }

    /**
     * Status attribute
     *
     * @return string
     */
    public function getStatusAttribute(): string
    {
        if ($this->is_reserved === true) {
            return 'Reserved';
        }

        if ($this->is_sold === true) {
            return 'Sold';
        }

        if (!empty($this->deleted_at)) {
            return 'Archived';
        }

        return $this->is_published ? 'Published' : 'Draft';
    }

    /**
     * Update Status value
     *
     * @param string $status
     * @return void
     */
    public function updateStatus(string $status)
    {
        match ($status) {
            'draft' => $this->update([
                'is_reserved' => false,
                'is_sold' => false,
                'is_published' => false,
            ]),
            'published' => $this->update([
                'is_reserved' => false,
                'is_sold' => false,
                'is_published' => true,
            ]),
            'reserved' => $this->update([
                'is_reserved' => true,
                'is_sold' => false,
                'is_published' => true,
            ]),
            'sold' => $this->update([
                'is_reserved' => false,
                'is_sold' => true,
                'is_published' => false,
            ]),
        };
    }

    public function getDaysInStockAttribute(): int
    {
        if (!empty($this->attributes['stock_arrival_date'])) {
            return Carbon::now()->diffInDays($this->attributes['stock_arrival_date']);
        }

        return Carbon::now()->diffInDays($this->created_at);
    }

    public function getCreatedAtSinceAttribute(): int
    {
        return $this->getDaysInStockAttribute();
    }

    public function getDaysSinceUpdateAttribute(): int
    {
        return Carbon::now()->diffInDays($this->updated_at);
    }

    /**
     * Search name text
     *
     * @return string
     */
    public function getSearchNameAttribute(): string
    {
        return $this->title;
    }

    /**
     * Search excerpt text
     *
     * @return string
     */
    public function getSearchExcerptAttribute(): string
    {
        return collect([
            $this->is_published ? '' : 'DRAFT',
            $this->vrm_condensed,
            $this->asCurrency($this->price ?? 0, Settings::get('app-details-currency')),
            $this->first_registration_date?->format('d/m/Y'),
            $this->derivative,
        ])->filter()->implode(' | ');
    }

    public function getImageCountAttribute()
    {
        if (isset($this->attributes['image_count'])) {
            return $this->attributes['image_count'];
        }
        return $this->mediaUses()->count();
    }


    public function getFinanceCountAttribute()
    {
        return $this->financeExamples()->count();
    }

    /**
     * Search result icon
     *
     * @return string
     */
    public function getSearchIcon(): string
    {
        return $this->getPreviewImage('mini-thumb')
            ?? 'cars';
    }

    /**
     * Route to viewing a vehicle as a part of search response
     *
     * @return string
     */
    public function getSearchResultRoute(): string
    {
        return UiUrlGenerator::make('manage-content/vehicles/edit/' . $this->id, [], false);
    }

    /**
     * Path to the front-end url
     *
     * @return string
     */
    public function urlPath(): string
    {
        $path = Settings::get('vehicles-url-path');

        $replacements = [
            '{{MAKE}}' => !empty($this->make?->slug) ? $this->make?->slug : 'make',
            '{{MODEL}}' => !empty($this->model?->slug) ? $this->model?->slug : 'model',
            '{{SLUG}}' => $this->slug,
        ];

        return str_replace(array_keys($replacements), $replacements, $path);
    }

    /**
     * Define thumbnail sizes to auto-generate for this model
     *
     * @return mixed
     */
    public function getDefaultAllowedMediaSizesAttribute()
    {
        return Config::get('automotive.vehicle_image_sizes', []);
    }

    public function archive()
    {
        $this->update([
            'is_published' => false,
            'is_archived' => true,
        ]);
    }

    public function getCustom(string $attribute): mixed
    {
        $value = $this->attributeValues()->where('slug', $attribute)->get()
            ->map(fn(VehicleAttributeValue $value) => $value->getValue());

        if (empty($value) || empty($value->count())) {
            return null;
        }

        return $this->getCustomAttribute($attribute)->count === 1
            ? $value->first()
            : $value;
    }

    public function getMonthlyCostType(): ?string
    {
        if (empty($this->monthly_cost_type)) {
            return null;
        }

        return collect(config('automotive.monthly_cost_types', []))
            ->filter(fn($type) => $type['id'] === $this->monthly_cost_type)
            ->first()['name'] ?? null;
    }

    public function getRegistrationNumberWithoutSpacesAttribute()
    {
        return str_replace(' ', '', $this->registration_number);
    }

    public function getComingSoonStatus(): bool
    {
        return $this->available_date
            ? $this->available_date > Carbon::now()->endOfDay()
            : false;
    }

    /**
     * @param string $attribute
     * @return VehicleAttribute|Model
     */
    protected function getCustomAttribute(string $attribute): VehicleAttribute
    {
        return VehicleAttribute::query()
            ->where('slug', $attribute)
            ->firstOrFail();
    }

    /**
     * @return array
     */
    public function replacementsAttributes(): array
    {
        return [
            'title',
            'derivative',
            'description'
        ];
    }

    public function getStock()
    {
        return 1;
    }

    public function getReservationAmount(): float
    {
        if (Feature::isEnabled('reservation-price-rules')) {
            return $this->getRuleBasedReservationPrice() ?? Settings::get('sales-reserve_price', 99);
        }
        $reservation_price_car = Settings::get('sales-reserve_price', 99);
        $reservation_price_lcv = Settings::get('sales-reserve_price_lcv', $reservation_price_car);
        $reservation_price_bike = Settings::get('sales-reserve-price-motorcycle', $reservation_price_car);

        return match (strtoupper($this->type)) {
            'LCV' => $reservation_price_lcv,
            'MOTORCYCLE' => $reservation_price_bike,
            default => $reservation_price_car,
        };
    }

    protected function getRuleBasedReservationPrice(): ?float
    {
        $prices = $this->loadReservationPriceRules();
        if ($prices->isEmpty()) {
            return null;
        }

        return $prices
            ->filter(fn(ReservationPrice $price) => $this->reservationPriceAppliesToVehicle($price))
            ->first()
            ?->price;
    }

    protected function loadReservationPriceRules(): Collection
    {
        // Only load if values not set or running tests.
        if (!isset(self::$reservation_price_rules) || app()->runningUnitTests()) {
            self::$reservation_price_rules = ReservationPrice::query()->where('active', 1)->get();
        }
        return self::$reservation_price_rules;
    }

    protected function reservationPriceAppliesToVehicle(ReservationPrice $price): bool
    {
        if ($price->rules->isEmpty()) {
            return true;
        }

        return $price->rules
            ->reject(fn(ReservationPriceRule $rule) => $this->priceRuleApplies($rule))
            ->isEmpty();
    }

    protected function priceRuleApplies(ReservationPriceRule $rule): bool
    {
        return match ($rule->condition) {
            '=' => $rule->value == $this->getAttribute($rule->field),
            '!=' => $rule->value != $this->getAttribute($rule->field),
            '>' => $this->getAttribute($rule->field) > $rule->value,
            '>=' => $this->getAttribute($rule->field) >= $rule->value,
            '<' => $this->getAttribute($rule->field) < $rule->value,
            '<=' => $this->getAttribute($rule->field) <= $rule->value,
            'is_null' => empty($this->getAttribute($rule->field)),
            default => throw new \Exception('Unknown condition check: ' . $rule->condition . json_encode([
                    'tenant' => tenant('id'),
                    'rule' => $rule,
                ])),
        };
    }

    public function imageChecksumAttribute(): string
    {
        return 'image_checksum';
    }

    public function reviews()
    {
        return $this->belongsToMany(
            VehicleReview::class,
            'vehicle_review_uses',
            'vehicle_id',
            'vehicle_review_id'
        );
    }

    public function getPriceAsIntegerAttribute(): int
    {
        return (int)$this->price;
    }

    public function getDaysSinceFirstRegistrationAttribute(): ?int
    {
        if (empty($this->first_registration_date)) {
            return null;
        }

        return $this->first_registration_date->diffInDays(now());
    }

    public function getCalculatedAvailableDateAttribute(): ?Carbon
    {
        $ageThresholdInDays = 90;
        if ($this->available_date || !$this->first_registration_date) {
            return $this->available_date;
        }

        if ($this->days_since_first_registration < $ageThresholdInDays) {
            $minimumAllowedDate = $this->first_registration_date->copy()->addDays($ageThresholdInDays + 1);
            if (!$this->available_date || $this->available_date->lt($minimumAllowedDate)) {
                return $minimumAllowedDate;
            }
        }

        return null;
    }

    public function getSiteUrlAttribute(): string
    {
        return !empty($this->slug) ? Site::vehicleUrl($this) : '';
    }

    public function getAutoTraderDataPublishAdvertStatusAttribute(): string
    {
        if (isset($this->autoTraderData->publish_advert)) {
            return $this->autoTraderData->publish_advert ? 'Y' : 'N';
        }

        return $this->is_published ? 'Y' : 'N';
    }
}
