<?php

namespace Mtc\MercuryDataModels\Finance\Jobs;

use Mtc\MercuryDataModels\Finance\Contracts\BatchRequestsQuotes;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceProvider;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceRequestData;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResultCollection;
use App\Facades\Settings;
use Mtc\MercuryDataModels\Finance\FinanceServiceHelper;
use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Vehicle;

class RunQuoteRequest implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * Create a new job instance.
     */
    public function __construct(private BatchRequestsQuotes $provider, private bool $all = false)
    {
        $this->onQueue('finance');
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $extra = [];
        if (Settings::get('finance-restricted-product-type')) {
            $extra['product_type'] = Settings::get('finance-restricted-product-type');
        }
        if (Settings::get('finance-deposit_percentage') > 100) {
            $extra['flat_deposit_rate'] = Settings::get('finance-deposit_percentage');
        }
        Vehicle::query()
            ->with([
                'dealership',
            ])
            ->whereNotNull('registration_number')
            ->when(empty($this->all), fn($conditionQuery) => $conditionQuery
                ->where(fn ($query) => $query->whereDoesntHave('financeExamples')
                    ->orWhereHas('financeExamples', fn($financeQuery) => $financeQuery
                        ->where('created_at', '<', Carbon::now()->subWeek()))))
            ->orderByDesc('created_at')
            ->chunk(25, function ($batch) use ($extra) {
                $batch = $batch->keyBy('registration_number');
                $field = $this->getFinanceProvider()->dealerIdDataField();
                $vehicles = $batch
                    ->filter(fn(Vehicle $vehicle) => !empty($vehicle->price))
                    ->reject(fn(Vehicle $vehicle) => !empty($vehicle->data['stop_finance_sync']))
                    ->map(fn(Vehicle $vehicle) => new FinanceRequestData(
                        uuid: $vehicle->uuid ?? $vehicle->id,
                        registration_number: $vehicle->registration_number,
                        cap_id: $vehicle->cap_id,
                        dealer_id: $vehicle->dealership->location_finance ?? '',
                        engine_size: $vehicle->engine_size_cc,
                        condition: $vehicle->condition,
                        registration_date: $vehicle->first_registration_date,
                        term: Settings::get('finance-term'),
                        mileage: $vehicle->odometer_mi,
                        annual_mileage: Settings::get('finance-annual_mileage'),
                        price: $vehicle->price,
                        deposit: Settings::get('finance-deposit_percentage'),
                        credit_rating: Settings::get('finance-credit_rating'),
                        clientKey: $vehicle->dealership?->data[$field] ?? null,
                        finance_option: $vehicle->getCustom('finance_option'),
                        extra: $extra,
                        is_vat_applicable: $vehicle->is_vat_applicable,
                    ));
                $this->processBatch($batch, $this->provider->batchRequest($vehicles));
            });
    }

    /**
     * Gert the finance provider for client
     *
     * @return FinanceProvider
     */
    protected function getFinanceProvider(): FinanceProvider
    {
        return FinanceServiceHelper::initializeForSite();
    }

    private function processBatch(Collection $batch, Collection $quotes): void
    {
        $quotes->each(fn(FinanceResultCollection $results, $index) => $this->store($batch[$index] ?? null, $results));
    }

    protected function store(?Vehicle $vehicle, FinanceResultCollection $results): FinanceResultCollection
    {
        if (!$vehicle) {
            return $results;
        }
        // Delete examples older than 1 week as they are likely to be outdated
        $vehicle->financeExamples()
            ->where('provider', '!=', 'manual')
            ->where('created_at', '<', Carbon::now()->subWeek())
            ->delete();

        $results->filter()
            ->each(fn($result) => $vehicle->financeExamples()->updateOrCreate([
                'term' => $result->term,
                'finance_type' => $result->finance_type,
                'annual_mileage' => $result->annual_mileage,
                'total_deposit' => $result->total_deposit,
            ], $result->toArray()));

        if ($results->count() > 0) {
            $lowest_monthly = $results->sortBy('monthly_price')->first();

            $vehicle->update([
                'monthly_price' => $lowest_monthly->monthly_price,
                'monthly_cost_type' => strtolower($lowest_monthly->finance_type->value),
                'deposit' => $lowest_monthly->total_deposit
            ]);
        }

        return $results;
    }
}
