<?php

namespace Mtc\MercuryDataModels\Leasing;

use App\Facades\Settings;
use App\Traits\CacheObject;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\KeyloopLeaseVehicleVariant;
use Mtc\MercuryDataModels\Leasing\Contracts\LeaseVariantFinance;

class KeyloopLeaseFinanceService
{
    use CacheObject;

    /*
     * Set cache lifetime to 90 minutes.
     * We expect finance to be regenerated every 60 minutes.
     * Caching for 90 minutes should ensure we have no periods of time with uncached finance.
     * We do not want to cache indefinitely,
     * because variants will be removed from the site and their finance will no longer be relevant.
     */
    private const CACHE_LIFETIME_MINUTES = 90;

    public function getBulkVariantFinance(array $variant_slugs): Collection
    {
        return collect($variant_slugs)->mapWithKeys(function ($variant_slug) {
            $variant = KeyloopLeaseVehicleVariant::query()->where('slug', $variant_slug)->first();

            return [
                $variant_slug => is_null($variant)
                    ? null
                    : $this->getVariantFinance($variant)
            ];
        });
    }

    /**
     * @param KeyloopLeaseVehicleVariant $variant
     * @param bool $update_cheapest_variant_price
     * @param bool $clear_cache
     * @return LeaseVariantFinance|null
     */
    public function getVariantFinance(
        KeyloopLeaseVehicleVariant $variant,
        bool $update_cheapest_variant_price = false,
        bool $clear_cache = false,
    ): ?LeaseVariantFinance {
        if ($clear_cache) {
            $this->clearCache($this->cacheKey($variant));
        }

        return $this->cache(
            $this->cacheKey($variant),
            self::CACHE_LIFETIME_MINUTES,
            fn() => $this->getConfiguredProvider()?->getVariantFinanceOptions($variant, $update_cheapest_variant_price)
        );
    }

    public function getRoadTaxFinance(KeyloopLeaseVehicleVariant $variant): ?LeaseVariantFinance
    {
        return $this->getConfiguredProvider()?->getRoadTaxFinanceOptions($variant);
    }

    public function getItemFinance(KeyloopLeaseVehicleVariant $variant, array $item): LeaseVariantFinance
    {
        return $this->getConfiguredProvider()?->getItemFinanceOptions($variant, $item);
    }

    /**
     * @return KeyloopLeaseFinanceCalculator|null
     */
    protected function getConfiguredProvider(): ?KeyloopLeaseFinanceCalculator
    {
        if (!Settings::get('leasing-keyloop-fleet-kompact-enabled')) {
            return null;
        }

        return match (strtolower(Settings::get('leasing-keyloop-fleet-kompact-provider-type'))) {
            'kearys' => new KearysLeaseFinanceCalculator(),
            default => null,
        };
    }

    protected function cacheKey(KeyloopLeaseVehicleVariant $variant): string
    {
        return 'lease-variant-finance-' . $variant->external_variant_id;
    }
}
