<?php

namespace Mtc\MercuryDataModels\Jobs;

use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Mtc\MercuryDataModels\ResourceView;

class TrackResourceView implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    protected array $allowedTypes = [
        'hits',
        'quick_view_hits',
        'filter_views',
        'compare_views',
        'compare_email_requests'
    ];

    /**
     * Create a new job instance.
     *
     * @param string $viewableType
     * @param int $viewableId
     * @param string $trackingType
     * @param int $incrementBy
     */
    public function __construct(
        private readonly string $viewableType,
        private readonly int $viewableId,
        private readonly string $trackingType = 'hits',
        private readonly int $incrementBy = 1,
    ) {
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        if (!in_array($this->trackingType, $this->allowedTypes, true)) {
            return;
        }

        $view = ResourceView::query()->firstOrCreate([
            'viewable_type' => $this->viewableType,
            'viewable_id' => $this->viewableId,
            'date' => Carbon::today()->toDateString(),
        ]);

        $view->increment($this->trackingType, $this->incrementBy);
    }

    /**
     * Static helper to track a page view for any model
     */
    public static function trackHit(Model $model): void
    {
        self::dispatch($model->getMorphClass(), $model->getKey(), 'hits');
    }

    /**
     * Static helper to track filter/search views
     */
    public static function trackFilterView(Model $model): void
    {
        self::dispatch($model->getMorphClass(), $model->getKey(), 'filter_views');
    }

    /**
     * Static helper to track quick view hits (e.g., modal previews)
     */
    public static function trackQuickView(Model $model): void
    {
        self::dispatch($model->getMorphClass(), $model->getKey(), 'quick_view_hits');
    }

    /**
     * Static helper to track compare views
     */
    public static function trackCompareView(Model $model): void
    {
        self::dispatch($model->getMorphClass(), $model->getKey(), 'compare_views');
    }
}
