<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('deals', function (Blueprint $table) {
            $table->id();
            $table->uuid('reference')->unique();
            $table->foreignId('customer_id')->index()->nullable();
            $table->foreignId('vehicle_id')->index();
            $table->foreignId('status_id')->index();
            $table->dateTime('ingested_at')->nullable()->index();
            $table->foreignId('assignee_id')->nullable()->index();
            $table->decimal('deposit_amount')->nullable();
            $table->decimal('customer_deposit', 10)->nullable();
            $table->decimal('part_ex_contribution', 10)->nullable();
            $table->decimal('total_amount')->nullable();
            $table->decimal('payable_amount')->nullable();
            $table->string('contact_number')->nullable();
            $table->string('first_name')->nullable();
            $table->string('last_name')->nullable();
            $table->string('email')->nullable();
            $table->string('payment_type')->nullable();
            $table->string('credit_rating')->nullable();
            $table->smallInteger('term')->nullable();
            $table->integer('annual_mileage')->nullable();
            $table->text('data')->nullable();
            $table->timestamps();
            $table->softDeletes();
        });

        Schema::create('deal_add_ons', function (Blueprint $table) {
            $table->id();
            $table->foreignId('deal_id')->index();
            $table->foreignId('add_on_id')->index();
            $table->decimal('price')->index();
            $table->string('payment_term', 20)->nullable();
            $table->timestamps();
        });

        Schema::create('deal_part_exchanges', function (Blueprint $table) {
            $table->id();
            $table->foreignId('deal_id')->index();
            $table->string('provider')->index();
            $table->string('registration')->index();
            $table->unsignedInteger('mileage')->index();
            $table->string('vehicle_type', 20)->nullable();
            $table->string('make', 50)->nullable();
            $table->string('model', 50)->nullable();
            $table->string('derivative')->nullable();
            $table->string('fuel_type', 50)->nullable();
            $table->string('engine_size', 50)->nullable();
            $table->string('body_type', 50)->nullable();
            $table->string('transmission', 50)->nullable();
            $table->date('date_of_registration')->nullable();
            $table->datetime('valuation_made_at')->nullable()->index();
            $table->decimal('retail_price', 10, 2)->nullable();
            $table->decimal('average_price', 10, 2)->nullable();
            $table->decimal('clean_price', 10, 2)->nullable();
            $table->decimal('below_price', 10, 2)->nullable();
            $table->decimal('outstanding_finance', 12)->nullable();
            $table->decimal('cashback_amount', 12)->nullable();
            $table->text('data')->nullable();
            $table->timestamps();
        });

        Schema::create('deal_finance', function (Blueprint $table) {
            $table->id();
            $table->foreignId('deal_id')->index();
            $table->string('provider')->nullable()->index();
            $table->string('type', 5)->nullable()->index();
            $table->string('lender_name')->nullable();
            $table->string('lender_logo')->nullable();
            $table->string('quote_reference')->nullable()->index();
            $table->dateTime('quote_valid_until')->nullable();
            $table->string('apply_url')->nullable();
            $table->string('retailer_apply_url')->nullable();
            $table->string('print_url')->nullable();
            $table->string('credit_check_url')->nullable();
            $table->boolean('selected')->default(0)->index();
            $table->integer('annual_mileage')->nullable();
            $table->decimal('first_monthly', 12)->nullable();
            $table->decimal('monthly_payment', 12)->nullable();
            $table->decimal('final_payment', 12)->nullable();
            $table->decimal('full_price', 12)->nullable();
            $table->decimal('deposit', 12)->nullable();
            $table->decimal('total_amount', 12)->nullable();
            $table->decimal('total_credit_amount', 12)->nullable();
            $table->decimal('apr', 5)->nullable();
            $table->decimal('interest_rate', 5)->nullable();
            $table->smallInteger('number_of_months')->nullable();
            $table->decimal('cash_price', 10, 2)->nullable();
            $table->decimal('dealer_deposit', 10, 2)->nullable();
            $table->decimal('customer_deposit', 10, 2)->nullable();
            $table->decimal('payable_amount', 10, 2)->nullable();
            $table->decimal('option_to_purchase_fee', 10)->nullable();
            $table->decimal('documentation_fee')->nullable();
            $table->decimal('excess_mileage_charge')->nullable();
            $table->text('terms_and_conditions')->nullable();
            $table->text('data')->nullable();
            $table->timestamps();
        });

        Schema::create('deal_history', function (Blueprint $table) {
            $table->id();
            $table->foreignId('deal_id')->index();
            $table->foreignId('status_id')->index();
            $table->foreignId('customer_id')->index()->nullable();
            $table->foreignId('user_id')->index()->nullable();
            $table->dateTime('notification_sent_at')->index()->nullable();
            $table->text('data')->nullable();
            $table->timestamps();
        });

        Schema::create('deal_notes', function (Blueprint $table) {
            $table->id();
            $table->foreignId('deal_id')->index();
            $table->foreignId('user_id')->index()->nullable();
            $table->text('message')->nullable();
            $table->text('data')->nullable();
            $table->timestamps();
        });

        Schema::create('deal_messages', function (Blueprint $table) {
            $table->id();
            $table->foreignId('deal_id')->index();
            $table->foreignId('user_id')->index()->nullable();
            $table->foreignId('customer_id')->index()->nullable();
            $table->dateTime('notification_sent_at')->index()->nullable();
            $table->text('message')->nullable();
            $table->text('data')->nullable();
            $table->timestamps();
        });

        Schema::create('deal_builder_add_ons', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->boolean('active')->default(0)->index();
            $table->text('description')->nullable();
            $table->decimal('price')->nullable();
            $table->boolean('allow_roll_into_monthly')->nullable();
            $table->timestamps();
        });

        Schema::create('deal_builder_add_on_conditions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('add_on_id')->index();
            $table->string('field', 50);
            $table->string('condition', 10);
            $table->string('value')->nullable();
            $table->timestamps();
        });

        Schema::create('deal_builder_statuses', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('colour')->nullable();
            $table->boolean('editable')->nullable()->index();
            $table->boolean('is_final')->nullable()->index();
            $table->boolean('is_initial')->nullable()->index();
            $table->timestamps();
        });

        Schema::create('deal_attachments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('deal_id')->index();
            $table->foreignId('deal_note_id')->nullable()->index();
            $table->foreignId('deal_message_id')->nullable()->index();
            $table->foreignId('user_id')->nullable()->index();
            $table->string('filename');
            $table->string('original_filename');
            $table->string('path');
            $table->string('mime_type')->nullable();
            $table->unsignedBigInteger('size')->nullable();
            $table->timestamps();
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('deals');
        Schema::dropIfExists('deal_add_ons');
        Schema::dropIfExists('deal_part_exchanges');
        Schema::dropIfExists('deal_finance');
        Schema::dropIfExists('deal_history');
        Schema::dropIfExists('deal_notes');
        Schema::dropIfExists('deal_messages');
        Schema::dropIfExists('deal_builder_add_ons');
        Schema::dropIfExists('deal_builder_add_on_conditions');
        Schema::dropIfExists('deal_builder_statuses');

        Schema::dropIfExists('deal_attachments');
    }
};
