<?php

namespace Mtc\MercuryDataModels\DealBuilder;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Invoice;
use Mtc\MercuryDataModels\Customer;
use Mtc\MercuryDataModels\DealBuilder\Factories\DealFactory;
use Mtc\MercuryDataModels\User;
use Mtc\MercuryDataModels\Vehicle;

/**
 * @property Vehicle $vehicle
 * @property null|Customer $customer
 * @property null|Status $status
 * @property int $customer_id
 * @property int $vehicle_id
 * @property int $status_id
 * @property Carbon|null $ingested_at
 * @property int $assignee_id
 * @property float $deposit_amount
 * @property float $customer_deposit
 * @property string $reference
 * @property float $total_amount
 * @property float $payable_amount
 * @property int $annual_mileage
 * @property int $term
 * @property string $credit_rating
 * @property null|DealPaymentType $payment_type
 * @property string $contact_number
 * @property string $email
 * @property null|array $data
 */
class Deal extends Model implements PayableContract
{
    use HasFactory;
    use SoftDeletes;

    protected static function newFactory(): DealFactory
    {
        return DealFactory::new();
    }

    protected $fillable = [
        'customer_id',
        'vehicle_id',
        'status_id',
        'ingested_at',
        'assignee_id',
        'deposit_amount',
        'customer_deposit',
        'part_ex_contribution',
        'total_amount',
        'payable_amount',
        'reference',
        'annual_mileage',
        'term',
        'credit_rating',
        'payment_type',
        'contact_number',
        'first_name',
        'last_name',
        'email',
        'data',
        'first_name',
        'last_name',
        'email',
    ];

    protected $casts = [
        'ingested_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'payment_type' => DealPaymentType::class,
        'data' => 'array',
    ];

    protected static function boot()
    {
        parent::boot();
        self::creating(fn(self $deal) => $deal->reference = Carbon::now()->format('Y-m-') . Str::upper(Str::random(8)));
    }

    public function status(): BelongsTo
    {
        return $this->belongsTo(Status::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'assignee_id');
    }

    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    public function vehicle(): BelongsTo
    {
        return $this->belongsTo(Vehicle::class);
    }

    public function addOns(): HasMany
    {
        return $this->hasMany(DealAddOn::class);
    }

    public function finance(): HasMany
    {
        return $this->hasMany(DealFinance::class);
    }

    public function partExchange(): HasMany
    {
        return $this->hasMany(DealPartExchange::class);
    }

    public function history(): HasMany
    {
        return $this->hasMany(DealHistory::class)->latest();
    }

    public function notes(): HasMany
    {
        return $this->hasMany(DealNote::class)->latest();
    }

    public function messages(): HasMany
    {
        return $this->hasMany(DealMessage::class)->latest();
    }

    public function invoices(): MorphMany
    {
        return $this->morphMany(Invoice::class, 'payable');
    }

    public function getTimeSinceSubmittedAttribute()
    {
        return $this->timeDiffString($this->ingested_at, Carbon::now());
    }

    public function getTimeSinceUpdatedAttribute()
    {
        return $this->timeDiffString($this->updated_at, Carbon::now());
    }

    protected function timeDiffString(?Carbon $oldest_time, Carbon $youngest_time): string
    {
        if (!$oldest_time) {
            return '';
        }
        $diff = $youngest_time->diff($oldest_time);
        $parts = [];
        if ($diff->d) {
            $parts[] = $diff->d . 'd';
        }
        if ($diff->h) {
            $parts[] = $diff->h . 'h';
        }
        if ($diff->i) {
            $parts[] = $diff->i . 'm';
        }

        return implode(' ', $parts);
    }

    public function invoice(): MorphOne
    {
        return $this->morphOne(Invoice::class, 'payable');
    }

    public function getPayableClass(): string
    {
        return self::getMorphClass();
    }

    public function getReference(): string
    {
        return $this->reference;
    }

    public function getMember()
    {
        return $this->customer_id;
    }

    public function getTemplate(): string
    {
        return '';
    }

    public function getAmount(): float
    {
        return $this->payable_amount ?? 0;
    }

    public function getItems(): Collection
    {
        $paymentTypeLabel = match ($this->payment_type) {
            DealPaymentType::RESERVATION => 'Reservation fee',
            DealPaymentType::DEPOSIT => 'Deposit',
            DealPaymentType::FULL_PAYMENT => 'Full payment',
            default => 'Payment',
        };

        return collect([
            [
                'purchasable_id' => $this->vehicle_id,
                'purchasable_type' => 'vehicle',
                'name' => $this->vehicle->vrm_condensed . ' - ' . $this->vehicle->title
                    . ' (' . $paymentTypeLabel . ')',
                'unit_price' => $this->payable_amount,
                'paid_price' => $this->payable_amount,
                'quantity' => 1,
                'vat_rate' => 0.2,
                'attribute_fields' => [],
            ],
        ]);
    }

    public function getLines(): array
    {
        return [];
    }

    public function getAddress(): array
    {
        return [
            'first_name' => $this->customer?->first_name ?? '',
            'last_name' => $this->customer?->last_name ?? '',
        ];
    }

    public function getAdditionalDetails()
    {
        return [];
    }

    public function useExVat(): bool
    {
        return false;
    }
}
