<?php

namespace Mtc\MercuryDataModels\Finance;

use App\Facades\Settings;
use Illuminate\Support\Facades\App;
use Mtc\MercuryDataModels\Finance\Config\CodeWeaversConfig;
use Mtc\MercuryDataModels\Finance\Config\EvolutionConfig;
use Mtc\MercuryDataModels\Finance\Config\IVendiConfig;
use Mtc\MercuryDataModels\Finance\Config\SantanderConfig;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceProvider;
use Mtc\MercuryDataModels\Finance\Services\CodeWeavers;
use Mtc\MercuryDataModels\Finance\Services\CodeWeaversPeterVardy;
use Mtc\MercuryDataModels\Finance\Services\EvolutionFinance;
use Mtc\MercuryDataModels\Finance\Services\IVendi;
use Mtc\MercuryDataModels\Finance\Services\IVendiPeterVardy;
use Mtc\MercuryDataModels\Finance\Services\SantanderFinance;

class FinanceServiceHelper
{
    /**
     * Call provider initialization (static)
     *
     * @return FinanceProvider
     */
    public static function initializeForSite(): FinanceProvider
    {
        return (new self())->initialize();
    }

    public static function getProvider(): ?string
    {
        return (new self())->determineProvider();
    }

    /**
     * Check if there is an enabled finance provider
     *
     * @return bool
     */
    public static function hasEnabledProvider(): bool
    {
        return !empty((new self())->determineProvider());
    }

    /**
     * Initialize provider
     *
     * @return FinanceProvider
     */
    protected function initialize(): FinanceProvider
    {
        return match ($this->determineProvider()) {
            'codeweavers-pv' => new CodeWeaversPeterVardy(
                App::make(CodeWeaversConfig::class),
                new FinanceEligibilityChecker()
            ),
            'codeweavers' => new CodeWeavers(
                App::make(CodeWeaversConfig::class),
                new FinanceEligibilityChecker()
            ),
            'ivendi' => new IVendi(
                App::make(IVendiConfig::class),
                new FinanceEligibilityChecker()
            ),
            'ivendi-pv' => new IVendiPeterVardy(
                App::make(IVendiConfig::class),
                new FinanceEligibilityChecker()
            ),
            'evolution' => new EvolutionFinance(
                App::make(EvolutionConfig::class)
            ),
            'santander' => new SantanderFinance(
                App::make(SantanderConfig::class)
            ),
        };
    }

    /**
     * Determine which provider is set for the client
     *
     * @return string|null
     */
    public function determineProvider(): ?string
    {
        // note that we return the first valid element, so ordering is important
        return collect([
            'codeweavers-pv' => Settings::get('finance-codeweavers-enabled')
                && Settings::get('vehicle-finance-type') === 'peter-vardy',
            'codeweavers' => Settings::get('finance-codeweavers-enabled'),
            'ivendi-pv' => Settings::get('finance-ivendi-enabled')
                && Settings::get('vehicle-finance-type') === 'peter-vardy',
            'ivendi' => Settings::get('finance-ivendi-enabled'),
            'evolution' => Settings::get('finance-evolution-enabled'),
            'santander' => Settings::get('finance-santander-enabled'),
        ])
            ->filter()
            ->keys()
            ->first();
    }
}
