<?php

namespace Mtc\MercuryDataModels\Finance\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\MercuryDataModels\Finance\Config\SantanderConfig;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceProvider;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceRequestData;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResult;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResultCollection;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceType;
use Mtc\MercuryDataModels\Vehicle;

class SantanderFinance implements FinanceProvider
{
    public function __construct(
        protected readonly SantanderConfig $config
    ) {
        //
    }

    /**
     * Perform calculation on Santander API
     */
    public function calculate(FinanceRequestData $data): FinanceResultCollection
    {
        $results = [];
        $quoteId = 1;

        // Request HP quote if product ID configured
        if ($this->config->hpProductId()) {
            $hpRequest = $this->mapRequest($data, $quoteId++, $this->config->hpProductId());
            $hpResponse = $this->call([$hpRequest]);
            $hpResults = $this->process($hpResponse, $data, FinanceType::HP);
            $results = $results + $hpResults;
        }

        // Request PCP quote if product ID configured
        if ($this->config->pcpProductId()) {
            $pcpRequest = $this->mapRequest($data, $quoteId++, $this->config->pcpProductId());
            $pcpResponse = $this->call([$pcpRequest]);
            $pcpResults = $this->process($pcpResponse, $data, FinanceType::PCP);
            $results = $results + $pcpResults;
        }

        // Fallback: if no product IDs configured, make request without ProductId
        if (empty($results) && !$this->config->hpProductId() && !$this->config->pcpProductId()) {
            $request = $this->mapRequest($data, $quoteId);
            $response = $this->call([$request]);
            $results = $this->process($response, $data, FinanceType::PCP);
        }

        return new FinanceResultCollection($results);
    }

    public function dealerIdDataField(): string
    {
        return '';
    }

    public function dealerSpecificAccountKey(): ?string
    {
        return null;
    }

    public function isEligibleForCreditRatingFinance(Vehicle $vehicle): bool
    {
        return false;
    }

    public function calculateRiskBased(FinanceRequestData $data): FinanceResultCollection
    {
        return new FinanceResultCollection();
    }

    /**
     * Map vehicle data to Santander request format
     */
    protected function mapRequest(FinanceRequestData $data, int $quoteId, ?string $productId = null): array
    {
        $request = [
            'QuoteId' => $quoteId,
            'DealerCode' => $this->config->dealerCode(),
            'CapCode' => $data->cap_id,
            'CostPrice' => $data->price,
            'Period' => $data->term,
            'Deposit' => $data->deposit,
            'EstimatedAnnualMileage' => $data->annual_mileage,
        ];

        if ($productId) {
            $request['ProductId'] = $productId;
        }

        return $request;
    }

    /**
     * Call the Santander API
     */
    protected function call(array $quoteRequests): array
    {
        $body = [
            'ThirdPartyUser' => $this->config->username(),
            'ThirdPartyPassword' => $this->config->password(),
            'QuoteRequestList' => $quoteRequests,
        ];

        $response = Http::post(
            $this->config->baseUrl() . $this->config->quotesPath(),
            $body
        );

        if ($response->failed()) {
            Log::warning('Failed Santander finance retrieval', [
                'tenant_id' => tenant('id') ?? 'unknown',
                'request' => $quoteRequests,
                'dealer_code' => $this->config->dealerCode(),
                'status_code' => $response->status(),
                'result' => $response->body(),
            ]);

            return [];
        }

        return $response->json('quoteResponseList') ?? [];
    }

    /**
     * Process Santander response into array of FinanceResult keyed by finance type
     */
    protected function process(array $response, FinanceRequestData $data, FinanceType $financeType): array
    {
        if (empty($response)) {
            return [];
        }

        $results = [];

        foreach ($response as $quote) {
            if (($quote['quoteStatus'] ?? 0) !== 1) {
                continue;
            }

            $payments = $this->parsePayments($quote['payments'] ?? []);
            $monthlyPayment = $this->extractMonthlyPayment($payments);

            $result = new FinanceResult(
                provider: 'santander',
                finance_type: $financeType,
                monthly_price: $monthlyPayment,
                term: $quote['period'] ?? $data->term,
                number_of_payments: $quote['period'] ?? $data->term,
                total_deposit: $quote['deposit'] ?? $data->deposit,
                apr: $quote['apr'] ?? 0,
                first_payment: $payments[0]['value'] ?? $monthlyPayment,
                final_payment: $quote['ballon'] ?? 0,
                interest_rate: $quote['rate'] ?? 0,
                apply_url: '',
                product_name: 'Santander ' . $financeType->value,
                annual_mileage: $data->annual_mileage,
                cash_price: $quote['costPrice'] ?? $data->price,
                dealer_deposit: 0,
                customer_deposit: $quote['deposit'] ?? $data->deposit,
                payable_amount: $quote['totalAmount'] ?? 0,
                option_to_purchase_fee: $quote['paf'] ?? 0,
                documentation_fee: $quote['bff'] ?? 0
            );

            // Key by finance type value - only keep the first (or cheapest) quote per type
            if (!isset($results[$financeType->value]) || $monthlyPayment < $results[$financeType->value]->monthly_price) {
                $results[$financeType->value] = $result;
            }
        }

        return $results;
    }

    /**
     * Parse payments array from response
     */
    protected function parsePayments(array $payments): array
    {
        return $payments;
    }

    /**
     * Extract monthly payment from payments array
     */
    protected function extractMonthlyPayment(array $payments): float
    {
        if (empty($payments)) {
            return 0;
        }

        if (count($payments) === 1) {
            return $payments[0]['value'] ?? 0;
        }

        // Find the regular monthly payment (months > 1)
        $monthly = collect($payments)
            ->filter(fn ($p) => ($p['months'] ?? 0) > 1)
            ->first();

        return $monthly['value'] ?? ($payments[0]['value'] ?? 0);
    }
}
