<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\SearchIndex;
use Mtc\MercuryDataModels\Traits\FilterIndexIds;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;

use function collect;

class ModelFilter extends IndexedFilter
{
    use FilterIndexIds;

    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->whereHas('model', fn($modelQuery) => $modelQuery->whereIn('slug', $selection));
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        $makeSelections = $selections['make'] ?? [];

        // If no make is selected, check if there's only one make available
        if (empty($makeSelections)) {
            $availableMakes = $this->getAvailableMakeSlugs();
            if ($availableMakes->isEmpty()) {
                return collect();
            }
        }

        $results = $this->getQuery()
            ->distinct()
            ->when(Settings::get('filter-apply-selections-to-results'), fn($query) => $query->where($product_filtering))
            ->pluck('model_id');

        return VehicleModel::query()
            ->whereIn('id', $results)
            ->with('make')
            ->orderBy('name')
            ->distinct()
            ->get();
    }

    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'hide_in_ui' => $this->hideInUi(),
            'has_load_more' => $this->hasLoadMore($collection),
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
            'results' => $this->formatResults($collection)
        ];
    }

    private function formatResults(Collection $collection): Collection
    {
        return $collection->map(function (Model $model) {
            return [
                'id' => $model->slug,
                'name' => $model->name,
                'count' => $model->result_count,
                'slug' => $model->slug,
                'make_id' => $model->make_id,
                'make_slug' => $model->make?->slug
            ];
        });
    }

    public function getIndexedResults(string $filter_type, int $limit, array $selections = []): Collection
    {
        $makeSelections = $selections['make'] ?? [];

        // If no make is selected, check if there's only one make available
        // This handles the case where the make filter is hidden due to having only one option
        if (empty($makeSelections)) {
            $availableMakes = $this->getAvailableMakeSlugs();
            if ($availableMakes->count() === 1) {
                $makeSelections = $availableMakes->toArray();
            } else {
                return collect();
            }
        }

        return SearchIndex::query()
            ->where('filter_type', $filter_type)
            ->whereIn('selection_hash', $makeSelections)
            ->orderBy('filter_id')
            ->get();
    }

    /**
     * Get available make slugs from indexed vehicles
     */
    private function getAvailableMakeSlugs(): Collection
    {
        $makeIds = $this->getQuery()
            ->distinct()
            ->pluck('make_id');

        if ($makeIds->count() > 1) {
            return collect();
        }

        return VehicleMake::query()
            ->whereIn('id', $makeIds)
            ->pluck('slug');
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return VehicleModel::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'Model';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->name;
    }

    /**
     * Whether this
     * @return bool
     */
    public function visibleByDefault(): bool
    {
        return true;
    }

    public function filterType(): string
    {
        return 'model';
    }
}
